<?php
$page_title = 'Attendance Dashboard';
require_once '../../includes/header.php';

$selected_date = $_GET['date'] ?? date('Y-m-d');
$selected_month = $_GET['month'] ?? date('Y-m');

// Get attendance for selected date
$stmt = $pdo->prepare("SELECT ar.*, 
       CONCAT(sw.first_name, ' ', sw.last_name) as worker_name,
       sw.employee_number, sw.department
       FROM attendance_records ar
       JOIN sponsored_workers sw ON ar.worker_id = sw.id
       WHERE ar.attendance_date = ? AND ar.business_profile_id = ?
       ORDER BY sw.last_name, sw.first_name");
$stmt->execute([$selected_date, $_SESSION['business_profile_id']]);
$attendance_records = $stmt->fetchAll();

// Get attendance summary
$stmt = $pdo->prepare("SELECT 
       COUNT(*) as total,
       SUM(CASE WHEN status = 'P' THEN 1 ELSE 0 END) as present,
       SUM(CASE WHEN status = 'PL' THEN 1 ELSE 0 END) as paid_leave,
       SUM(CASE WHEN status = 'UL' THEN 1 ELSE 0 END) as unpaid_leave,
       SUM(CASE WHEN status = 'FO' THEN 1 ELSE 0 END) as full_off,
       SUM(CASE WHEN status = 'HO' THEN 1 ELSE 0 END) as half_off
       FROM attendance_records
       WHERE attendance_date = ? AND business_profile_id = ?");
$stmt->execute([$selected_date, $_SESSION['business_profile_id']]);
$summary = $stmt->fetch();

// Get total active workers
$stmt = $pdo->prepare("SELECT COUNT(*) FROM sponsored_workers 
                      WHERE is_active = 1 AND business_profile_id = ?");
$stmt->execute([$_SESSION['business_profile_id']]);
$total_workers = $stmt->fetchColumn();

// Status definitions
$status_codes = [
    'P' => ['label' => 'Present', 'class' => 'success', 'icon' => 'check-circle'],
    'PL' => ['label' => 'Paid Leave', 'class' => 'info', 'icon' => 'calendar-check'],
    'UL' => ['label' => 'Unpaid Leave', 'class' => 'warning', 'icon' => 'calendar-x'],
    'FO' => ['label' => 'Full Day Off', 'class' => 'secondary', 'icon' => 'moon'],
    'HO' => ['label' => 'Half Day Off', 'class' => 'primary', 'icon' => 'sunrise']
];
?>

<!-- Top Action Bar -->
<div class="stat-card mb-4">
    <div class="row align-items-center">
        <div class="col-md-4">
            <label class="form-label mb-2">Select Date</label>
            <input type="date" id="dateSelector" class="form-control" 
                   value="<?= $selected_date ?>" 
                   onchange="window.location='?date='+this.value">
        </div>
        <div class="col-md-8 text-end">
            <div class="btn-group">
                <a href="mark.php?date=<?= $selected_date ?>" class="btn btn-success">
                    <i class="bi bi-plus-circle"></i> Mark Attendance
                </a>
                <a href="?action=export_template" class="btn btn-primary">
                    <i class="bi bi-download"></i> Download Template
                </a>
                <button type="button" class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#uploadModal">
                    <i class="bi bi-upload"></i> Upload Excel
                </button>
                <a href="export.php?type=monthly&month=<?= $selected_month ?>" class="btn btn-info">
                    <i class="bi bi-file-earmark-spreadsheet"></i> Export Month
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Summary Cards -->
<div class="row mb-4">
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-dark"><?= $total_workers ?></h3>
            <small class="text-muted">Total Workers</small>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-success"><?= $summary['present'] ?? 0 ?></h3>
            <small class="text-muted">Present (P)</small>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-info"><?= $summary['paid_leave'] ?? 0 ?></h3>
            <small class="text-muted">Paid Leave (PL)</small>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-warning"><?= $summary['unpaid_leave'] ?? 0 ?></h3>
            <small class="text-muted">Unpaid (UL)</small>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-secondary"><?= $summary['full_off'] ?? 0 ?></h3>
            <small class="text-muted">Full Off (FO)</small>
        </div>
    </div>
    <div class="col-md-2">
        <div class="stat-card text-center">
            <h3 class="text-primary"><?= $summary['half_off'] ?? 0 ?></h3>
            <small class="text-muted">Half Off (HO)</small>
        </div>
    </div>
</div>

<!-- Attendance Table -->
<div class="stat-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="bi bi-calendar-check"></i> Attendance for <?= format_date($selected_date) ?></h5>
        <span class="badge bg-info"><?= count($attendance_records) ?> / <?= $total_workers ?> Marked</span>
    </div>
    
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th width="5%">#</th>
                    <th width="15%">Employee #</th>
                    <th width="25%">Name</th>
                    <th width="15%">Department</th>
                    <th width="15%">Status</th>
                    <th width="15%">Marked By</th>
                    <th width="10%">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($attendance_records)): ?>
                <tr>
                    <td colspan="7" class="text-center py-5">
                        <i class="bi bi-calendar-x" style="font-size: 48px; color: #ccc;"></i>
                        <p class="mt-3 text-muted">No attendance marked for this date</p>
                        <a href="mark.php?date=<?= $selected_date ?>" class="btn btn-success">
                            <i class="bi bi-plus-circle"></i> Mark Attendance Now
                        </a>
                    </td>
                </tr>
                <?php else: ?>
                <?php $count = 1; foreach ($attendance_records as $record): ?>
                <tr>
                    <td><?= $count++ ?></td>
                    <td><?= htmlspecialchars($record['employee_number']) ?></td>
                    <td><strong><?= htmlspecialchars($record['worker_name']) ?></strong></td>
                    <td><?= htmlspecialchars($record['department']) ?></td>
                    <td>
                        <?php 
                        $status_info = $status_codes[$record['status']] ?? ['label' => $record['status'], 'class' => 'secondary', 'icon' => 'question-circle'];
                        ?>
                        <span class="badge bg-<?= $status_info['class'] ?>">
                            <i class="bi bi-<?= $status_info['icon'] ?>"></i> 
                            <?= $record['status'] ?> - <?= $status_info['label'] ?>
                        </span>
                    </td>
                    <td>
                        <?php
                        if ($record['marked_by']) {
                            $stmt = $pdo->prepare("SELECT full_name FROM users WHERE id = ?");
                            $stmt->execute([$record['marked_by']]);
                            echo htmlspecialchars($stmt->fetchColumn());
                        } else {
                            echo '<span class="text-muted">System</span>';
                        }
                        ?>
                    </td>
                    <td>
                        <button type="button" class="btn btn-sm btn-outline-primary" 
                                data-bs-toggle="modal" 
                                data-bs-target="#editModal<?= $record['id'] ?>">
                            <i class="bi bi-pencil"></i> Edit
                        </button>
                    </td>
                </tr>
                
                <!-- Edit Modal -->
                <div class="modal fade" id="editModal<?= $record['id'] ?>" tabindex="-1">
                    <div class="modal-dialog">
                        <form method="POST" action="edit_attendance.php">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title">Edit Attendance</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                </div>
                                <div class="modal-body">
                                    <input type="hidden" name="record_id" value="<?= $record['id'] ?>">
                                    <input type="hidden" name="return_date" value="<?= $selected_date ?>">
                                    
                                    <p><strong>Worker:</strong> <?= htmlspecialchars($record['worker_name']) ?></p>
                                    <p><strong>Date:</strong> <?= format_date($selected_date) ?></p>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Status</label>
                                        <select name="status" class="form-select" required>
                                            <?php foreach ($status_codes as $code => $info): ?>
                                            <option value="<?= $code ?>" <?= $record['status'] === $code ? 'selected' : '' ?>>
                                                <?= $code ?> - <?= $info['label'] ?>
                                            </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">Notes (Optional)</label>
                                        <textarea name="notes" class="form-control" rows="2"><?= htmlspecialchars($record['notes'] ?? '') ?></textarea>
                                    </div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-save"></i> Save Changes
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Upload Excel Modal -->
<div class="modal fade" id="uploadModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-upload"></i> Upload Attendance from Excel</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="upload_attendance.php" enctype="multipart/form-data">
                <div class="modal-body">
                    <div class="alert alert-info">
                        <h6><i class="bi bi-info-circle"></i> How to Upload</h6>
                        <ol class="mb-0">
                            <li>Download the template first (button above)</li>
                            <li>Fill in attendance using codes: P, PL, UL, FO, HO</li>
                            <li>Save the Excel file</li>
                            <li>Upload it here</li>
                        </ol>
                    </div>
                    
                    <div class="card bg-light mb-3">
                        <div class="card-body">
                            <h6>Status Codes Reference</h6>
                            <table class="table table-sm mb-0">
                                <tr>
                                    <td><strong>P</strong></td>
                                    <td>Present (Full Day)</td>
                                </tr>
                                <tr>
                                    <td><strong>PL</strong></td>
                                    <td>Paid Leave (Authorized)</td>
                                </tr>
                                <tr>
                                    <td><strong>UL</strong></td>
                                    <td>Unpaid Leave (Unauthorized/Deducted)</td>
                                </tr>
                                <tr>
                                    <td><strong>FO</strong></td>
                                    <td>Full Day Off (Weekend/Holiday)</td>
                                </tr>
                                <tr>
                                    <td><strong>HO</strong></td>
                                    <td>Half Day Off (Morning/Afternoon off)</td>
                                </tr>
                            </table>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">Select Excel File</label>
                        <input type="file" name="excel_file" class="form-control" 
                               accept=".xlsx,.xls" required>
                    </div>
                    
                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" 
                               name="overwrite" id="overwrite">
                        <label class="form-check-label" for="overwrite">
                            Overwrite existing attendance for matching dates
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">
                        <i class="bi bi-upload"></i> Upload & Import
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php require_once '../../includes/footer.php'; ?>
