<?php
/**
 * SCMS v3.0 - Error Log Viewer
 * View and manage system error logs (Admin only)
 */
session_start();
require_once 'config/database.php';
require_once 'includes/session_manager.php';
require_once 'includes/error_handler.php';

// Require authentication and admin permission
require_authentication();

// Check permission level (only 90+ can view logs)
if ($_SESSION['permission_level'] < 90) {
    header('Location: dashboard.php?error=permission_denied');
    exit;
}

// Handle log clearing
if (isset($_POST['clear_log']) && $_SESSION['permission_level'] >= 100) {
    if (clear_error_log()) {
        $success_msg = "Error log cleared successfully";
    } else {
        $error_msg = "Failed to clear error log";
    }
}

// Get filter parameters
$filter_level = $_GET['level'] ?? null;
$filter_module = $_GET['module'] ?? null;
$limit = (int)($_GET['limit'] ?? 100);

// Get error logs
$error_logs = get_recent_errors($limit, $filter_level, $filter_module);

// Get unique modules and levels for filtering
$modules = [];
$levels = [];
foreach ($error_logs as $log) {
    if (!in_array($log['module'], $modules)) {
        $modules[] = $log['module'];
    }
    if (!in_array($log['level'], $levels)) {
        $levels[] = $log['level'];
    }
}
sort($modules);
sort($levels);

include 'includes/header.php';
?>

<div class="container-fluid py-4">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header bg-danger text-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">
                            <i class="bi bi-bug-fill"></i> System Error Logs
                        </h4>
                        <div>
                            <a href="settings.php?tab=system" class="btn btn-light btn-sm me-2">
                                <i class="bi bi-gear"></i>
                            </a>
                            <?php if ($_SESSION['permission_level'] >= 100): ?>
                            <button type="button" onclick="if(confirm('Are you sure you want to clear the error log?')) { document.getElementById('clearLogForm').submit(); }" class="btn btn-danger btn-sm">
                                <i class="bi bi-trash"></i> Clear Log
                            </button>
                            <form id="clearLogForm" method="POST" style="display:none;">
                                <input type="hidden" name="clear_log" value="1">
                            </form>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <div class="card-body">
                    <?php if (isset($success_msg)): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <i class="bi bi-check-circle"></i> <?= htmlspecialchars($success_msg) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>

                    <?php if (isset($error_msg)): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <i class="bi bi-exclamation-triangle"></i> <?= htmlspecialchars($error_msg) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                    <?php endif; ?>

                    <!-- Filters -->
                    <form method="GET" class="row g-3 mb-4">
                        <div class="col-md-3">
                            <label class="form-label">Error Level</label>
                            <select name="level" class="form-select">
                                <option value="">All Levels</option>
                                <?php foreach ($levels as $lvl): ?>
                                <option value="<?= htmlspecialchars($lvl) ?>" <?= $filter_level === $lvl ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($lvl) ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="form-label">Module</label>
                            <select name="module" class="form-select">
                                <option value="">All Modules</option>
                                <?php foreach ($modules as $mod): ?>
                                <option value="<?= htmlspecialchars($mod) ?>" <?= $filter_module === $mod ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($mod) ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="col-md-3">
                            <label class="form-label">Limit</label>
                            <select name="limit" class="form-select">
                                <option value="50" <?= $limit === 50 ? 'selected' : '' ?>>50</option>
                                <option value="100" <?= $limit === 100 ? 'selected' : '' ?>>100</option>
                                <option value="250" <?= $limit === 250 ? 'selected' : '' ?>>250</option>
                                <option value="500" <?= $limit === 500 ? 'selected' : '' ?>>500</option>
                            </select>
                        </div>

                        <div class="col-md-3 d-flex align-items-end">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="bi bi-funnel"></i> Apply Filters
                            </button>
                        </div>
                    </form>

                    <!-- Error Log Table -->
                    <?php if (empty($error_logs)): ?>
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> No error logs found
                    </div>
                    <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover table-sm">
                            <thead class="table-dark">
                                <tr>
                                    <th style="width: 150px;">Timestamp</th>
                                    <th style="width: 80px;">Level</th>
                                    <th style="width: 100px;">Module</th>
                                    <th style="width: 80px;">User</th>
                                    <th style="width: 120px;">IP Address</th>
                                    <th>Message</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($error_logs as $log): ?>
                                <?php
                                // Color code by severity
                                $row_class = '';
                                switch (strtoupper($log['level'])) {
                                    case 'CRITICAL':
                                    case 'EMERGENCY':
                                    case 'ALERT':
                                        $row_class = 'table-danger';
                                        $badge_class = 'bg-danger';
                                        break;
                                    case 'ERROR':
                                        $row_class = 'table-warning';
                                        $badge_class = 'bg-warning text-dark';
                                        break;
                                    case 'WARNING':
                                        $badge_class = 'bg-orange text-white';
                                        break;
                                    case 'INFO':
                                        $badge_class = 'bg-info text-white';
                                        break;
                                    default:
                                        $badge_class = 'bg-secondary';
                                }
                                ?>
                                <tr class="<?= $row_class ?>">
                                    <td class="small"><?= htmlspecialchars($log['timestamp']) ?></td>
                                    <td>
                                        <span class="badge <?= $badge_class ?>">
                                            <?= htmlspecialchars($log['level']) ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge bg-primary">
                                            <?= htmlspecialchars($log['module']) ?>
                                        </span>
                                    </td>
                                    <td class="small"><?= htmlspecialchars($log['user_id']) ?></td>
                                    <td class="small font-monospace"><?= htmlspecialchars($log['ip']) ?></td>
                                    <td class="small">
                                        <div style="max-height: 100px; overflow-y: auto; word-break: break-all;">
                                            <?= htmlspecialchars($log['message']) ?>
                                        </div>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <div class="mt-3">
                        <small class="text-muted">
                            Showing <?= count($error_logs) ?> most recent error(s)
                            <?= $filter_level ? " | Level: " . htmlspecialchars($filter_level) : '' ?>
                            <?= $filter_module ? " | Module: " . htmlspecialchars($filter_module) : '' ?>
                        </small>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.bg-orange {
    background-color: #fd7e14 !important;
}
.font-monospace {
    font-family: 'Courier New', monospace;
}
</style>

<?php include 'includes/footer.php'; ?>
