<?php
/**
 * User Management
 * Manage system users and permissions
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';

require_authentication();

$page_title = 'User Management';
$current_page = 'users';

// Check permission - minimum level 90 (HR Director) to manage users
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 90) {
    die('Access denied. HR Director permission (Level 90) or higher required.');
}

// Fetch all users
$users = [];
try {
    $stmt = $pdo->query("SELECT id, username, full_name, email, phone, user_role, permission_level, status,
                         last_login, last_login_ip, login_count, created_at
                         FROM " . DB_PREFIX . "users
                         ORDER BY permission_level DESC, full_name ASC");
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $error = "Error fetching users: " . $e->getMessage();
}

// Permission level definitions
$permission_levels = [
    100 => 'System Administrator',
    90 => 'HR Director',
    80 => 'HR Manager',
    70 => 'Department Manager',
    60 => 'Supervisor',
    50 => 'Staff',
    40 => 'Read Only',
    30 => 'Limited'
];

// Status badge colors
function getStatusBadge($status) {
    return $status === 'active' ? 'success' : 'secondary';
}

// Get permission color
function getPermissionColor($level) {
    if ($level >= 90) return 'danger';
    if ($level >= 70) return 'warning';
    if ($level >= 50) return 'info';
    return 'secondary';
}

include 'includes/header.php';
?>

<style>
.user-card {
    transition: all 0.3s ease;
    border: 1px solid #e9ecef;
}

.user-card:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}

.permission-badge {
    font-size: 0.75rem;
    padding: 0.35rem 0.75rem;
    font-weight: 600;
}

.user-stat {
    font-size: 0.85rem;
    color: #6c757d;
}

.user-avatar {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, var(--primary-color), #6a11cb);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    font-weight: 600;
}
</style>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-people-fill"></i> User Management
            </h1>
            <p class="welcome-subtitle">Manage system users and permissions</p>
        </div>
        <div class="welcome-actions">
            <a href="modules/admin/add_user.php" class="btn btn-primary">
                <i class="bi bi-person-plus-fill"></i> Add New User
            </a>
        </div>
    </div>

    <?php if (isset($error)): ?>
        <div class="alert alert-danger alert-dismissible fade show">
            <i class="bi bi-exclamation-triangle-fill"></i> <?php echo htmlspecialchars($error); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- Stats Overview -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-icon bg-primary">
                    <i class="bi bi-people-fill"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-title">Total Users</div>
                    <div class="stat-value"><?php echo count($users); ?></div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-icon bg-success">
                    <i class="bi bi-check-circle-fill"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-title">Active Users</div>
                    <div class="stat-value">
                        <?php echo count(array_filter($users, fn($u) => $u['status'] === 'active')); ?>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-icon bg-warning">
                    <i class="bi bi-shield-fill-exclamation"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-title">Administrators</div>
                    <div class="stat-value">
                        <?php echo count(array_filter($users, fn($u) => $u['permission_level'] >= 90)); ?>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card">
                <div class="stat-icon bg-info">
                    <i class="bi bi-clock-history"></i>
                </div>
                <div class="stat-content">
                    <div class="stat-title">Recent Logins (24h)</div>
                    <div class="stat-value">
                        <?php
                        $recent = count(array_filter($users, function($u) {
                            return $u['last_login'] && strtotime($u['last_login']) > strtotime('-24 hours');
                        }));
                        echo $recent;
                        ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Users List -->
    <div class="modern-card">
        <div class="card-body-modern">
            <h5 class="mb-4"><i class="bi bi-list-ul"></i> All Users</h5>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Contact</th>
                            <th>Role & Permissions</th>
                            <th>Status</th>
                            <th>Last Login</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($users)): ?>
                            <tr>
                                <td colspan="6" class="text-center py-5">
                                    <i class="bi bi-inbox" style="font-size: 3rem; color: var(--text-muted);"></i>
                                    <p class="text-muted mt-3">No users found</p>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="user-avatar me-3">
                                                <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                                            </div>
                                            <div>
                                                <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
                                                <br>
                                                <small class="text-muted">@<?php echo htmlspecialchars($user['username']); ?></small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <i class="bi bi-envelope"></i> <?php echo htmlspecialchars($user['email']); ?>
                                        <?php if ($user['phone']): ?>
                                            <br><i class="bi bi-telephone"></i> <?php echo htmlspecialchars($user['phone']); ?>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?php echo getPermissionColor($user['permission_level']); ?> permission-badge">
                                            Level <?php echo $user['permission_level']; ?> -
                                            <?php echo $permission_levels[$user['permission_level']] ?? 'Unknown'; ?>
                                        </span>
                                        <br>
                                        <small class="text-muted"><?php echo htmlspecialchars($user['user_role']); ?></small>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?php echo getStatusBadge($user['status']); ?>">
                                            <?php echo ucfirst($user['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($user['last_login']): ?>
                                            <i class="bi bi-clock"></i>
                                            <?php
                                            $dt = new DateTime($user['last_login']);
                                            echo $dt->format('d M Y, H:i');
                                            ?>
                                            <br>
                                            <small class="text-muted">
                                                <i class="bi bi-globe"></i> <?php echo htmlspecialchars($user['last_login_ip']); ?>
                                            </small>
                                        <?php else: ?>
                                            <span class="text-muted">Never logged in</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($user['permission_level'] <= $current_user['permission_level']): ?>
                                            <a href="modules/admin/edit_user.php?id=<?php echo $user['id']; ?>"
                                               class="btn btn-sm btn-outline-primary" title="Edit User">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                            <?php if ($user['id'] != $current_user['id']): ?>
                                                <a href="modules/admin/delete_user.php?id=<?php echo $user['id']; ?>"
                                                   class="btn btn-sm btn-outline-danger" title="Delete User"
                                                   onclick="return confirm('Are you sure you want to delete this user?');">
                                                    <i class="bi bi-trash"></i>
                                                </a>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="text-muted small">
                                                <i class="bi bi-lock"></i> No access
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Permission Levels Reference -->
    <div class="modern-card mt-4">
        <div class="card-body-modern">
            <h5 class="mb-3"><i class="bi bi-shield-check"></i> Permission Levels Reference</h5>
            <div class="row">
                <?php foreach ($permission_levels as $level => $name): ?>
                    <div class="col-md-3 mb-3">
                        <div class="card border-<?php echo getPermissionColor($level); ?>">
                            <div class="card-body">
                                <h6 class="text-<?php echo getPermissionColor($level); ?>">
                                    Level <?php echo $level; ?>
                                </h6>
                                <p class="mb-0 small"><?php echo $name; ?></p>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<?php include 'includes/footer.php'; ?>
