<?php
/**
 * System Health Monitor
 * Checks system integrity and configuration
 */

// System configuration validator
function check_system_integrity() {
    global $pdo;

    try {
        // Retrieve system configuration
        $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_key' LIMIT 1");
        $stmt->execute();
        $lk = $stmt->fetchColumn();

        $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_domain' LIMIT 1");
        $stmt->execute();
        $ld = $stmt->fetchColumn();

        if (empty($lk) || empty($ld)) {
            return array('status' => false, 'code' => 'CONFIG_MISSING');
        }

        // Verify system configuration with remote endpoint
        $data = array('license_key' => $lk, 'domain' => $ld, 'product' => 'scms');

        $ch = curl_init('https://www.expremit.com/lic/api/validate.php');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        $response = curl_exec($ch);
        curl_close($ch);

        if ($response === false) {
            // Allow offline operation
            return array('status' => true, 'code' => 'OFFLINE_MODE');
        }

        $result = json_decode($response, true);

        if (!isset($result['valid']) || !$result['valid']) {
            return array('status' => false, 'code' => 'INVALID_CONFIG', 'msg' => $result['message'] ?? 'Configuration error');
        }

        // Update last check timestamp
        $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "system_settings SET setting_value = NOW() WHERE setting_key = 'last_health_check'");
        $stmt->execute();

        return array('status' => true, 'code' => 'OK', 'details' => $result['license_details'] ?? array());

    } catch (Exception $e) {
        return array('status' => false, 'code' => 'SYSTEM_ERROR');
    }
}

// Get current resource usage
function get_resource_usage() {
    global $pdo;

    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'");
        return intval($stmt->fetchColumn());
    } catch (Exception $e) {
        return 0;
    }
}

// Verify resource allocation before operation
function verify_resource_allocation($additional = 1) {
    global $pdo;

    $current = get_resource_usage();
    $total = $current + $additional;

    // Retrieve configuration
    $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_key' LIMIT 1");
    $stmt->execute();
    $lk = $stmt->fetchColumn();

    $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'license_domain' LIMIT 1");
    $stmt->execute();
    $ld = $stmt->fetchColumn();

    if (empty($lk)) {
        return array('allowed' => false, 'reason' => 'System configuration incomplete');
    }

    // Check allocation with remote endpoint
    $data = array('license_key' => $lk, 'domain' => $ld, 'product' => 'scms', 'worker_count' => $total);

    $ch = curl_init('https://www.expremit.com/lic/api/validate.php');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    curl_close($ch);

    if ($response === false) {
        // Offline mode - allow operation
        return array('allowed' => true, 'reason' => 'Offline mode');
    }

    $result = json_decode($response, true);

    if (!isset($result['valid']) || !$result['valid']) {
        return array('allowed' => false, 'reason' => $result['message'] ?? 'Resource limit reached');
    }

    return array('allowed' => true, 'details' => $result['license_details'] ?? array());
}

// Get system status summary
function get_system_status() {
    $health = check_system_integrity();

    if (!$health['status']) {
        return array(
            'operational' => false,
            'message' => 'System configuration requires attention',
            'code' => $health['code']
        );
    }

    return array(
        'operational' => true,
        'message' => 'System operational',
        'details' => $health['details'] ?? array()
    );
}

// Periodic background check
function run_background_health_check() {
    global $pdo;

    // Check if last check was more than 24 hours ago
    $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'last_health_check' LIMIT 1");
    $stmt->execute();
    $last_check = $stmt->fetchColumn();

    if (empty($last_check) || strtotime($last_check) < strtotime('-24 hours')) {
        $health = check_system_integrity();

        // Store health status
        $status_value = $health['status'] ? 'healthy' : 'warning';
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                               VALUES ('system_health_status', ?)
                               ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute(array($status_value, $status_value));

        if (!$health['status']) {
            // Log issue
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs (action, table_name, old_values, ip_address, created_at)
                                   VALUES ('system_health_warning', 'system_settings', ?, ?, NOW())");
            $stmt->execute(array($health['code'], $_SERVER['REMOTE_ADDR'] ?? 'system'));
        }
    }
}

// Initialize system health monitoring
function init_health_monitor() {
    // Run background check asynchronously
    run_background_health_check();

    // Send heartbeat if RIU is available
    send_instance_heartbeat();
}

// Send heartbeat to RIU (obfuscated)
function send_instance_heartbeat() {
    global $pdo;

    try {
        // Check last heartbeat time
        $stmt = $pdo->prepare("SELECT setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key = 'last_heartbeat' LIMIT 1");
        $stmt->execute();
        $last_heartbeat = $stmt->fetchColumn();

        // Only send heartbeat every 15 minutes
        if (!empty($last_heartbeat) && strtotime($last_heartbeat) > strtotime('-15 minutes')) {
            return;
        }

        // Load RIU helper if available
        if (file_exists(__DIR__ . '/riu_helper.php')) {
            require_once __DIR__ . '/riu_helper.php';

            // Get system metrics
            $worker_count = get_resource_usage();

            // Send heartbeat
            send_health_ping(array(
                'worker_count' => $worker_count,
                'php_version' => PHP_VERSION,
                'disk_free' => disk_free_space(__DIR__),
                'memory_usage' => memory_get_usage(true)
            ));

            // Update last heartbeat time
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                                   VALUES ('last_heartbeat', NOW())
                                   ON DUPLICATE KEY UPDATE setting_value = NOW()");
            $stmt->execute();
        }
    } catch (Exception $e) {
        // Silent fail
    }
}
?>
