<?php
/**
 * System Error Page
 * Displays system configuration errors
 */
session_start();

// Get error details from session
$error_code = $_SESSION['system_code'] ?? 'UNKNOWN';
$error_message = $_SESSION['system_warning'] ?? 'System configuration error';
$license_status = $_SESSION['license_status'] ?? null;
$license_message = $_SESSION['license_message'] ?? null;

// Determine error details
$title = 'System Configuration Error';
$description = 'There is an issue with the system configuration.';
$action_text = 'Go to Login';
$action_link = 'login.php';

switch ($error_code) {
    case 'CONFIG_MISSING':
        $title = 'License Configuration Missing';
        $description = 'The system license key is not configured. Please contact your administrator to configure the license.';
        break;

    case 'INVALID_CONFIG':
        $title = 'Invalid License Configuration';
        $description = 'The license key is invalid or has been suspended. Please contact support.';
        break;

    case 'OFFLINE_NO_HISTORY':
        $title = 'License Server Unreachable';
        $description = 'Cannot validate license - the license server is unreachable and there is no previous validation history.';
        break;

    case 'GRACE_PERIOD_EXPIRED':
        $title = 'Grace Period Expired';
        $description = 'Cannot reach license server. The offline grace period has expired.';
        break;
}

// Override with specific license messages if available
if ($license_status === 'suspended') {
    $title = 'License Suspended';
    $description = $license_message ?? 'Your license has been suspended. Please contact support.';
} elseif ($license_status === 'expired') {
    $title = 'License Expired';
    $description = $license_message ?? 'Your license has expired. Please renew your license to continue using the system.';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($title) ?> - SCMS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.0/font/bootstrap-icons.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
        }
        .error-container {
            width: 100%;
            max-width: 600px;
            padding: 20px;
        }
        .error-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
            overflow: hidden;
        }
        .error-header {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            color: white;
            padding: 40px 30px;
            text-align: center;
        }
        .error-header i {
            font-size: 60px;
            margin-bottom: 15px;
        }
        .error-body {
            padding: 40px 30px;
        }
        .error-footer {
            background: #f8f9fa;
            padding: 20px 30px;
            text-align: center;
        }
        .debug-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            font-family: monospace;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <div class="error-card">
            <div class="error-header">
                <i class="bi bi-exclamation-triangle-fill"></i>
                <h3 class="mb-0"><?= htmlspecialchars($title) ?></h3>
            </div>

            <div class="error-body">
                <p class="lead"><?= htmlspecialchars($description) ?></p>

                <?php if ($error_message && $error_message !== 'System configuration requires attention'): ?>
                <div class="alert alert-warning">
                    <i class="bi bi-info-circle"></i> <strong>Details:</strong> <?= htmlspecialchars($error_message) ?>
                </div>
                <?php endif; ?>

                <div class="d-grid gap-2 mt-4">
                    <a href="<?= htmlspecialchars($action_link) ?>" class="btn btn-primary btn-lg">
                        <i class="bi bi-arrow-left-circle"></i> <?= htmlspecialchars($action_text) ?>
                    </a>
                    <a href="logout.php" class="btn btn-outline-secondary btn-lg">
                        <i class="bi bi-box-arrow-right"></i> Logout and Clear Session
                    </a>
                    <a href="clear_session_and_login.php" class="btn btn-success btn-lg">
                        <i class="bi bi-arrow-clockwise"></i> Reset Session & Fresh Login
                    </a>
                </div>

                <?php if (isset($_SESSION['permission_level']) && $_SESSION['permission_level'] >= 100): ?>
                <!-- Debug info for admins -->
                <div class="debug-info">
                    <strong>Debug Information (Admin Only):</strong><br>
                    Error Code: <?= htmlspecialchars($error_code) ?><br>
                    Session User ID: <?= isset($_SESSION['user_id']) ? htmlspecialchars($_SESSION['user_id']) : 'NOT SET' ?><br>
                    System Verified: <?= isset($_SESSION['system_verified']) ? date('Y-m-d H:i:s', $_SESSION['system_verified']) : 'NO' ?><br>
                    License Validated: <?= isset($_SESSION['license_validated_at']) ? date('Y-m-d H:i:s', $_SESSION['license_validated_at']) : 'NO' ?><br>

                    <?php if (isset($_SESSION['license_details'])): ?>
                    <br><strong>License Details:</strong><br>
                    <?php foreach ($_SESSION['license_details'] as $key => $value): ?>
                        <?= htmlspecialchars($key) ?>: <?= htmlspecialchars($value) ?><br>
                    <?php endforeach; ?>
                    <?php endif; ?>
                </div>
                <?php endif; ?>
            </div>

            <div class="error-footer">
                <small class="text-muted">
                    SCMS v3.0 &copy; <?= date('Y') ?> | System Error
                </small>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
