<?php
/**
 * Settings Page - System Settings with Tabbed Interface
 * Includes: General, Modules, License, System tabs
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';
require_once 'includes/ModuleManager.php';
require_once 'includes/resource_manager.php';
require_once 'includes/error_handler.php';

// Require authentication
require_authentication();

// Check admin permission
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 100) {
    die('Access denied. Admin permission required.');
}

$page_title = 'Settings';
$current_page = 'settings';
$active_tab = $_GET['tab'] ?? 'general';

// Initialize Module Manager for modules tab
$moduleManager = new ModuleManager($pdo);

// Handle module actions
$message = '';
$message_type = 'success';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Debug logging for module actions
    if (in_array($action, ['activate', 'deactivate'])) {
        error_log("Settings.php: Received {$action} request for module: " . ($_POST['module_id'] ?? 'N/A'));
    }

    // Handle error log clearing
    if ($action === 'clear_error_log') {
        if (clear_error_log()) {
            $message = "Error log cleared successfully";
            $message_type = 'success';
        } else {
            $message = "Failed to clear error log";
            $message_type = 'danger';
        }
    }

    // Handle error log settings
    elseif ($action === 'save_error_log_settings') {
        try {
            $auto_delete_days = (int)($_POST['auto_delete_days'] ?? 30);
            $entries_per_page = (int)($_POST['entries_per_page'] ?? 15);

            // Validate values
            $auto_delete_days = max(1, min(365, $auto_delete_days));
            $entries_per_page = max(10, min(500, $entries_per_page));

            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

            $stmt->execute(['error_log_auto_delete_days', $auto_delete_days]);
            $stmt->execute(['error_log_entries_per_page', $entries_per_page]);

            $message = "Error log settings saved successfully";
            $message_type = 'success';
        } catch (PDOException $e) {
            $message = "Error saving settings: " . $e->getMessage();
            $message_type = 'danger';
        }
    }

    // Handle department and role actions
    elseif ($action === 'add_department') {
        $name = trim($_POST['dept_name'] ?? '');
        $description = trim($_POST['dept_description'] ?? '');

        if (!empty($name)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "departments (name, description) VALUES (?, ?)");
                $stmt->execute([$name, $description]);
                $message = "Department added successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Error: " . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'edit_department') {
        $id = (int)$_POST['dept_id'];
        $name = trim($_POST['dept_name'] ?? '');
        $description = trim($_POST['dept_description'] ?? '');
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        if (!empty($name) && $id > 0) {
            try {
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "departments SET name = ?, description = ?, is_active = ? WHERE id = ?");
                $stmt->execute([$name, $description, $is_active, $id]);
                $message = "Department updated successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Error: " . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'delete_department') {
        $id = (int)$_POST['dept_id'];
        if ($id > 0) {
            try {
                $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "departments WHERE id = ?");
                $stmt->execute([$id]);
                $message = "Department deleted successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Cannot delete department (may have associated roles)";
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'save_smtp') {
        // Save SMTP settings
        try {
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

            $smtp_settings = array(
                array('smtp_enabled', isset($_POST['smtp_enabled']) ? '1' : '0'),
                array('smtp_host', $_POST['smtp_host'] ?? ''),
                array('smtp_port', $_POST['smtp_port'] ?? '587'),
                array('smtp_secure', $_POST['smtp_secure'] ?? 'tls'),
                array('smtp_username', $_POST['smtp_username'] ?? ''),
                array('smtp_password', $_POST['smtp_password'] ?? ''),
                array('smtp_from_email', $_POST['smtp_from_email'] ?? ''),
                array('smtp_from_name', $_POST['smtp_from_name'] ?? ''),
                array('smtp_reply_to', $_POST['smtp_reply_to'] ?? ''),
                array('smtp_auth_required', isset($_POST['smtp_auth_required']) ? '1' : '0')
            );

            foreach ($smtp_settings as $setting) {
                $stmt->execute($setting);
            }

            $message = "SMTP settings saved successfully.";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error saving SMTP settings: " . $e->getMessage();
            $message_type = 'danger';
        }
    }

    elseif ($action === 'add_role') {
        $title = trim($_POST['role_title'] ?? '');
        $description = trim($_POST['role_description'] ?? '');
        $level = $_POST['role_level'] ?? 'entry';
        $department_id = !empty($_POST['department_id']) ? (int)$_POST['department_id'] : null;

        if (!empty($title)) {
            try {
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "job_roles (title, description, level, department_id) VALUES (?, ?, ?, ?)");
                $stmt->execute([$title, $description, $level, $department_id]);
                $message = "Job role added successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Error: " . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'edit_role') {
        $id = (int)$_POST['role_id'];
        $title = trim($_POST['role_title'] ?? '');
        $description = trim($_POST['role_description'] ?? '');
        $level = $_POST['role_level'] ?? 'entry';
        $department_id = !empty($_POST['department_id']) ? (int)$_POST['department_id'] : null;
        $is_active = isset($_POST['is_active']) ? 1 : 0;

        if (!empty($title) && $id > 0) {
            try {
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "job_roles SET title = ?, description = ?, level = ?, department_id = ?, is_active = ? WHERE id = ?");
                $stmt->execute([$title, $description, $level, $department_id, $is_active, $id]);
                $message = "Job role updated successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Error: " . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'delete_role') {
        $id = (int)$_POST['role_id'];
        if ($id > 0) {
            try {
                $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "job_roles WHERE id = ?");
                $stmt->execute([$id]);
                $message = "Job role deleted successfully";
                $message_type = 'success';
            } catch (PDOException $e) {
                $message = "Error: " . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    elseif ($action === 'save_general') {
        // Save general settings
        try {
            // Handle logo upload - keep existing logo if no new upload
            $logo_path = $settings['logo_path'] ?? $settings['company_logo'] ?? '';

            if (isset($_FILES['logo_upload']) && $_FILES['logo_upload']['error'] === UPLOAD_ERR_OK) {
                // Ensure upload directory exists
                $upload_dir = 'uploads/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }

                $file_extension = strtolower(pathinfo($_FILES['logo_upload']['name'], PATHINFO_EXTENSION));
                $allowed_extensions = array('png', 'jpg', 'jpeg', 'gif');

                if (in_array($file_extension, $allowed_extensions) && $_FILES['logo_upload']['size'] <= 2097152) {
                    $new_filename = 'logo_' . time() . '.' . $file_extension;
                    $upload_path = $upload_dir . $new_filename;

                    if (move_uploaded_file($_FILES['logo_upload']['tmp_name'], $upload_path)) {
                        $logo_path = $upload_path;
                    }
                }
            }

            // Ensure logo_path has proper format
            if (!empty($logo_path) && strpos($logo_path, '/') === false) {
                $logo_path = 'uploads/' . $logo_path;
            }

            // Collect all form data
            $company_name = $_POST['company_name'] ?? '';
            $company_reg = $_POST['company_reg'] ?? '';
            $industry = $_POST['industry'] ?? '';
            $address = $_POST['address'] ?? '';
            $city = $_POST['city'] ?? '';
            $postcode = $_POST['postcode'] ?? '';
            $phone = $_POST['phone'] ?? '';
            $contact_email = $_POST['contact_email'] ?? '';
            $sponsor_license = $_POST['sponsor_license'] ?? '';

            // Update system settings using INSERT ... ON DUPLICATE KEY UPDATE
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

            $settings_to_save = array(
                array('company_name', $company_name),
                array('company_reg', $company_reg),
                array('industry', $industry),
                array('address', $address),
                array('city', $city),
                array('postcode', $postcode),
                array('phone', $phone),
                array('email', $contact_email),
                array('contact_email', $contact_email),
                array('sponsor_license_number', $sponsor_license),
                array('logo_path', $logo_path),
                array('company_logo', $logo_path) // Save to both keys for compatibility
            );

            foreach ($settings_to_save as $setting) {
                $stmt->execute($setting);
            }

            $message = "Company information saved successfully.";
            $message_type = 'success';
        } catch (Exception $e) {
            $message = "Error saving settings: " . $e->getMessage();
            $message_type = 'danger';
        }
    } else if (in_array($action, ['scan', 'install', 'activate', 'deactivate', 'uninstall', 'update_settings'])) {
        // Module actions
        $module_id = $_POST['module_id'] ?? '';

        switch ($action) {
            case 'scan':
                $count = $moduleManager->scanAndRegister();
                $message = "Scanned modules directory. Found {$count} module(s).";
                break;

            case 'install':
                // Install a discovered module by running scanAndRegister
                // This will register the module and run install.sql
                $count = $moduleManager->scanAndRegister();
                $message = "Module installed successfully.";
                $message_type = 'success';
                break;

            case 'activate':
                error_log("Settings.php: Activating module {$module_id}");
                $result = $moduleManager->activate($module_id);
                error_log("Settings.php: Activate result: " . json_encode($result));
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'danger';
                break;

            case 'deactivate':
                error_log("Settings.php: Deactivating module {$module_id}");
                $result = $moduleManager->deactivate($module_id);
                error_log("Settings.php: Deactivate result: " . json_encode($result));
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'danger';
                break;

            case 'uninstall':
                if (isset($_POST['confirm']) && $_POST['confirm'] === 'yes') {
                    $result = $moduleManager->uninstall($module_id);
                    $message = $result['message'];
                    $message_type = $result['success'] ? 'success' : 'danger';
                }
                break;

            case 'update_settings':
                $settings = $_POST['settings'] ?? array();
                $result = $moduleManager->updateSettings($module_id, $settings);
                $message = $result['message'];
                $message_type = $result['success'] ? 'success' : 'danger';
                break;
        }

        // Reload module manager
        $moduleManager = new ModuleManager($pdo);
    }
}

// Get system settings
$settings = array();
$stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings");
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Handle legacy logo key - installer saves as 'company_logo', we use 'logo_path'
if (!empty($settings['company_logo']) && empty($settings['logo_path'])) {
    // If company_logo exists but logo_path doesn't, use company_logo
    $logo_filename = $settings['company_logo'];
    // If it's just a filename (no path), prepend uploads/
    if (strpos($logo_filename, '/') === false && !empty($logo_filename)) {
        $settings['logo_path'] = 'uploads/' . $logo_filename;
    } else {
        $settings['logo_path'] = $logo_filename;
    }
}

// Get system capacity for license tab
$capacity = get_system_capacity();

// Get departments and roles for departments tab
$departments = [];
$job_roles = [];
if ($active_tab === 'departments') {
    $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "departments ORDER BY name");
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt = $pdo->query("SELECT jr.*, d.name as department_name
                         FROM " . DB_PREFIX . "job_roles jr
                         LEFT JOIN " . DB_PREFIX . "departments d ON jr.department_id = d.id
                         ORDER BY jr.title");
    $job_roles = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// Get modules for modules tab - Direct query to avoid caching issues
$stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "modules_registry ORDER BY sort_order, module_name");
$modules_raw = $stmt->fetchAll(PDO::FETCH_ASSOC);

$modules = array();
$active_modules = array();
$inactive_modules = array();

foreach ($modules_raw as $module) {
    $modules[$module['module_id']] = $module;
    if ($module['status'] === 'active') {
        $active_modules[] = $module;
    } else {
        $inactive_modules[] = $module;
    }
}

// Function to generate color scheme based on module ID
function getModuleColors($module_id) {
    // Predefined color schemes for specific modules (ensures visual variety)
    $predefined = array(
        'workers'     => array('from' => '#4481eb', 'to' => '#04befe', 'border' => '#4481eb', 'badge_bg' => 'rgba(68, 129, 235, 0.1)', 'badge_text' => '#4481eb'), // Blue
        'rtw'         => array('from' => '#0ba360', 'to' => '#3cba92', 'border' => '#0ba360', 'badge_bg' => 'rgba(11, 163, 96, 0.1)', 'badge_text' => '#0ba360'), // Green
        'cos'         => array('from' => '#667eea', 'to' => '#764ba2', 'border' => '#667eea', 'badge_bg' => 'rgba(102, 126, 234, 0.1)', 'badge_text' => '#667eea'), // Purple
        'locations'   => array('from' => '#00d2ff', 'to' => '#3a47d5', 'border' => '#00d2ff', 'badge_bg' => 'rgba(0, 210, 255, 0.1)', 'badge_text' => '#00d2ff'), // Cyan
        'attendance'  => array('from' => '#ff9a56', 'to' => '#ff6a88', 'border' => '#ff9a56', 'badge_bg' => 'rgba(255, 154, 86, 0.1)', 'badge_text' => '#ff9a56'), // Orange
        'documents'   => array('from' => '#f857a6', 'to' => '#ff5858', 'border' => '#f857a6', 'badge_bg' => 'rgba(248, 87, 166, 0.1)', 'badge_text' => '#f857a6'), // Hot Pink
        'holidays'    => array('from' => '#ffecd2', 'to' => '#fcb69f', 'border' => '#fcb69f', 'badge_bg' => 'rgba(252, 182, 159, 0.15)', 'badge_text' => '#fc9d6f'), // Peach
        'admin'       => array('from' => '#e0c3fc', 'to' => '#8ec5fc', 'border' => '#b695f0', 'badge_bg' => 'rgba(224, 195, 252, 0.15)', 'badge_text' => '#9f7ae0'), // Lavender
    );

    // Check if module has predefined color
    if (isset($predefined[$module_id])) {
        return $predefined[$module_id];
    }

    // Fallback color palette for new/unknown modules
    $palettes = array(
        array('from' => '#43e97b', 'to' => '#38f9d7', 'border' => '#43e97b', 'badge_bg' => 'rgba(67, 233, 123, 0.1)', 'badge_text' => '#43e97b'), // Green-Teal
        array('from' => '#fa709a', 'to' => '#fee140', 'border' => '#fa709a', 'badge_bg' => 'rgba(250, 112, 154, 0.1)', 'badge_text' => '#fa709a'), // Pink-Yellow
        array('from' => '#30cfd0', 'to' => '#330867', 'border' => '#30cfd0', 'badge_bg' => 'rgba(48, 207, 208, 0.1)', 'badge_text' => '#30cfd0'), // Cyan-Purple
        array('from' => '#a8edea', 'to' => '#fed6e3', 'border' => '#a8edea', 'badge_bg' => 'rgba(168, 237, 234, 0.15)', 'badge_text' => '#5bc5c0'), // Light Cyan-Pink
        array('from' => '#f093fb', 'to' => '#f5576c', 'border' => '#f093fb', 'badge_bg' => 'rgba(240, 147, 251, 0.1)', 'badge_text' => '#f093fb'), // Magenta
        array('from' => '#4facfe', 'to' => '#00f2fe', 'border' => '#4facfe', 'badge_bg' => 'rgba(79, 172, 254, 0.1)', 'badge_text' => '#4facfe'), // Sky Blue
        array('from' => '#a1c4fd', 'to' => '#c2e9fb', 'border' => '#a1c4fd', 'badge_bg' => 'rgba(161, 196, 253, 0.15)', 'badge_text' => '#6fa3f7'), // Light Blue
        array('from' => '#ff6e7f', 'to' => '#bfe9ff', 'border' => '#ff6e7f', 'badge_bg' => 'rgba(255, 110, 127, 0.1)', 'badge_text' => '#ff6e7f'), // Coral
        array('from' => '#fbc2eb', 'to' => '#a6c1ee', 'border' => '#fbc2eb', 'badge_bg' => 'rgba(251, 194, 235, 0.15)', 'badge_text' => '#d895d8'), // Pink-Blue
        array('from' => '#fdcbf1', 'to' => '#e6dee9', 'border' => '#f5a3df', 'badge_bg' => 'rgba(253, 203, 241, 0.15)', 'badge_text' => '#d87ac5'), // Rose
    );

    // Generate index from module_id hash for fallback
    $hash = crc32($module_id);
    $index = abs($hash) % count($palettes);

    return $palettes[$index];
}

// Backward compatibility wrapper
function getModuleGradient($module_id) {
    $colors = getModuleColors($module_id);
    return "background: linear-gradient(135deg, {$colors['from']} 0%, {$colors['to']} 100%);";
}

// Function to check if module is complete and ready for installation
function isModuleComplete($module_path, $module_id) {
    $required_files = array('module.json');
    $optional_files = array('list.php', 'add.php', 'edit.php', 'view.php', 'delete.php');

    // Check required files
    foreach ($required_files as $file) {
        if (!file_exists($module_path . '/' . $file)) {
            return array('complete' => false, 'missing' => $file);
        }
    }

    // Check if module has at least one functional PHP file
    $has_functional_file = false;
    foreach ($optional_files as $file) {
        if (file_exists($module_path . '/' . $file)) {
            $has_functional_file = true;
            break;
        }
    }

    if (!$has_functional_file) {
        return array('complete' => false, 'missing' => 'functional files (list.php, add.php, etc.)');
    }

    // All checks passed
    return array('complete' => true, 'missing' => null);
}

// Discover modules in filesystem that are not yet installed
$discovered_modules = array();
$modules_dir = dirname(__FILE__) . '/modules';

if (is_dir($modules_dir)) {
    $dirs = scandir($modules_dir);

    foreach ($dirs as $dir) {
        if ($dir === '.' || $dir === '..') continue;

        $module_path = $modules_dir . '/' . $dir;
        $info_file = $module_path . '/module.json';

        if (is_dir($module_path) && file_exists($info_file)) {
            $info = json_decode(file_get_contents($info_file), true);

            if ($info && isset($info['id']) && isset($info['name'])) {
                // Check if this module is already registered
                $is_registered = false;
                foreach ($modules as $registered_module) {
                    if ($registered_module['module_id'] === $info['id']) {
                        $is_registered = true;
                        break;
                    }
                }

                // If not registered, check completeness and add to discovered list
                if (!$is_registered) {
                    $completeness = isModuleComplete($module_path, $info['id']);
                    $info['is_complete'] = $completeness['complete'];
                    $info['missing_files'] = $completeness['missing'];
                    $discovered_modules[] = $info;
                }
            }
        }
    }
}

// Get error logs for error_logs tab
$error_logs = [];
$error_log_settings = [];
if ($active_tab === 'error_logs') {
    $filter_level = $_GET['level'] ?? null;
    $filter_module = $_GET['module'] ?? null;

    // Load error log settings
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key IN ('error_log_auto_delete_days', 'error_log_entries_per_page')");
    while ($row = $stmt->fetch()) {
        $error_log_settings[$row['setting_key']] = $row['setting_value'];
    }

    $default_entries = (int)($error_log_settings['error_log_entries_per_page'] ?? 15);
    $limit = (int)($_GET['limit'] ?? $default_entries);

    $error_logs = get_recent_errors($limit, $filter_level, $filter_module);

    // Get unique modules and levels for filtering
    $log_modules = [];
    $log_levels = [];
    foreach ($error_logs as $log) {
        if (!in_array($log['module'], $log_modules)) {
            $log_modules[] = $log['module'];
        }
        if (!in_array($log['level'], $log_levels)) {
            $log_levels[] = $log['level'];
        }
    }
    sort($log_modules);
    sort($log_levels);
}

include 'includes/header.php';
?>

<div class="dashboard-container">
    <!-- Page Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-gear"></i> System Settings
            </h1>
            <p class="welcome-subtitle">Configure system settings and manage modules</p>
        </div>
    </div>

    <!-- Alert Message -->
    <?php if ($message): ?>
    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show modern-alert" role="alert">
        <i class="bi bi-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>-fill"></i>
        <?php echo htmlspecialchars($message); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Tabs Navigation -->
    <div class="modern-card mb-4">
        <ul class="nav nav-tabs settings-tabs" role="tablist">
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'general' ? 'active' : ''; ?>"
                   href="?tab=general">
                    <i class="bi bi-sliders"></i> General
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'modules' ? 'active' : ''; ?>"
                   href="?tab=modules">
                    <i class="bi bi-puzzle"></i> Modules
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'departments' ? 'active' : ''; ?>"
                   href="?tab=departments">
                    <i class="bi bi-diagram-3"></i> Departments & Roles
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'users' ? 'active' : ''; ?>"
                   href="?tab=users">
                    <i class="bi bi-people"></i> Users
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'license' ? 'active' : ''; ?>"
                   href="?tab=license">
                    <i class="bi bi-shield-check"></i> License
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'smtp' ? 'active' : ''; ?>"
                   href="?tab=smtp">
                    <i class="bi bi-envelope-at"></i> SMTP
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link <?php echo $active_tab === 'error_logs' ? 'active' : ''; ?>"
                   href="?tab=error_logs">
                    <i class="bi bi-bug-fill"></i> Error Logs
                </a>
            </li>
        </ul>
    </div>

    <!-- Tab Content -->
    <div class="tab-content">

        <!-- General Settings Tab -->
        <?php if ($active_tab === 'general'): ?>
        <div class="modern-card">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-building"></i> Company Information
                </h5>
            </div>
            <div class="card-body-modern">
                <form method="POST" enctype="multipart/form-data">
                    <input type="hidden" name="action" value="save_general">

                    <!-- Company Logo -->
                    <?php if (!empty($settings['logo_path'])): ?>
                    <div class="mb-4 text-center">
                        <img src="<?php echo htmlspecialchars($settings['logo_path']); ?>"
                             alt="Company Logo" class="img-thumbnail" style="max-height: 120px;">
                    </div>
                    <?php endif; ?>

                    <div class="mb-4">
                        <label class="form-label"><strong><i class="bi bi-image"></i> Company Logo</strong></label>
                        <input type="file" class="form-control" name="logo_upload" accept="image/png,image/jpeg,image/jpg,image/gif">
                        <small class="text-muted">Upload company logo for letterheads and reports (PNG, JPG, GIF - Max 2MB)</small>
                    </div>

                    <hr class="my-4">

                    <!-- Basic Information -->
                    <h6 class="mb-3"><i class="bi bi-info-circle"></i> Basic Information</h6>

                    <div class="row">
                        <div class="col-md-8 mb-3">
                            <label class="form-label"><strong>Company Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="company_name"
                                   value="<?php echo htmlspecialchars($settings['company_name'] ?? ''); ?>" required>
                        </div>

                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Registration Number</strong></label>
                            <input type="text" class="form-control" name="company_reg"
                                   value="<?php echo htmlspecialchars($settings['company_reg'] ?? ''); ?>">
                            <small class="text-muted">Companies House number</small>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label"><strong>Industry Sector</strong></label>
                        <select class="form-select" name="industry">
                            <option value="">Select industry...</option>
                            <?php
                            $industries = array('Care & Health', 'Construction', 'Information Technology', 'Hospitality',
                                                'Manufacturing', 'Education', 'Finance & Banking', 'Retail', 'Other');
                            foreach ($industries as $ind) {
                                $selected = (isset($settings['industry']) && $settings['industry'] == $ind) ? 'selected' : '';
                                echo "<option value=\"{$ind}\" {$selected}>{$ind}</option>";
                            }
                            ?>
                        </select>
                    </div>

                    <hr class="my-4">

                    <!-- Address Information -->
                    <h6 class="mb-3"><i class="bi bi-geo-alt"></i> Address</h6>

                    <div class="mb-3">
                        <label class="form-label"><strong>Address Line 1</strong></label>
                        <input type="text" class="form-control" name="address"
                               value="<?php echo htmlspecialchars($settings['address'] ?? ''); ?>">
                    </div>

                    <div class="row">
                        <div class="col-md-8 mb-3">
                            <label class="form-label"><strong>City</strong></label>
                            <input type="text" class="form-control" name="city"
                                   value="<?php echo htmlspecialchars($settings['city'] ?? ''); ?>">
                        </div>

                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Postcode</strong></label>
                            <input type="text" class="form-control" name="postcode"
                                   value="<?php echo htmlspecialchars($settings['postcode'] ?? ''); ?>"
                                   placeholder="SW1A 1AA">
                        </div>
                    </div>

                    <hr class="my-4">

                    <!-- Contact Information -->
                    <h6 class="mb-3"><i class="bi bi-telephone"></i> Contact Details</h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Phone Number</strong></label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?php echo htmlspecialchars($settings['phone'] ?? ''); ?>"
                                   placeholder="+44 20 1234 5678">
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Company Email</strong> <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="contact_email"
                                   value="<?php echo htmlspecialchars($settings['email'] ?? $settings['contact_email'] ?? ''); ?>" required>
                        </div>
                    </div>

                    <hr class="my-4">

                    <!-- UK Home Office Sponsorship -->
                    <h6 class="mb-3"><i class="bi bi-shield-check"></i> UK Home Office Sponsorship</h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Sponsor License Number</strong></label>
                            <input type="text" class="form-control" name="sponsor_license"
                                   value="<?php echo htmlspecialchars($settings['sponsor_license_number'] ?? ''); ?>">
                            <small class="text-muted">Your UK Home Office sponsor license number</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>System Timezone</strong></label>
                            <select class="form-select" name="timezone">
                                <option value="Europe/London" selected>Europe/London (GMT/BST)</option>
                            </select>
                        </div>
                    </div>

                    <hr class="my-4">

                    <!-- System Information Section -->
                    <h6 class="mb-3"><i class="bi bi-cpu"></i> System Information</h6>
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <div class="info-item mb-3">
                                <label class="text-muted small">SCMS Version</label>
                                <div class="fw-bold">3.0.0</div>
                            </div>
                            <div class="info-item mb-3">
                                <label class="text-muted small">PHP Version</label>
                                <div class="fw-bold"><?php echo phpversion(); ?></div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="info-item mb-3">
                                <label class="text-muted small">Database</label>
                                <div class="fw-bold">MySQL <?php echo $pdo->query('SELECT VERSION()')->fetchColumn(); ?></div>
                            </div>
                            <div class="info-item mb-3">
                                <label class="text-muted small">Server Time</label>
                                <div class="fw-bold"><?php echo date('Y-m-d H:i:s'); ?></div>
                            </div>
                        </div>
                    </div>

                    <hr class="my-4">

                    <div class="d-flex justify-content-end">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Save Settings
                        </button>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>

        <!-- Departments & Roles Tab -->
        <?php if ($active_tab === 'departments'): ?>
        <div class="row">
            <!-- Departments Section -->
            <div class="col-lg-6 mb-4">
                <div class="modern-card">
                    <div class="card-header-modern">
                        <h5 class="card-title-modern">
                            <i class="bi bi-diagram-3"></i> Departments
                        </h5>
                        <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addDepartmentModal">
                            <i class="bi bi-plus-circle"></i> Add Department
                        </button>
                    </div>
                    <div class="card-body-modern">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Description</th>
                                        <th>Status</th>
                                        <th class="text-end">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($departments as $dept): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($dept['name']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($dept['description'] ?? ''); ?></td>
                                        <td>
                                            <?php if ($dept['is_active']): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Inactive</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end">
                                            <button class="btn btn-sm btn-outline-primary"
                                                    onclick="editDepartment(<?php echo htmlspecialchars(json_encode($dept)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <form method="POST" class="d-inline" onsubmit="return confirm('Delete this department?');">
                                                <input type="hidden" name="action" value="delete_department">
                                                <input type="hidden" name="dept_id" value="<?php echo $dept['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-danger">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Job Roles Section -->
            <div class="col-lg-6 mb-4">
                <div class="modern-card">
                    <div class="card-header-modern">
                        <h5 class="card-title-modern">
                            <i class="bi bi-briefcase"></i> Job Roles
                        </h5>
                        <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addRoleModal">
                            <i class="bi bi-plus-circle"></i> Add Role
                        </button>
                    </div>
                    <div class="card-body-modern">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Title</th>
                                        <th>Department</th>
                                        <th>Level</th>
                                        <th>Status</th>
                                        <th class="text-end">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($job_roles as $role): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($role['title']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($role['department_name'] ?? 'N/A'); ?></td>
                                        <td><span class="badge bg-info"><?php echo ucfirst($role['level']); ?></span></td>
                                        <td>
                                            <?php if ($role['is_active']): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Inactive</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-end">
                                            <button class="btn btn-sm btn-outline-primary"
                                                    onclick="editRole(<?php echo htmlspecialchars(json_encode($role)); ?>)">
                                                <i class="bi bi-pencil"></i>
                                            </button>
                                            <form method="POST" class="d-inline" onsubmit="return confirm('Delete this role?');">
                                                <input type="hidden" name="action" value="delete_role">
                                                <input type="hidden" name="role_id" value="<?php echo $role['id']; ?>">
                                                <button type="submit" class="btn btn-sm btn-outline-danger">
                                                    <i class="bi bi-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Add Department Modal -->
        <div class="modal fade" id="addDepartmentModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Add Department</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_department">
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Department Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="dept_name" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Description</label>
                                <textarea class="form-control" name="dept_description" rows="3"></textarea>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary">Add Department</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Edit Department Modal -->
        <div class="modal fade" id="editDepartmentModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Department</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <form method="POST">
                        <input type="hidden" name="action" value="edit_department">
                        <input type="hidden" name="dept_id" id="edit_dept_id">
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Department Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="dept_name" id="edit_dept_name" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Description</label>
                                <textarea class="form-control" name="dept_description" id="edit_dept_description" rows="3"></textarea>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" name="is_active" id="edit_dept_active" value="1">
                                <label class="form-check-label" for="edit_dept_active">Active</label>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary">Update Department</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Add Role Modal -->
        <div class="modal fade" id="addRoleModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Add Job Role</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_role">
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Role Title <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="role_title" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Description</label>
                                <textarea class="form-control" name="role_description" rows="2"></textarea>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Level</label>
                                <select class="form-select" name="role_level">
                                    <option value="entry">Entry</option>
                                    <option value="junior">Junior</option>
                                    <option value="mid">Mid</option>
                                    <option value="senior">Senior</option>
                                    <option value="manager">Manager</option>
                                    <option value="director">Director</option>
                                    <option value="executive">Executive</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Department</label>
                                <select class="form-select" name="department_id">
                                    <option value="">None</option>
                                    <?php foreach ($departments as $dept): ?>
                                        <option value="<?php echo $dept['id']; ?>"><?php echo htmlspecialchars($dept['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary">Add Role</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Edit Role Modal -->
        <div class="modal fade" id="editRoleModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Edit Job Role</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <form method="POST">
                        <input type="hidden" name="action" value="edit_role">
                        <input type="hidden" name="role_id" id="edit_role_id">
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">Role Title <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" name="role_title" id="edit_role_title" required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Description</label>
                                <textarea class="form-control" name="role_description" id="edit_role_description" rows="2"></textarea>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Level</label>
                                <select class="form-select" name="role_level" id="edit_role_level">
                                    <option value="entry">Entry</option>
                                    <option value="junior">Junior</option>
                                    <option value="mid">Mid</option>
                                    <option value="senior">Senior</option>
                                    <option value="manager">Manager</option>
                                    <option value="director">Director</option>
                                    <option value="executive">Executive</option>
                                </select>
                            </div>
                            <div class="mb-3">
                                <label class="form-label">Department</label>
                                <select class="form-select" name="department_id" id="edit_role_dept">
                                    <option value="">None</option>
                                    <?php foreach ($departments as $dept): ?>
                                        <option value="<?php echo $dept['id']; ?>"><?php echo htmlspecialchars($dept['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" name="is_active" id="edit_role_active" value="1">
                                <label class="form-check-label" for="edit_role_active">Active</label>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <button type="submit" class="btn btn-primary">Update Role</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Modules Tab -->
        <?php if ($active_tab === 'modules'): ?>
        <div class="d-flex justify-content-end mb-3">
            <form method="POST" class="d-inline">
                <input type="hidden" name="action" value="scan">
                <button type="submit" class="btn btn-outline-primary">
                    <i class="bi bi-arrow-clockwise"></i> Scan for Modules
                </button>
            </form>
        </div>

        <!-- Discovered Modules (Not Yet Installed) -->
        <?php if (!empty($discovered_modules)): ?>
        <div class="modern-card mb-4">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-search text-info"></i> Discovered Modules
                    <span class="badge bg-info ms-2"><?php echo count($discovered_modules); ?></span>
                </h5>
                <p class="text-muted small mb-0">These modules are available but not yet installed</p>
            </div>
            <div class="card-body-modern">
                <div class="row g-4">
                    <?php foreach ($discovered_modules as $module):
                        $colors = getModuleColors($module['id']);
                    ?>
                    <div class="col-md-6 col-lg-4">
                        <div class="module-card" style="border-top: 3px solid <?php echo $colors['border']; ?>;">
                            <div class="module-header">
                                <div class="module-icon" style="background: linear-gradient(135deg, <?php echo $colors['from']; ?> 0%, <?php echo $colors['to']; ?> 100%);">
                                    <i class="bi <?php echo htmlspecialchars($module['icon'] ?? 'bi-puzzle'); ?>"></i>
                                </div>
                                <?php if (!$module['is_complete']): ?>
                                <span class="module-badge" style="background: #dc3545; color: white;">Not Complete</span>
                                <?php else: ?>
                                <span class="module-badge" style="background: <?php echo $colors['badge_bg']; ?>; color: <?php echo $colors['badge_text']; ?>;">Not Installed</span>
                                <?php endif; ?>
                            </div>
                            <div class="module-body">
                                <h6 class="module-title"><?php echo htmlspecialchars($module['module_name']); ?></h6>
                                <p class="module-description"><?php echo htmlspecialchars($module['description'] ?? 'No description available'); ?></p>
                                <div class="module-meta">
                                    <span class="module-version">
                                        <i class="bi bi-tag"></i> v<?php echo htmlspecialchars($module['version']); ?>
                                    </span>
                                    <?php if (isset($module['author'])): ?>
                                    <span class="module-author">
                                        <i class="bi bi-person"></i> <?php echo htmlspecialchars($module['author']); ?>
                                    </span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="module-footer">
                                <?php if ($module['is_complete']): ?>
                                <form method="POST" class="w-100" data-module-action="install">
                                    <input type="hidden" name="module_id" value="<?php echo htmlspecialchars($module['id']); ?>">
                                    <button type="submit" class="btn btn-sm btn-info w-100">
                                        <i class="bi bi-download"></i> Install Module
                                    </button>
                                </form>
                                <?php else: ?>
                                <div class="alert alert-warning mb-0 small">
                                    <i class="bi bi-exclamation-triangle"></i>
                                    <strong>Incomplete:</strong> Missing <?php echo htmlspecialchars($module['missing_files']); ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Active Modules -->
        <div class="modern-card mb-4">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-check-circle-fill text-success"></i> Active Modules
                    <span class="badge bg-success ms-2"><?php echo count($active_modules); ?></span>
                </h5>
            </div>
            <div class="card-body-modern">
                <?php if (empty($active_modules)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-inbox" style="font-size: 3rem; color: var(--text-muted);"></i>
                        <p class="text-muted mt-3">No active modules. Activate modules from the available list below.</p>
                    </div>
                <?php else: ?>
                    <div class="row g-4">
                        <?php foreach ($active_modules as $module):
                            $info = $moduleManager->getModuleInfo($module['module_id']);
                            $colors = getModuleColors($module['module_id']);
                        ?>
                        <div class="col-md-6 col-lg-4">
                            <div class="module-card" style="border-top: 3px solid <?php echo $colors['border']; ?>;">
                                <div class="module-header">
                                    <div class="module-icon" style="background: linear-gradient(135deg, <?php echo $colors['from']; ?> 0%, <?php echo $colors['to']; ?> 100%);">
                                        <i class="bi <?php echo htmlspecialchars($module['icon'] ?? 'bi-puzzle'); ?>"></i>
                                    </div>
                                    <span class="module-badge" style="background: <?php echo $colors['badge_bg']; ?>; color: <?php echo $colors['badge_text']; ?>;">Active</span>
                                </div>
                                <div class="module-body">
                                    <h6 class="module-title"><?php echo htmlspecialchars($module['module_name']); ?></h6>
                                    <p class="module-description">
                                        <?php echo htmlspecialchars($module['description'] ?? 'No description available'); ?>
                                    </p>
                                    <div class="module-meta">
                                        <span class="module-version">
                                            <i class="bi bi-tag"></i> v<?php echo htmlspecialchars($module['version'] ?? '1.0.0'); ?>
                                        </span>
                                        <?php if (isset($module['author']) && !empty($module['author'])): ?>
                                        <span class="module-author">
                                            <i class="bi bi-person"></i> <?php echo htmlspecialchars($module['author']); ?>
                                        </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="module-footer">
                                    <?php
                                    $is_locked = in_array($module['module_id'], ['dashboard', 'settings']);
                                    if (!$is_locked):
                                    ?>
                                    <div class="d-flex gap-2 mb-2">
                                        <button class="btn btn-sm btn-outline-secondary flex-fill reorder-btn"
                                                data-module-id="<?php echo $module['module_id']; ?>"
                                                data-action="up"
                                                <?php echo (array_search($module, $active_modules) === 0 || $module['module_id'] === 'dashboard') ? 'disabled' : ''; ?>
                                                title="Move Up">
                                            <i class="bi bi-arrow-up"></i>
                                        </button>
                                        <button class="btn btn-sm btn-outline-secondary flex-fill reorder-btn"
                                                data-module-id="<?php echo $module['module_id']; ?>"
                                                data-action="down"
                                                <?php echo (array_search($module, $active_modules) === count($active_modules) - 1 || $module['module_id'] === 'settings') ? 'disabled' : ''; ?>
                                                title="Move Down">
                                            <i class="bi bi-arrow-down"></i>
                                        </button>
                                    </div>
                                    <?php endif; ?>
                                    <div class="btn-group w-100">
                                        <?php
                                        // Check if module has custom settings page
                                        $has_custom_settings = file_exists("modules/{$module['module_id']}/settings.php");
                                        if ($has_custom_settings || ($info && isset($info['has_settings']) && $info['has_settings'])):
                                        ?>
                                        <a href="<?php echo $has_custom_settings ? "modules/{$module['module_id']}/settings.php" : "module_settings.php?module={$module['module_id']}"; ?>"
                                           class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-gear"></i> Settings
                                        </a>
                                        <?php endif; ?>
                                        <form method="POST" class="flex-fill" data-module-action="deactivate" onsubmit="return confirm('Deactivate this module?');">
                                            <input type="hidden" name="action" value="deactivate">
                                            <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-outline-warning w-100">
                                                <i class="bi bi-pause-circle"></i> Deactivate
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>

                            <!-- Settings Modal -->
                            <?php if ($info && isset($info['has_settings']) && $info['has_settings']): ?>
                            <div class="modal fade" id="settingsModal<?php echo $module['module_id']; ?>" tabindex="-1">
                                <div class="modal-dialog">
                                    <div class="modal-content">
                                        <div class="modal-header">
                                            <h5 class="modal-title"><?php echo htmlspecialchars($module['module_name']); ?> Settings</h5>
                                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                        </div>
                                        <form method="POST">
                                            <div class="modal-body">
                                                <input type="hidden" name="action" value="update_settings">
                                                <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">

                                                <?php
                                                $current_settings = $moduleManager->getSettings($module['module_id']);
                                                if (isset($info['settings_fields'])):
                                                    foreach ($info['settings_fields'] as $field):
                                                ?>
                                                    <div class="mb-3">
                                                        <label class="form-label"><?php echo htmlspecialchars($field['label']); ?></label>
                                                        <?php if ($field['type'] === 'text'): ?>
                                                            <input type="text" class="form-control"
                                                                   name="settings[<?php echo $field['name']; ?>]"
                                                                   value="<?php echo htmlspecialchars($current_settings[$field['name']] ?? $field['default'] ?? ''); ?>">
                                                        <?php elseif ($field['type'] === 'checkbox'): ?>
                                                            <div class="form-check">
                                                                <input type="checkbox" class="form-check-input"
                                                                       name="settings[<?php echo $field['name']; ?>]" value="1"
                                                                       <?php echo (!empty($current_settings[$field['name']]) ? 'checked' : ''); ?>>
                                                                <label class="form-check-label"><?php echo htmlspecialchars($field['help'] ?? ''); ?></label>
                                                            </div>
                                                        <?php endif; ?>
                                                        <?php if (isset($field['help'])): ?>
                                                            <small class="text-muted"><?php echo htmlspecialchars($field['help']); ?></small>
                                                        <?php endif; ?>
                                                    </div>
                                                <?php
                                                    endforeach;
                                                endif;
                                                ?>
                                            </div>
                                            <div class="modal-footer">
                                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                                <button type="submit" class="btn btn-primary">Save Settings</button>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Available Modules -->
        <div class="modern-card">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-box-seam"></i> Available Modules
                    <span class="badge bg-secondary ms-2"><?php echo count($inactive_modules); ?></span>
                </h5>
            </div>
            <div class="card-body-modern">
                <?php if (empty($inactive_modules)): ?>
                    <div class="text-center py-5">
                        <i class="bi bi-check-circle" style="font-size: 3rem; color: var(--text-success);"></i>
                        <p class="text-muted mt-3">All available modules are active!</p>
                    </div>
                <?php else: ?>
                    <div class="row g-4">
                        <?php foreach ($inactive_modules as $module):
                            $colors = getModuleColors($module['module_id']);
                        ?>
                        <div class="col-md-6 col-lg-4">
                            <div class="module-card" style="border-top: 3px solid <?php echo $colors['border']; ?>;">
                                <div class="module-header">
                                    <div class="module-icon" style="background: linear-gradient(135deg, <?php echo $colors['from']; ?> 0%, <?php echo $colors['to']; ?> 100%);">
                                        <i class="bi <?php echo htmlspecialchars($module['icon'] ?? 'bi-puzzle'); ?>"></i>
                                    </div>
                                    <span class="module-badge" style="background: <?php echo $colors['badge_bg']; ?>; color: <?php echo $colors['badge_text']; ?>;">Inactive</span>
                                </div>
                                <div class="module-body">
                                    <h6 class="module-title"><?php echo htmlspecialchars($module['module_name']); ?></h6>
                                    <p class="module-description"><?php echo htmlspecialchars($module['description']); ?></p>
                                    <div class="module-meta">
                                        <span class="module-version">
                                            <i class="bi bi-tag"></i> v<?php echo htmlspecialchars($module['version']); ?>
                                        </span>
                                        <?php
                                        $info = $moduleManager->getModuleInfo($module['module_id']);
                                        if ($info && isset($info['requires']['modules']) && !empty($info['requires']['modules'])):
                                        ?>
                                        <span class="module-author">
                                            <i class="bi bi-link-45deg"></i> Requires: <?php echo htmlspecialchars(implode(', ', $info['requires']['modules'])); ?>
                                        </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="module-footer">
                                    <form method="POST" class="w-100" data-module-action="activate">
                                        <input type="hidden" name="action" value="activate">
                                        <input type="hidden" name="module_id" value="<?php echo $module['module_id']; ?>">
                                        <button type="submit" class="btn btn-sm btn-success w-100">
                                            <i class="bi bi-play-circle"></i> Activate
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- License Tab -->
        <?php if ($active_tab === 'license'): ?>
        <div class="modern-card">
            <div class="card-header-modern">
                <div class="header-content">
                    <h5 class="card-title-modern">
                        <i class="bi bi-graph-up-arrow"></i> License Capacity
                    </h5>
                    <p class="card-subtitle-modern">Monitor your worker capacity usage</p>
                </div>
                <span class="capacity-badge <?php echo $capacity['percentage'] > 80 ? 'badge-danger' : ($capacity['percentage'] > 60 ? 'badge-warning' : 'badge-success'); ?>">
                    <?php echo $capacity['percentage']; ?>% Used
                </span>
            </div>
            <div class="card-body-modern">
                <div class="capacity-info">
                    <div class="capacity-stats">
                        <div class="capacity-stat">
                            <span class="capacity-number"><?php echo $capacity['current_usage']; ?></span>
                            <span class="capacity-text">Used</span>
                        </div>
                        <div class="capacity-divider">/</div>
                        <div class="capacity-stat">
                            <span class="capacity-number"><?php echo $capacity['capacity']; ?></span>
                            <span class="capacity-text">Total</span>
                        </div>
                    </div>
                </div>
                <div class="modern-progress">
                    <div class="progress-track">
                        <div class="progress-fill <?php echo $capacity['percentage'] > 80 ? 'progress-danger' : ($capacity['percentage'] > 60 ? 'progress-warning' : 'progress-success'); ?>"
                             style="width: <?php echo $capacity['percentage']; ?>%">
                            <span class="progress-label"><?php echo $capacity['percentage']; ?>%</span>
                        </div>
                    </div>
                </div>
                <?php if ($capacity['percentage'] > 80): ?>
                <div class="alert alert-warning mt-3 mb-0">
                    <i class="bi bi-exclamation-triangle-fill"></i> <strong>Warning:</strong> Approaching capacity limit
                </div>
                <?php endif; ?>

                <!-- License Information Section -->
                <hr class="my-4">
                <h6 class="mb-3"><i class="bi bi-shield-check"></i> License Details</h6>
                <div class="row">
                    <div class="col-md-4">
                        <div class="info-item mb-3">
                            <label class="text-muted small d-flex align-items-center gap-1">
                                <i class="bi bi-award-fill"></i> License Type
                            </label>
                            <div class="fw-bold text-capitalize"><?php echo htmlspecialchars($capacity['license_type'] ?? 'Unknown'); ?></div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="info-item mb-3">
                            <label class="text-muted small d-flex align-items-center gap-1">
                                <i class="bi bi-calendar-check"></i> Expires
                            </label>
                            <div class="fw-bold"><?php echo htmlspecialchars($settings['license_expires'] ?? 'Unknown'); ?></div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="info-item mb-3">
                            <label class="text-muted small d-flex align-items-center gap-1">
                                <i class="bi bi-check-circle-fill"></i> Status
                            </label>
                            <div><span class="badge bg-success">Active</span></div>
                        </div>
                    </div>
                </div>
                <div class="info-item">
                    <label class="text-muted small d-flex align-items-center gap-1">
                        <i class="bi bi-key"></i> License Key
                    </label>
                    <div class="fw-bold font-monospace"><?php echo htmlspecialchars(substr($settings['license_key'] ?? 'Not Set', 0, 40) . '...'); ?></div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Users Tab -->
        <?php if ($active_tab === 'users'): ?>
        <?php
        // Fetch all users for user management tab
        $users_list = [];
        try {
            $stmt = $pdo->query("SELECT id, username, first_name, last_name, full_name, email, phone,
                                 user_role, permission_level, status, worker_id,
                                 last_login, last_login_ip, login_count, created_at
                                 FROM " . DB_PREFIX . "users
                                 ORDER BY permission_level DESC, full_name ASC");
            $users_list = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            $message = "Error fetching users: " . $e->getMessage();
            $message_type = 'danger';
        }

        // Permission level definitions
        $permission_levels = [
            100 => 'System Administrator',
            90 => 'HR Director',
            80 => 'HR Manager',
            70 => 'Department Manager',
            60 => 'Supervisor',
            50 => 'Staff',
            40 => 'Read Only',
            30 => 'Limited'
        ];

        function getUserStatusBadge($status) {
            return $status === 'active' ? 'success' : 'secondary';
        }

        function getUserPermissionColor($level) {
            if ($level >= 90) return 'danger';
            if ($level >= 70) return 'warning';
            if ($level >= 50) return 'info';
            return 'secondary';
        }
        ?>

        <style>
        .user-avatar-small {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color), #6a11cb);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.2rem;
            font-weight: 600;
        }
        .permission-badge-small {
            font-size: 0.7rem;
            padding: 0.25rem 0.5rem;
            font-weight: 600;
        }
        </style>

        <!-- User Stats -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="stat-icon bg-primary">
                        <i class="bi bi-people-fill"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Total Users</div>
                        <div class="stat-value"><?php echo count($users_list); ?></div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="stat-icon bg-success">
                        <i class="bi bi-check-circle-fill"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Active Users</div>
                        <div class="stat-value">
                            <?php echo count(array_filter($users_list, fn($u) => $u['status'] === 'active')); ?>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="stat-icon bg-warning">
                        <i class="bi bi-shield-fill-exclamation"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Administrators</div>
                        <div class="stat-value">
                            <?php echo count(array_filter($users_list, fn($u) => $u['permission_level'] >= 90)); ?>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="stat-card">
                    <div class="stat-icon bg-info">
                        <i class="bi bi-clock-history"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Recent Logins (24h)</div>
                        <div class="stat-value">
                            <?php
                            $recent = count(array_filter($users_list, function($u) {
                                return $u['last_login'] && strtotime($u['last_login']) > strtotime('-24 hours');
                            }));
                            echo $recent;
                            ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Users List -->
        <div class="modern-card">
            <div class="card-header-modern">
                <div class="d-flex justify-content-between align-items-center w-100">
                    <h5 class="card-title-modern mb-0">
                        <i class="bi bi-people"></i> System Users
                    </h5>
                    <div class="ms-auto">
                        <a href="modules/admin/add_user.php" class="btn btn-primary btn-sm">
                            <i class="bi bi-person-plus-fill"></i> Add New User
                        </a>
                    </div>
                </div>
            </div>
            <div class="card-body-modern">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>User</th>
                                <th>Contact</th>
                                <th>Role & Permissions</th>
                                <th>Status</th>
                                <th>Last Login</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($users_list)): ?>
                                <tr>
                                    <td colspan="6" class="text-center py-5">
                                        <i class="bi bi-inbox" style="font-size: 3rem; color: var(--text-muted);"></i>
                                        <p class="text-muted mt-3">No users found</p>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($users_list as $user): ?>
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="user-avatar-small me-3">
                                                    <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                                                </div>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
                                                    <br>
                                                    <small class="text-muted">@<?php echo htmlspecialchars($user['username']); ?></small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>
                                            <small>
                                                <i class="bi bi-envelope"></i> <?php echo htmlspecialchars($user['email']); ?>
                                                <?php if ($user['phone']): ?>
                                                    <br><i class="bi bi-telephone"></i> <?php echo htmlspecialchars($user['phone']); ?>
                                                <?php endif; ?>
                                            </small>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo getUserPermissionColor($user['permission_level']); ?> permission-badge-small">
                                                Level <?php echo $user['permission_level']; ?> -
                                                <?php echo htmlspecialchars($user['user_role']) ?: ($permission_levels[$user['permission_level']] ?? 'Unknown'); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo getUserStatusBadge($user['status']); ?>">
                                                <?php echo ucfirst($user['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($user['last_login']): ?>
                                                <small>
                                                    <i class="bi bi-clock"></i>
                                                    <?php
                                                    $dt = new DateTime($user['last_login']);
                                                    echo $dt->format('d M Y, H:i');
                                                    ?>
                                                    <br>
                                                    <span class="text-muted">
                                                        <i class="bi bi-globe"></i> <?php echo htmlspecialchars($user['last_login_ip']); ?>
                                                    </span>
                                                </small>
                                            <?php else: ?>
                                                <small class="text-muted">Never logged in</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($user['permission_level'] <= $current_user['permission_level']): ?>
                                                <a href="modules/admin/edit_user.php?id=<?php echo $user['id']; ?>"
                                                   class="btn btn-sm btn-outline-primary" title="Edit User">
                                                    <i class="bi bi-pencil"></i>
                                                </a>
                                                <?php if ($user['id'] != $current_user['id']): ?>
                                                    <?php if ($current_user['permission_level'] >= 100): ?>
                                                        <a href="modules/admin/delete_user.php?id=<?php echo $user['id']; ?>"
                                                           class="btn btn-sm btn-outline-danger" title="Delete/Deactivate User (Permanent delete available)">
                                                            <i class="bi bi-trash"></i><i class="bi bi-shield-fill-exclamation text-danger" style="font-size: 0.7em; margin-left: -3px;"></i>
                                                        </a>
                                                    <?php else: ?>
                                                        <a href="modules/admin/delete_user.php?id=<?php echo $user['id']; ?>"
                                                           class="btn btn-sm btn-outline-danger" title="Deactivate User">
                                                            <i class="bi bi-trash"></i>
                                                        </a>
                                                    <?php endif; ?>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="text-muted small">
                                                    <i class="bi bi-lock"></i> No access
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Permission Levels Reference -->
        <div class="modern-card mt-4">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-shield-check"></i> Permission Levels Reference
                </h5>
            </div>
            <div class="card-body-modern">
                <div class="row">
                    <?php foreach ($permission_levels as $level => $name): ?>
                        <div class="col-md-3 mb-3">
                            <div class="card border-<?php echo getUserPermissionColor($level); ?> permission-card"
                                 style="cursor: pointer; transition: transform 0.2s;"
                                 onclick="showPermissionDetails(<?php echo $level; ?>)"
                                 onmouseover="this.style.transform='translateY(-3px)'"
                                 onmouseout="this.style.transform='translateY(0)'">
                                <div class="card-body">
                                    <h6 class="text-<?php echo getUserPermissionColor($level); ?>">
                                        Level <?php echo $level; ?>
                                    </h6>
                                    <p class="mb-0 small"><?php echo $name; ?></p>
                                    <small class="text-muted d-block mt-2">
                                        <i class="bi bi-info-circle"></i> Click for details
                                    </small>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- Permission Details Modal -->
        <div class="modal fade" id="permissionDetailsModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">
                            <i class="bi bi-shield-check"></i> <span id="modalPermissionTitle"></span>
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div id="permissionDetailsContent"></div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>

        <script>
        const permissionDetails = {
            100: {
                name: 'System Administrator',
                color: 'danger',
                description: 'Full system access with complete control over all features and settings.',
                permissions: [
                    'Complete access to all system modules and features',
                    'Create, edit, and permanently delete all records',
                    'Manage all user accounts including other administrators',
                    'Configure system settings and SMTP',
                    'Access all reports and audit logs',
                    'Manage CoS records and assignments',
                    'Manage work locations and departments',
                    'Override all security restrictions',
                    'Perform database operations and migrations',
                    'Access system health and diagnostics'
                ],
                restrictions: 'None - unrestricted access to all system functions'
            },
            90: {
                name: 'HR Director',
                color: 'danger',
                description: 'Senior HR management with extensive permissions across HR operations.',
                permissions: [
                    'Manage all sponsored workers (add, edit, soft delete)',
                    'Create and manage user accounts (Level 90 and below)',
                    'Full access to CoS management',
                    'Approve and manage holiday requests',
                    'Access all HR reports and compliance data',
                    'Manage work locations',
                    'View and edit all worker documents',
                    'Perform RTW checks and audits',
                    'Access audit logs',
                    'Manage attendance records'
                ],
                restrictions: 'Cannot permanently delete records or modify system settings'
            },
            80: {
                name: 'HR Manager',
                color: 'warning',
                description: 'HR management role with permissions for daily HR operations.',
                permissions: [
                    'Add and edit sponsored workers',
                    'Soft delete worker records',
                    'Manage CoS assignments',
                    'Process holiday requests',
                    'Perform RTW checks',
                    'Upload and manage worker documents',
                    'Generate standard reports',
                    'View audit logs',
                    'Manage attendance records',
                    'Update worker employment details'
                ],
                restrictions: 'Cannot create users, delete permanently, or access system settings'
            },
            70: {
                name: 'Department Manager',
                color: 'warning',
                description: 'Departmental management with permissions for their team.',
                permissions: [
                    'View workers in their department',
                    'Edit worker basic information',
                    'Approve/reject holiday requests for their team',
                    'View department attendance records',
                    'Generate department reports',
                    'Add notes to worker records',
                    'View RTW check status',
                    'Access department documents',
                    'Update work schedules'
                ],
                restrictions: 'Cannot add/delete workers, manage CoS, or access other departments'
            },
            60: {
                name: 'Supervisor',
                color: 'info',
                description: 'Supervisory role with limited management capabilities.',
                permissions: [
                    'View workers in their area',
                    'Record daily attendance',
                    'Submit holiday requests on behalf of workers',
                    'View worker schedules',
                    'Add shift notes',
                    'View basic worker information',
                    'Access assigned location information',
                    'View department reports'
                ],
                restrictions: 'Cannot edit worker records, approve holidays, or access sensitive data'
            },
            50: {
                name: 'Staff',
                color: 'info',
                description: 'Standard staff member with basic system access.',
                permissions: [
                    'View own worker record',
                    'Submit own holiday requests',
                    'View own attendance records',
                    'Update own contact information',
                    'View assigned work location',
                    'Access own documents',
                    'View company holidays',
                    'View own RTW status'
                ],
                restrictions: 'Cannot view other workers or access management features'
            },
            40: {
                name: 'Read Only',
                color: 'secondary',
                description: 'View-only access for auditors or external reviewers.',
                permissions: [
                    'View worker lists (limited fields)',
                    'View CoS records',
                    'View work locations',
                    'Access public reports',
                    'View compliance status',
                    'View attendance summaries'
                ],
                restrictions: 'Cannot create, edit, or delete any records'
            },
            30: {
                name: 'Limited',
                color: 'secondary',
                description: 'Minimal access for temporary or restricted users.',
                permissions: [
                    'View dashboard only',
                    'Access assigned documents',
                    'View own profile',
                    'View public announcements'
                ],
                restrictions: 'Extremely limited access - no management features available'
            }
        };

        function showPermissionDetails(level) {
            const details = permissionDetails[level];
            if (!details) return;

            const modal = new bootstrap.Modal(document.getElementById('permissionDetailsModal'));

            document.getElementById('modalPermissionTitle').innerHTML =
                `<span class="badge bg-${details.color}">Level ${level}</span> ${details.name}`;

            let content = `
                <div class="alert alert-${details.color} alert-dismissible fade show">
                    <i class="bi bi-info-circle"></i>
                    <strong>Description:</strong> ${details.description}
                </div>

                <h6 class="mt-4 mb-3"><i class="bi bi-check-circle text-success"></i> Permissions & Capabilities</h6>
                <ul class="list-group mb-4">
                    ${details.permissions.map(perm => `
                        <li class="list-group-item">
                            <i class="bi bi-check2 text-success me-2"></i>${perm}
                        </li>
                    `).join('')}
                </ul>

                <h6 class="mb-3"><i class="bi bi-x-circle text-danger"></i> Restrictions</h6>
                <div class="alert alert-warning">
                    <i class="bi bi-exclamation-triangle"></i> ${details.restrictions}
                </div>
            `;

            document.getElementById('permissionDetailsContent').innerHTML = content;
            modal.show();
        }
        </script>
        <?php endif; ?>

        <!-- SMTP Settings Tab -->
        <?php if ($active_tab === 'smtp'): ?>
        <div class="modern-card">
            <div class="card-header-modern">
                <h5 class="card-title-modern">
                    <i class="bi bi-envelope-at"></i> SMTP Email Settings
                </h5>
            </div>
            <div class="card-body-modern">
                <form method="POST">
                    <input type="hidden" name="action" value="save_smtp">

                    <!-- SMTP Status -->
                    <div class="alert alert-info mb-4">
                        <i class="bi bi-info-circle"></i> Configure SMTP settings to enable email functionality across all modules.
                    </div>

                    <div class="form-check form-switch mb-4">
                        <input type="checkbox" class="form-check-input" id="smtp_enabled" name="smtp_enabled"
                               <?php echo (!empty($settings['smtp_enabled']) && $settings['smtp_enabled'] == '1') ? 'checked' : ''; ?>>
                        <label class="form-check-label" for="smtp_enabled">
                            <strong>Enable SMTP Email</strong>
                        </label>
                        <small class="d-block text-muted">Enable sending emails through SMTP server</small>
                    </div>

                    <hr class="my-4">

                    <!-- SMTP Server Configuration -->
                    <h6 class="mb-3"><i class="bi bi-server"></i> Server Configuration</h6>

                    <div class="row">
                        <div class="col-md-8 mb-3">
                            <label class="form-label"><strong>SMTP Host</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="smtp_host"
                                   value="<?php echo htmlspecialchars($settings['smtp_host'] ?? ''); ?>"
                                   placeholder="smtp.gmail.com or mail.yourdomain.com">
                            <small class="text-muted">Your SMTP server address</small>
                        </div>

                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Port</strong> <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" name="smtp_port"
                                   value="<?php echo htmlspecialchars($settings['smtp_port'] ?? '587'); ?>">
                            <small class="text-muted">Common: 587 (TLS), 465 (SSL), 25</small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Encryption</strong></label>
                            <select class="form-select" name="smtp_secure">
                                <option value="none" <?php echo (isset($settings['smtp_secure']) && $settings['smtp_secure'] == 'none') ? 'selected' : ''; ?>>None</option>
                                <option value="tls" <?php echo (!isset($settings['smtp_secure']) || $settings['smtp_secure'] == 'tls') ? 'selected' : ''; ?>>TLS</option>
                                <option value="ssl" <?php echo (isset($settings['smtp_secure']) && $settings['smtp_secure'] == 'ssl') ? 'selected' : ''; ?>>SSL</option>
                            </select>
                            <small class="text-muted">Encryption method</small>
                        </div>

                        <div class="col-md-8 mb-3">
                            <label class="form-label"><strong>Authentication Required</strong></label>
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" id="smtp_auth_required" name="smtp_auth_required"
                                       <?php echo (!isset($settings['smtp_auth_required']) || $settings['smtp_auth_required'] == '1') ? 'checked' : ''; ?>>
                                <label class="form-check-label" for="smtp_auth_required">
                                    Server requires authentication (username & password)
                                </label>
                            </div>
                        </div>
                    </div>

                    <hr class="my-4">

                    <!-- Authentication Credentials -->
                    <h6 class="mb-3"><i class="bi bi-key"></i> Authentication Credentials</h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>SMTP Username</strong></label>
                            <input type="text" class="form-control" name="smtp_username"
                                   value="<?php echo htmlspecialchars($settings['smtp_username'] ?? ''); ?>"
                                   placeholder="your-email@domain.com">
                            <small class="text-muted">Usually your email address</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>SMTP Password</strong></label>
                            <input type="password" class="form-control" name="smtp_password"
                                   value="<?php echo htmlspecialchars($settings['smtp_password'] ?? ''); ?>"
                                   placeholder="••••••••">
                            <small class="text-muted">Your email password or app-specific password</small>
                        </div>
                    </div>

                    <hr class="my-4">

                    <!-- Email Settings -->
                    <h6 class="mb-3"><i class="bi bi-envelope"></i> Email Settings</h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>From Email</strong> <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="smtp_from_email"
                                   value="<?php echo htmlspecialchars($settings['smtp_from_email'] ?? $settings['email'] ?? ''); ?>"
                                   placeholder="noreply@yourdomain.com">
                            <small class="text-muted">Default sender email address</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>From Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="smtp_from_name"
                                   value="<?php echo htmlspecialchars($settings['smtp_from_name'] ?? $settings['company_name'] ?? ''); ?>"
                                   placeholder="Company Name">
                            <small class="text-muted">Default sender name</small>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label"><strong>Reply-To Email</strong></label>
                        <input type="email" class="form-control" name="smtp_reply_to"
                               value="<?php echo htmlspecialchars($settings['smtp_reply_to'] ?? ''); ?>"
                               placeholder="hr@yourdomain.com">
                        <small class="text-muted">Optional: Where replies should be sent</small>
                    </div>

                    <hr class="my-4">

                    <!-- Test Email Section -->
                    <h6 class="mb-3"><i class="bi bi-send"></i> Test Email Configuration</h6>

                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle"></i> Save your SMTP settings first before sending a test email.
                    </div>

                    <div class="row align-items-end">
                        <div class="col-md-8 mb-3">
                            <label class="form-label">Test Email Address</label>
                            <input type="email" class="form-control" id="test_email" placeholder="test@example.com">
                        </div>
                        <div class="col-md-4 mb-3">
                            <button type="button" class="btn btn-outline-primary w-100" onclick="sendTestEmail()">
                                <i class="bi bi-send"></i> Send Test Email
                            </button>
                        </div>
                    </div>

                    <div id="test_result" class="mt-3"></div>

                    <hr class="my-4">

                    <!-- Common SMTP Providers Help -->
                    <div class="accordion" id="smtpHelpAccordion">
                        <div class="accordion-item">
                            <h2 class="accordion-header">
                                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#smtpHelp">
                                    <i class="bi bi-question-circle me-2"></i> Common SMTP Provider Settings
                                </button>
                            </h2>
                            <div id="smtpHelp" class="accordion-collapse collapse" data-bs-parent="#smtpHelpAccordion">
                                <div class="accordion-body">
                                    <table class="table table-sm">
                                        <thead>
                                            <tr>
                                                <th>Provider</th>
                                                <th>SMTP Host</th>
                                                <th>Port</th>
                                                <th>Encryption</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <tr>
                                                <td><strong>Gmail</strong></td>
                                                <td>smtp.gmail.com</td>
                                                <td>587</td>
                                                <td>TLS</td>
                                            </tr>
                                            <tr>
                                                <td><strong>Outlook/Office365</strong></td>
                                                <td>smtp.office365.com</td>
                                                <td>587</td>
                                                <td>TLS</td>
                                            </tr>
                                            <tr>
                                                <td><strong>Yahoo</strong></td>
                                                <td>smtp.mail.yahoo.com</td>
                                                <td>587</td>
                                                <td>TLS</td>
                                            </tr>
                                            <tr>
                                                <td><strong>SendGrid</strong></td>
                                                <td>smtp.sendgrid.net</td>
                                                <td>587</td>
                                                <td>TLS</td>
                                            </tr>
                                        </tbody>
                                    </table>
                                    <div class="alert alert-info mt-3">
                                        <strong>Note:</strong> For Gmail and Yahoo, you may need to use an app-specific password instead of your regular password.
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <hr class="my-4">

                    <div class="d-flex justify-content-end">
                        <button type="submit" class="btn btn-primary">
                            <i class="bi bi-save"></i> Save SMTP Settings
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <script>
        function sendTestEmail() {
            const testEmail = document.getElementById('test_email').value;
            const resultDiv = document.getElementById('test_result');

            if (!testEmail) {
                resultDiv.innerHTML = '<div class="alert alert-danger">Please enter a test email address</div>';
                return;
            }

            resultDiv.innerHTML = '<div class="alert alert-info"><i class="bi bi-hourglass-split"></i> Sending test email...</div>';

            // Create form data
            const formData = new FormData();
            formData.append('action', 'test_smtp');
            formData.append('test_email', testEmail);

            fetch('ajax_test_smtp.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    resultDiv.innerHTML = '<div class="alert alert-success"><i class="bi bi-check-circle"></i> ' + data.message + '</div>';
                } else {
                    resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> ' + data.message + '</div>';
                }
            })
            .catch(error => {
                resultDiv.innerHTML = '<div class="alert alert-danger"><i class="bi bi-x-circle"></i> Error: ' + error.message + '</div>';
            });
        }
        </script>
        <?php endif; ?>

        <!-- Error Logs Tab -->
        <?php if ($active_tab === 'error_logs'): ?>
        <div class="modern-card">
            <div class="card-header-modern">
                <div class="d-flex justify-content-between align-items-center w-100">
                    <h5 class="card-title-modern mb-0">
                        <i class="bi bi-bug-fill"></i> System Error Logs
                    </h5>
                    <div class="ms-auto">
                        <button type="button" class="btn btn-sm btn-outline-secondary me-2" data-bs-toggle="modal" data-bs-target="#errorLogSettingsModal" title="Error Log Settings">
                            <i class="bi bi-gear"></i>
                        </button>
                        <?php if ($current_user['permission_level'] >= 100): ?>
                        <form method="POST" onsubmit="return confirm('Are you sure you want to clear the error log?');" class="mb-0 d-inline-block">
                            <input type="hidden" name="action" value="clear_error_log">
                            <button type="submit" class="btn btn-sm btn-danger">
                                <i class="bi bi-trash"></i> Clear Log
                            </button>
                        </form>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="card-body-modern">
                <!-- Filters -->
                <form method="GET" class="row g-3 mb-4">
                    <input type="hidden" name="tab" value="error_logs">
                    <div class="col-md-3">
                        <label class="form-label">Error Level</label>
                        <select name="level" class="form-select form-select-sm">
                            <option value="">All Levels</option>
                            <?php if (isset($log_levels)): foreach ($log_levels as $lvl): ?>
                            <option value="<?= htmlspecialchars($lvl) ?>" <?= $filter_level === $lvl ? 'selected' : '' ?>>
                                <?= htmlspecialchars($lvl) ?>
                            </option>
                            <?php endforeach; endif; ?>
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label class="form-label">Module</label>
                        <select name="module" class="form-select form-select-sm">
                            <option value="">All Modules</option>
                            <?php if (isset($log_modules)): foreach ($log_modules as $mod): ?>
                            <option value="<?= htmlspecialchars($mod) ?>" <?= $filter_module === $mod ? 'selected' : '' ?>>
                                <?= htmlspecialchars($mod) ?>
                            </option>
                            <?php endforeach; endif; ?>
                        </select>
                    </div>

                    <div class="col-md-3">
                        <label class="form-label">Limit</label>
                        <select name="limit" class="form-select form-select-sm">
                            <option value="15" <?= $limit === 15 ? 'selected' : '' ?>>15</option>
                            <option value="25" <?= $limit === 25 ? 'selected' : '' ?>>25</option>
                            <option value="50" <?= $limit === 50 ? 'selected' : '' ?>>50</option>
                            <option value="100" <?= $limit === 100 ? 'selected' : '' ?>>100</option>
                            <option value="250" <?= $limit === 250 ? 'selected' : '' ?>>250</option>
                            <option value="500" <?= $limit === 500 ? 'selected' : '' ?>>500</option>
                        </select>
                    </div>

                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary btn-sm w-100">
                            <i class="bi bi-funnel"></i> Apply Filters
                        </button>
                    </div>
                </form>

                <!-- Error Log Table -->
                <?php if (empty($error_logs)): ?>
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> No error logs found
                </div>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover table-sm">
                        <thead class="table-dark">
                            <tr>
                                <th style="width: 150px;">Timestamp</th>
                                <th style="width: 80px;">Level</th>
                                <th style="width: 100px;">Module</th>
                                <th style="width: 80px;">User</th>
                                <th style="width: 120px;">IP Address</th>
                                <th>Message</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($error_logs as $log): ?>
                            <?php
                            // Color code by severity
                            $row_class = '';
                            $badge_class = 'bg-secondary';
                            switch (strtoupper($log['level'])) {
                                case 'CRITICAL':
                                case 'EMERGENCY':
                                case 'ALERT':
                                    $row_class = 'table-danger';
                                    $badge_class = 'bg-danger';
                                    break;
                                case 'ERROR':
                                    $row_class = 'table-warning';
                                    $badge_class = 'bg-warning text-dark';
                                    break;
                                case 'WARNING':
                                    $badge_class = 'bg-orange text-white';
                                    break;
                                case 'INFO':
                                    $badge_class = 'bg-info text-white';
                                    break;
                            }
                            ?>
                            <tr class="<?= $row_class ?>">
                                <td class="small"><?= htmlspecialchars($log['timestamp']) ?></td>
                                <td>
                                    <span class="badge <?= $badge_class ?>">
                                        <?= htmlspecialchars($log['level']) ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-primary">
                                        <?= htmlspecialchars($log['module']) ?>
                                    </span>
                                </td>
                                <td class="small"><?= htmlspecialchars($log['user_id']) ?></td>
                                <td class="small font-monospace"><?= htmlspecialchars($log['ip']) ?></td>
                                <td class="small">
                                    <div style="max-height: 100px; overflow-y: auto; word-break: break-all;">
                                        <?= htmlspecialchars($log['message']) ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="mt-3">
                    <small class="text-muted">
                        Showing <?= count($error_logs) ?> most recent error(s)
                        <?= $filter_level ? " | Level: " . htmlspecialchars($filter_level) : '' ?>
                        <?= $filter_module ? " | Module: " . htmlspecialchars($filter_module) : '' ?>
                    </small>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Error Log Settings Modal -->
        <div class="modal fade" id="errorLogSettingsModal" tabindex="-1" aria-labelledby="errorLogSettingsModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="errorLogSettingsModalLabel">
                            <i class="bi bi-gear"></i> Error Log Settings
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <form method="POST" action="?tab=error_logs">
                        <input type="hidden" name="action" value="save_error_log_settings">
                        <div class="modal-body">
                            <div class="mb-3">
                                <label class="form-label">
                                    <i class="bi bi-calendar-x"></i> Auto-Delete After (Days)
                                </label>
                                <input type="number"
                                       class="form-control"
                                       name="auto_delete_days"
                                       value="<?= (int)($error_log_settings['error_log_auto_delete_days'] ?? 30) ?>"
                                       min="1"
                                       max="365"
                                       required>
                                <small class="text-muted">Error logs older than this will be automatically deleted (1-365 days)</small>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">
                                    <i class="bi bi-list-ol"></i> Entries Per Page (Default)
                                </label>
                                <select class="form-select" name="entries_per_page" required>
                                    <?php
                                    $current_per_page = (int)($error_log_settings['error_log_entries_per_page'] ?? 15);
                                    $per_page_options = [15, 25, 50, 100, 250, 500];
                                    foreach ($per_page_options as $option):
                                    ?>
                                    <option value="<?= $option ?>" <?= $current_per_page === $option ? 'selected' : '' ?>>
                                        <?= $option ?> entries
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Recommended: 15 entries per page for better performance</small>
                            </div>

                            <div class="alert alert-info">
                                <i class="bi bi-info-circle"></i>
                                <strong>Note:</strong> Changes will take effect immediately after saving.
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="bi bi-x"></i> Cancel
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-save"></i> Save Settings
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <?php endif; ?>

    </div>
</div>

<style>
.bg-orange {
    background-color: #fd7e14 !important;
}
.font-monospace {
    font-family: 'Courier New', monospace;
}
.settings-tabs {
    border: none;
    background: var(--card-bg);
    padding: 1rem 1rem 0 1rem;
    border-radius: 12px 12px 0 0;
}

.settings-tabs .nav-link {
    border: none;
    color: var(--text-secondary);
    padding: 0.75rem 1.5rem;
    border-radius: 8px 8px 0 0;
    transition: all 0.2s;
}

.settings-tabs .nav-link:hover {
    background: var(--bg-secondary);
    color: var(--text-primary);
}

.settings-tabs .nav-link.active {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    font-weight: 600;
}

.module-card {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 15px;
    overflow: hidden;
    transition: all 0.3s ease;
}

.module-card:hover {
    transform: translateY(-4px);
    box-shadow: 0 12px 24px var(--shadow-lg);
}

.module-header {
    padding: 1.5rem;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
}

.module-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.75rem;
}

.module-badge {
    padding: 0.35rem 0.75rem;
    border-radius: 50px;
    font-size: 0.75rem;
    font-weight: 600;
}

.module-body {
    padding: 0 1.5rem 1.5rem;
    min-height: 140px;
}

.module-title {
    font-size: 1.1rem;
    font-weight: 700;
    color: var(--text-primary);
    margin-bottom: 0.75rem;
}

.module-description {
    color: var(--text-primary);
    opacity: 0.8;
    font-size: 0.875rem;
    margin-bottom: 1rem;
    line-height: 1.6;
    min-height: 60px;
    display: block;
}

.module-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 1rem;
    font-size: 0.8rem;
    color: var(--text-muted);
}

.module-version, .module-author {
    display: inline-flex;
    align-items: center;
    gap: 0.25rem;
}

.module-footer {
    padding: 1rem 1.5rem;
    border-top: 1px solid var(--border-color);
}

.info-item label {
    display: block;
    margin-bottom: 0.25rem;
}
</style>

<script>
// AJAX Module Actions
function handleModuleAction(form, action, moduleId) {
    const formData = new FormData();
    formData.append('action', action);
    formData.append('module_id', moduleId);

    // Show loading state
    const button = form.querySelector('button[type="submit"]');
    const originalHTML = button.innerHTML;
    button.disabled = true;
    button.innerHTML = '<i class="bi bi-hourglass-split"></i> Processing...';

    fetch('ajax_module_action.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show success message
            showMessage(data.message, 'success');
            // Reload page after short delay
            setTimeout(() => {
                window.location.reload();
            }, 1000);
        } else {
            // Show error message
            showMessage(data.message, 'danger');
            button.disabled = false;
            button.innerHTML = originalHTML;
        }
    })
    .catch(error => {
        showMessage('Error: ' + error.message, 'danger');
        button.disabled = false;
        button.innerHTML = originalHTML;
    });

    return false; // Prevent form submission
}

function showMessage(message, type) {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show modern-alert`;
    alertDiv.setAttribute('role', 'alert');
    alertDiv.innerHTML = `
        <i class="bi bi-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}-fill"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;

    const container = document.querySelector('.dashboard-container');
    const welcomeSection = container.querySelector('.welcome-section');
    container.insertBefore(alertDiv, welcomeSection.nextSibling);

    // Auto-dismiss after 5 seconds
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}

// Handle module reordering
function handleReorder(button, moduleId, action) {
    const originalHTML = button.innerHTML;
    button.disabled = true;
    button.innerHTML = '<span class="spinner-border spinner-border-sm"></span>';

    const formData = new FormData();
    formData.append('module_id', moduleId);
    formData.append('action', action);

    fetch('ajax_reorder_modules.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showMessage(data.message, 'success');
            setTimeout(() => {
                window.location.reload();
            }, 500);
        } else {
            showMessage(data.message, 'danger');
            button.disabled = false;
            button.innerHTML = originalHTML;
        }
    })
    .catch(error => {
        showMessage('Error: ' + error.message, 'danger');
        button.disabled = false;
        button.innerHTML = originalHTML;
    });
}

// Add event listeners to all module action forms
document.addEventListener('DOMContentLoaded', function() {
    document.querySelectorAll('form[data-module-action]').forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            const action = this.dataset.moduleAction;
            const moduleId = this.querySelector('input[name="module_id"]').value;
            handleModuleAction(this, action, moduleId);
        });
    });

    // Add event listeners for reorder buttons
    document.querySelectorAll('.reorder-btn').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const moduleId = this.dataset.moduleId;
            const action = this.dataset.action;
            handleReorder(this, moduleId, action);
        });
    });
});

// Functions for Department and Role editing
function editDepartment(dept) {
    document.getElementById('edit_dept_id').value = dept.id;
    document.getElementById('edit_dept_name').value = dept.name;
    document.getElementById('edit_dept_description').value = dept.description || '';
    document.getElementById('edit_dept_active').checked = dept.is_active == 1;
    new bootstrap.Modal(document.getElementById('editDepartmentModal')).show();
}

function editRole(role) {
    document.getElementById('edit_role_id').value = role.id;
    document.getElementById('edit_role_title').value = role.title;
    document.getElementById('edit_role_description').value = role.description || '';
    document.getElementById('edit_role_level').value = role.level;
    document.getElementById('edit_role_dept').value = role.department_id || '';
    document.getElementById('edit_role_active').checked = role.is_active == 1;
    new bootstrap.Modal(document.getElementById('editRoleModal')).show();
}
</script>

<?php include 'includes/footer.php'; ?>
