<?php
/**
 * SCMS Dynamic Modular Router
 * Central routing system - loads modules dynamically from database
 * No hardcoded routes - completely database-driven
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';
require_once 'includes/helpers.php';

// Require authentication for all module access
require_authentication();

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: login.php');
    exit;
}

// Get routing parameters from URL
$module_id = $_GET['module'] ?? $_GET['m'] ?? '';
$action = $_GET['action'] ?? $_GET['a'] ?? 'list';

// Validate module_id
if (empty($module_id)) {
    header('Location: dashboard.php');
    exit;
}

// For attendance module, use calendar_by_location as default instead of list
if ($module_id === 'attendance' && ($action === 'list' || $action === 'calendar')) {
    $action = 'calendar_by_location';
}

// Get database connection
$pdo = get_database_connection();

// Check if module exists and is active in database
try {
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "modules_registry WHERE module_id = ? AND status = 'active' LIMIT 1");
    $stmt->execute([$module_id]);
    $module_db = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$module_db) {
        http_response_code(404);
        include 'includes/header.php';
        echo '<div class="dashboard-container">';
        echo '<div class="alert alert-danger">';
        echo '<i class="bi bi-exclamation-triangle"></i> ';
        echo '<strong>Module Not Found</strong><br>';
        echo 'The module "' . htmlspecialchars($module_id) . '" is not installed or inactive.';
        echo '</div>';
        echo '</div>';
        include 'includes/footer.php';
        exit;
    }
} catch (Exception $e) {
    http_response_code(500);
    die('Database error: ' . $e->getMessage());
}

// Load module.json for configuration
$module_json_path = __DIR__ . "/modules/{$module_id}/module.json";
if (!file_exists($module_json_path)) {
    http_response_code(404);
    include 'includes/header.php';
    echo '<div class="dashboard-container">';
    echo '<div class="alert alert-danger">';
    echo '<i class="bi bi-exclamation-triangle"></i> ';
    echo '<strong>Module Configuration Missing</strong><br>';
    echo 'The module configuration file (module.json) was not found for "' . htmlspecialchars($module_id) . '".';
    echo '</div>';
    echo '</div>';
    include 'includes/footer.php';
    exit;
}

$module_info = json_decode(file_get_contents($module_json_path), true);
if (!$module_info) {
    http_response_code(500);
    die('Module configuration error: Invalid JSON in module.json');
}

// Check user permissions for this action
$required_permission = 30; // Default view permission
if (isset($module_info['permissions'][$action])) {
    $required_permission = $module_info['permissions'][$action];
} elseif (isset($module_info['permissions']['view'])) {
    $required_permission = $module_info['permissions']['view'];
}

if ($current_user['permission_level'] < $required_permission) {
    http_response_code(403);
    include 'includes/header.php';
    echo '<div class="dashboard-container">';
    echo '<div class="alert alert-danger">';
    echo '<i class="bi bi-shield-lock"></i> ';
    echo '<strong>Access Denied</strong><br>';
    echo 'You do not have sufficient permissions to access this module.';
    echo '<br>Required permission level: ' . $required_permission;
    echo '<br>Your permission level: ' . $current_user['permission_level'];
    echo '</div>';
    echo '</div>';
    include 'includes/footer.php';
    exit;
}

// Set page context for header
$page_title = $module_db['module_name'] ?? $module_info['name'];
$current_page = $module_id;

// Include header with sidebar
include 'includes/header.php';

// Define constants so module knows it's being loaded through router
define('MODULE_EMBEDDED', true);
define('MODULE_ACTION', $action);

// Determine which file to load based on action
$action_file_map = [
    'list' => 'list.php',
    'add' => 'add.php',
    'edit' => 'edit.php',
    'view' => 'view.php',
    'delete' => 'delete.php',
    'calendar' => 'calendar.php',
    'calendar_by_location' => 'calendar_by_location.php',
    'dashboard' => 'dashboard.php',
    'report' => 'report.php',
    'report_monthly' => 'report_monthly.php',
    'bulk_mark' => 'bulk_mark.php',
    'absence_alerts' => 'absence_alerts.php',
    'upload' => 'upload.php',
    'settings' => 'settings.php'
];

// Get file to load (from map or construct filename)
$file_to_load = $action_file_map[$action] ?? $action . '.php';
$module_file_path = __DIR__ . "/modules/{$module_id}/{$file_to_load}";

// Check if file exists
if (!file_exists($module_file_path)) {
    // Fall back to list.php if action file doesn't exist
    $module_file_path = __DIR__ . "/modules/{$module_id}/list.php";
    if (!file_exists($module_file_path)) {
        echo '<div class="dashboard-container">';
        echo '<div class="alert alert-danger">';
        echo '<i class="bi bi-exclamation-triangle"></i> ';
        echo '<strong>Module File Not Found</strong><br>';
        echo 'The file "' . htmlspecialchars($file_to_load) . '" was not found in the "' . htmlspecialchars($module_id) . '" module.';
        echo '</div>';
        echo '</div>';
        include 'includes/footer.php';
        exit;
    }
}

// Load the module file with error handling
echo '<div class="dashboard-container">';
try {
    ob_start();
    include $module_file_path;
    $module_output = ob_get_clean();
    echo $module_output;
} catch (Exception $e) {
    ob_end_clean();
    echo '<div class="alert alert-danger">';
    echo '<i class="bi bi-exclamation-triangle"></i> ';
    echo '<strong>Module Error</strong><br>';
    echo htmlspecialchars($e->getMessage());
    echo '</div>';
} catch (Error $e) {
    ob_end_clean();
    echo '<div class="alert alert-danger">';
    echo '<i class="bi bi-exclamation-triangle"></i> ';
    echo '<strong>Fatal Error</strong><br>';
    echo htmlspecialchars($e->getMessage());
    echo '<br>File: ' . $e->getFile() . ':' . $e->getLine();
    echo '</div>';
}
echo '</div>';

// Include footer
include 'includes/footer.php';
?>
