<?php
/**
 * User Profile Management
 * Update personal info, change password, change PIN
 */

session_start();
require_once 'config/database.php';
require_once 'includes/database.php';
require_once 'includes/session_manager.php';

// Require authentication
require_authentication();

$page_title = 'My Profile';
$current_page = 'profile';

// Get current user
$current_user = get_logged_in_user();
$user_id = $current_user['id'];

$message = '';
$message_type = 'success';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    // Update Profile
    if ($action === 'update_profile') {
        $first_name = trim($_POST['first_name'] ?? '');
        $last_name = trim($_POST['last_name'] ?? '');
        $email = trim($_POST['email'] ?? '');
        $phone = trim($_POST['phone'] ?? '');

        if (empty($first_name) || empty($last_name) || empty($email)) {
            $message = 'First name, last name, and email are required.';
            $message_type = 'danger';
        } else {
            try {
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "users SET first_name = ?, last_name = ?, email = ?, phone = ? WHERE id = ?");
                if ($stmt->execute([$first_name, $last_name, $email, $phone, $user_id])) {
                    $_SESSION['first_name'] = $first_name;
                    $_SESSION['last_name'] = $last_name;
                    $_SESSION['email'] = $email;
                    $current_user['first_name'] = $first_name;
                    $current_user['last_name'] = $last_name;
                    $current_user['email'] = $email;
                    $current_user['phone'] = $phone;

                    // Log the action
                    $log_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs (user_id, action, ip_address) VALUES (?, 'Profile updated', ?)");
                    $log_stmt->execute([$user_id, $_SERVER['REMOTE_ADDR']]);

                    $message = 'Profile updated successfully!';
                    $message_type = 'success';
                }
            } catch (PDOException $e) {
                $message = 'Error updating profile: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    // Change Password
    if ($action === 'change_password') {
        $current_password = $_POST['current_password'] ?? '';
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';

        if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
            $message = 'All password fields are required.';
            $message_type = 'danger';
        } elseif (!password_verify($current_password, $current_user['password_hash'])) {
            $message = 'Current password is incorrect.';
            $message_type = 'danger';
        } elseif ($new_password !== $confirm_password) {
            $message = 'New passwords do not match.';
            $message_type = 'danger';
        } elseif (strlen($new_password) < 8) {
            $message = 'New password must be at least 8 characters.';
            $message_type = 'danger';
        } else {
            try {
                $new_hash = password_hash($new_password, PASSWORD_BCRYPT);
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "users SET password_hash = ? WHERE id = ?");
                if ($stmt->execute([$new_hash, $user_id])) {
                    // Log the action
                    $log_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs (user_id, action, ip_address) VALUES (?, 'Password changed', ?)");
                    $log_stmt->execute([$user_id, $_SERVER['REMOTE_ADDR']]);

                    $message = 'Password changed successfully!';
                    $message_type = 'success';
                }
            } catch (PDOException $e) {
                $message = 'Error changing password: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }

    // Change PIN
    if ($action === 'change_pin') {
        $current_pin = $_POST['current_pin'] ?? '';
        $new_pin = $_POST['new_pin'] ?? '';
        $confirm_pin = $_POST['confirm_pin'] ?? '';

        if (empty($current_pin) || empty($new_pin) || empty($confirm_pin)) {
            $message = 'All PIN fields are required.';
            $message_type = 'danger';
        } elseif (!password_verify($current_pin, $current_user['pin_hash'])) {
            $message = 'Current PIN is incorrect.';
            $message_type = 'danger';
        } elseif ($new_pin !== $confirm_pin) {
            $message = 'New PINs do not match.';
            $message_type = 'danger';
        } elseif (!preg_match('/^\d{4}$/', $new_pin)) {
            $message = 'PIN must be exactly 4 digits.';
            $message_type = 'danger';
        } else {
            try {
                $new_pin_hash = password_hash($new_pin, PASSWORD_BCRYPT);
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "users SET pin_hash = ? WHERE id = ?");
                if ($stmt->execute([$new_pin_hash, $user_id])) {
                    // Log the action
                    $log_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "audit_logs (user_id, action, ip_address) VALUES (?, 'PIN changed', ?)");
                    $log_stmt->execute([$user_id, $_SERVER['REMOTE_ADDR']]);

                    $message = 'PIN changed successfully!';
                    $message_type = 'success';
                }
            } catch (PDOException $e) {
                $message = 'Error changing PIN: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }
}

// Get recent activity
$activity_stmt = $pdo->prepare("SELECT action, ip_address, created_at FROM " . DB_PREFIX . "audit_logs WHERE user_id = ? ORDER BY created_at DESC LIMIT 10");
$activity_stmt->execute([$user_id]);
$recent_activity = $activity_stmt->fetchAll();

include 'includes/header.php';
?>

<div class="dashboard-container">
    <!-- Page Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-person-circle"></i> My Profile
            </h1>
            <p class="welcome-subtitle">Manage your account settings and security</p>
        </div>
    </div>

    <!-- Alert Message -->
    <?php if ($message): ?>
    <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show modern-alert" role="alert">
        <i class="bi bi-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-triangle'; ?>-fill"></i>
        <?php echo htmlspecialchars($message); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <div class="row g-4">
        <!-- Personal Information -->
        <div class="col-lg-8">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-person-badge"></i> Personal Information
                    </h5>
                </div>
                <div class="card-body-modern">
                    <form method="POST">
                        <input type="hidden" name="action" value="update_profile">

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label"><strong>First Name</strong></label>
                                <input type="text" class="form-control" name="first_name"
                                       value="<?php echo htmlspecialchars($current_user['first_name'] ?? ''); ?>"
                                       required minlength="2">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label"><strong>Last Name</strong></label>
                                <input type="text" class="form-control" name="last_name"
                                       value="<?php echo htmlspecialchars($current_user['last_name'] ?? ''); ?>"
                                       required minlength="2">
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>Email Address</strong></label>
                            <input type="email" class="form-control" name="email"
                                   value="<?php echo htmlspecialchars($current_user['email'] ?? ''); ?>"
                                   required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>Phone Number</strong></label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?php echo htmlspecialchars($current_user['phone'] ?? ''); ?>"
                                   placeholder="+44 1234 567890">
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label"><strong>Username</strong></label>
                                <input type="text" class="form-control"
                                       value="<?php echo htmlspecialchars($current_user['username']); ?>"
                                       disabled>
                                <small class="text-muted">Username cannot be changed</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label"><strong>Permission Level</strong></label>
                                <input type="text" class="form-control"
                                       value="<?php echo $current_user['permission_level'] == 100 ? 'Administrator' : 'User'; ?>"
                                       disabled>
                            </div>
                        </div>

                        <div class="text-end">
                            <button type="submit" class="btn-modern-primary">
                                <i class="bi bi-check-circle"></i> Update Profile
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Change Password -->
            <div class="modern-card mt-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-lock"></i> Change Password
                    </h5>
                </div>
                <div class="card-body-modern">
                    <form method="POST" id="passwordForm">
                        <input type="hidden" name="action" value="change_password">

                        <div class="mb-3">
                            <label class="form-label"><strong>Current Password</strong></label>
                            <input type="password" class="form-control" name="current_password" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>New Password</strong></label>
                            <input type="password" class="form-control" name="new_password"
                                   id="new_password" required minlength="8">
                            <small class="text-muted">Minimum 8 characters</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>Confirm New Password</strong></label>
                            <input type="password" class="form-control" name="confirm_password"
                                   id="confirm_password" required>
                        </div>

                        <div class="text-end">
                            <button type="submit" class="btn-modern-primary">
                                <i class="bi bi-shield-check"></i> Change Password
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Change PIN -->
            <div class="modern-card mt-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-key"></i> Change Security PIN
                    </h5>
                </div>
                <div class="card-body-modern">
                    <form method="POST" id="pinForm">
                        <input type="hidden" name="action" value="change_pin">

                        <div class="mb-3">
                            <label class="form-label"><strong>Current PIN</strong></label>
                            <input type="password" class="form-control" name="current_pin"
                                   maxlength="4" pattern="\d{4}" required>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>New PIN</strong></label>
                            <input type="password" class="form-control" name="new_pin"
                                   id="new_pin" maxlength="4" pattern="\d{4}" required>
                            <small class="text-muted">4-digit numeric PIN</small>
                        </div>

                        <div class="mb-3">
                            <label class="form-label"><strong>Confirm New PIN</strong></label>
                            <input type="password" class="form-control" name="confirm_pin"
                                   id="confirm_pin" maxlength="4" pattern="\d{4}" required>
                        </div>

                        <div class="text-end">
                            <button type="submit" class="btn-modern-primary">
                                <i class="bi bi-shield-check"></i> Change PIN
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="col-lg-4">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-clock-history"></i> Recent Activity
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($recent_activity)): ?>
                        <p class="text-muted text-center py-3">No recent activity</p>
                    <?php else: ?>
                        <div class="activity-list">
                            <?php foreach ($recent_activity as $activity): ?>
                            <div class="activity-item">
                                <div class="activity-icon">
                                    <i class="bi bi-check-circle-fill text-success"></i>
                                </div>
                                <div class="activity-content">
                                    <div class="activity-action"><?php echo htmlspecialchars($activity['action']); ?></div>
                                    <div class="activity-meta">
                                        <small class="text-muted">
                                            <?php echo date('d M Y, H:i', strtotime($activity['created_at'])); ?>
                                        </small>
                                        <small class="text-muted">• IP: <?php echo htmlspecialchars($activity['ip_address']); ?></small>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.activity-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.activity-item {
    display: flex;
    gap: 0.75rem;
    padding: 0.75rem;
    background: var(--bg-secondary);
    border-radius: 8px;
}

.activity-icon {
    display: flex;
    align-items: flex-start;
    padding-top: 0.25rem;
}

.activity-content {
    flex: 1;
}

.activity-action {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 0.25rem;
}

.activity-meta {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}
</style>

<script>
// Password match validation
document.getElementById('passwordForm').addEventListener('submit', function(e) {
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = document.getElementById('confirm_password').value;

    if (newPassword !== confirmPassword) {
        e.preventDefault();
        alert('New passwords do not match!');
    }
});

// PIN match validation
document.getElementById('pinForm').addEventListener('submit', function(e) {
    const newPin = document.getElementById('new_pin').value;
    const confirmPin = document.getElementById('confirm_pin').value;

    if (newPin !== confirmPin) {
        e.preventDefault();
        alert('New PINs do not match!');
    }
});
</script>

<?php include 'includes/footer.php'; ?>
