<?php
/**
 * Worker Profile View - Complete with all integrations
 * Shows: Profile, Employment, Sponsorship, RTW, Documents, Notes
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

$page_title = 'Worker Profile';
$current_page = 'workers';

$worker_id = $_GET['id'] ?? '';

if (empty($worker_id)) {
    header('Location: list.php');
    exit;
}

// Fetch worker details
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$worker) {
    header('Location: list.php?error=Worker not found');
    exit;
}

// Parse full_name into parts for display
$name_parts = explode(' ', $worker['full_name']);
$display_first = $name_parts[0] ?? '';
$display_last = count($name_parts) > 1 ? end($name_parts) : '';
$display_middle = '';
if (count($name_parts) > 2) {
    $display_middle = implode(' ', array_slice($name_parts, 1, -1));
}

// Fetch RTW Checks
$stmt = $pdo->prepare("SELECT r.*, CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name
                       FROM " . DB_PREFIX . "worker_rtw_checks r
                       LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
                       WHERE r.worker_id = ?
                       ORDER BY r.check_date DESC");
$stmt->execute([$worker_id]);
$rtw_checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch CoS History
$stmt = $pdo->prepare("SELECT c.*, CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as created_by_name
                       FROM " . DB_PREFIX . "worker_cos_history c
                       LEFT JOIN " . DB_PREFIX . "users u ON c.created_by = u.id
                       WHERE c.worker_id = ?
                       ORDER BY c.created_at DESC");
$stmt->execute([$worker_id]);
$cos_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Documents
$stmt = $pdo->prepare("SELECT d.*,
                       CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as uploaded_by_name,
                       CONCAT(v.first_name, ' ', COALESCE(v.last_name, '')) as verified_by_name
                       FROM " . DB_PREFIX . "worker_documents d
                       LEFT JOIN " . DB_PREFIX . "users u ON d.uploaded_by = u.id
                       LEFT JOIN " . DB_PREFIX . "users v ON d.verified_by = v.id
                       WHERE d.worker_id = ? AND d.deleted_at IS NULL
                       ORDER BY d.uploaded_at DESC");
$stmt->execute([$worker_id]);
$documents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Contact History
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_contact_history
                       WHERE worker_id = ?
                       ORDER BY valid_from DESC");
$stmt->execute([$worker_id]);
$contact_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Employment History
$stmt = $pdo->prepare("SELECT e.*, w.name as location_name,
                       CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as created_by_name
                       FROM " . DB_PREFIX . "worker_employment_history e
                       LEFT JOIN " . DB_PREFIX . "work_locations w ON e.work_location_id = w.id
                       LEFT JOIN " . DB_PREFIX . "users u ON e.created_by = u.id
                       WHERE e.worker_id = ?
                       ORDER BY e.start_date DESC");
$stmt->execute([$worker_id]);
$employment_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Notes
$stmt = $pdo->prepare("SELECT n.*, CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as created_by_name
                       FROM " . DB_PREFIX . "worker_notes n
                       LEFT JOIN " . DB_PREFIX . "users u ON n.created_by = u.id
                       WHERE n.worker_id = ?
                       ORDER BY n.created_at DESC");
$stmt->execute([$worker_id]);
$notes = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate visa expiry status
$days_to_visa_expiry = null;
$visa_status = '';
if (!empty($worker['visa_expiry'])) {
    $days_to_visa_expiry = (strtotime($worker['visa_expiry']) - time()) / 86400;
    if ($days_to_visa_expiry < 0) {
        $visa_status = 'expired';
    } elseif ($days_to_visa_expiry <= 30) {
        $visa_status = 'urgent';
    } elseif ($days_to_visa_expiry <= 90) {
        $visa_status = 'warning';
    } else {
        $visa_status = 'valid';
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <!-- Header Section -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-person-circle"></i> <?php echo htmlspecialchars($worker['full_name']); ?>
            </h1>
            <p class="welcome-subtitle">
                Employee #<?php echo htmlspecialchars($worker['employee_number']); ?> |
                <?php echo htmlspecialchars($worker['job_title'] ?? 'No Title'); ?>
                <?php if ($worker['status'] === 'active'): ?>
                    <span class="badge bg-success ms-2">Active</span>
                <?php else: ?>
                    <span class="badge bg-secondary ms-2"><?php echo ucfirst($worker['status'] ?? ''); ?></span>
                <?php endif; ?>
            </p>
        </div>
        <div class="welcome-actions">
            <a href="edit.php?id=<?php echo $worker_id; ?>" class="btn btn-primary">
                <i class="bi bi-pencil"></i> Edit Profile
            </a>
            <a href="list.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
    </div>

    <!-- Compliance Alerts -->
    <?php if ($visa_status === 'expired' || $visa_status === 'urgent'): ?>
    <div class="alert alert-danger">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>Visa Compliance Alert:</strong>
        <?php if ($visa_status === 'expired'): ?>
            Visa expired <?php echo abs(floor($days_to_visa_expiry)); ?> days ago. Immediate action required.
        <?php else: ?>
            Visa expires in <?php echo floor($days_to_visa_expiry); ?> days. Urgent attention needed.
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <div class="row g-4">
        <!-- Left Column - Core Information -->
        <div class="col-lg-4">
            <!-- Photo & Basic Info Card -->
            <div class="modern-card mb-4">
                <div class="card-body-modern text-center">
                    <?php if (!empty($worker['photo'])): ?>
                        <img src="../../uploads/workers/<?php echo htmlspecialchars($worker['photo']); ?>"
                             alt="Photo" class="rounded-circle mb-3" style="width: 150px; height: 150px; object-fit: cover;">
                    <?php else: ?>
                        <div class="rounded-circle bg-light d-inline-flex align-items-center justify-content-center mb-3"
                             style="width: 150px; height: 150px;">
                            <i class="bi bi-person-fill" style="font-size: 4rem; color: #ccc;"></i>
                        </div>
                    <?php endif; ?>

                    <h4><?php echo htmlspecialchars($worker['full_name']); ?></h4>
                    <p class="text-muted mb-3"><?php echo htmlspecialchars($worker['job_title'] ?? 'No Title'); ?></p>

                    <div class="d-grid gap-2">
                        <?php if (!empty($worker['email'])): ?>
                        <a href="mailto:<?php echo htmlspecialchars($worker['email']); ?>" class="btn btn-outline-primary btn-sm">
                            <i class="bi bi-envelope"></i> Email
                        </a>
                        <?php endif; ?>
                        <?php if (!empty($worker['phone'])): ?>
                        <a href="tel:<?php echo htmlspecialchars($worker['phone']); ?>" class="btn btn-outline-success btn-sm">
                            <i class="bi bi-telephone"></i> Call
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Personal Details Card -->
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-person-badge"></i> Personal Details</h5>
                </div>
                <div class="card-body-modern">
                    <div class="info-row">
                        <span class="info-label">Full Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['full_name']); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Date of Birth:</span>
                        <span class="info-value"><?php echo $worker['date_of_birth'] ? date('d/m/Y', strtotime($worker['date_of_birth'])) : '-'; ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Nationality:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['nationality'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">NI Number:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['ni_number'] ?? '-'); ?></span>
                    </div>
                </div>
            </div>

            <!-- Contact Details Card -->
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-envelope"></i> Contact Details</h5>
                </div>
                <div class="card-body-modern">
                    <div class="info-row">
                        <span class="info-label">Email:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['email'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Phone:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['phone'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Address:</span>
                        <span class="info-value"><?php echo nl2br(htmlspecialchars($worker['address'] ?? '-')); ?></span>
                    </div>
                    <?php if (!empty($worker['emergency_contact']) || !empty($worker['emergency_phone'])): ?>
                    <hr>
                    <div class="info-row">
                        <span class="info-label">Emergency Contact:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['emergency_contact'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Emergency Phone:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['emergency_phone'] ?? '-'); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Right Column - Tabbed Information -->
        <div class="col-lg-8">
            <!-- Navigation Tabs -->
            <ul class="nav nav-tabs modern-tabs mb-4" role="tablist">
                <li class="nav-item">
                    <a class="nav-link active" data-bs-toggle="tab" href="#employment">
                        <i class="bi bi-briefcase"></i> Employment
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#sponsorship">
                        <i class="bi bi-shield-check"></i> Sponsorship
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#rtw">
                        <i class="bi bi-file-earmark-check"></i> RTW (<?php echo count($rtw_checks); ?>)
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#documents">
                        <i class="bi bi-files"></i> Documents (<?php echo count($documents); ?>)
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#notes">
                        <i class="bi bi-journal-text"></i> Notes (<?php echo count($notes); ?>)
                    </a>
                </li>
            </ul>

            <!-- Tab Content -->
            <div class="tab-content">
                <!-- Employment Tab -->
                <div class="tab-pane fade show active" id="employment">
                    <div class="modern-card mb-3">
                        <div class="card-header-modern">
                            <h5 class="card-title-modern"><i class="bi bi-briefcase"></i> Current Employment</h5>
                        </div>
                        <div class="card-body-modern">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Job Title:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['job_title'] ?? '-'); ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Department:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['department'] ?? '-'); ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Contract Type:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['contract_type'] ?? '-'); ?></span>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Start Date:</span>
                                        <span class="info-value"><?php echo $worker['start_date'] ? date('d/m/Y', strtotime($worker['start_date'])) : '-'; ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Status:</span>
                                        <span class="info-value">
                                            <?php if ($worker['status'] === 'active'): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary"><?php echo ucfirst($worker['status'] ?? ''); ?></span>
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <?php if (!empty($employment_history)): ?>
                            <hr>
                            <h6 class="mb-3"><i class="bi bi-clock-history"></i> Employment History</h6>
                            <div class="table-responsive">
                                <table class="table table-sm table-hover">
                                    <thead>
                                        <tr>
                                            <th>Period</th>
                                            <th>Job Title</th>
                                            <th>Department</th>
                                            <th>Change Reason</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($employment_history as $emp): ?>
                                        <tr>
                                            <td>
                                                <?php echo date('d/m/Y', strtotime($emp['start_date'])); ?>
                                                <?php if ($emp['end_date']): ?>
                                                    - <?php echo date('d/m/Y', strtotime($emp['end_date'])); ?>
                                                <?php else: ?>
                                                    <span class="badge bg-success">Current</span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo htmlspecialchars($emp['job_title']); ?></td>
                                            <td><?php echo htmlspecialchars($emp['department']); ?></td>
                                            <td><?php echo ucfirst(str_replace('_', ' ', $emp['change_reason'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Sponsorship Tab -->
                <div class="tab-pane fade" id="sponsorship">
                    <div class="modern-card mb-3">
                        <div class="card-header-modern">
                            <h5 class="card-title-modern"><i class="bi bi-shield-check"></i> Visa & Sponsorship</h5>
                        </div>
                        <div class="card-body-modern">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Visa Type:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['visa_type'] ?? '-'); ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">CoS Number:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['cos_number'] ?? '-'); ?></span>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Visa Expiry:</span>
                                        <span class="info-value">
                                            <?php if (!empty($worker['visa_expiry'])): ?>
                                                <?php echo date('d/m/Y', strtotime($worker['visa_expiry'])); ?>
                                                <?php if ($visa_status === 'expired'): ?>
                                                    <span class="badge bg-danger ms-2">Expired</span>
                                                <?php elseif ($visa_status === 'urgent'): ?>
                                                    <span class="badge bg-warning ms-2"><?php echo floor($days_to_visa_expiry); ?> days</span>
                                                <?php elseif ($visa_status === 'warning'): ?>
                                                    <span class="badge bg-info ms-2"><?php echo floor($days_to_visa_expiry); ?> days</span>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <?php if (!empty($cos_history)): ?>
                            <hr>
                            <h6 class="mb-3"><i class="bi bi-clock-history"></i> CoS History</h6>
                            <div class="table-responsive">
                                <table class="table table-sm table-hover">
                                    <thead>
                                        <tr>
                                            <th>CoS Number</th>
                                            <th>Route</th>
                                            <th>Type</th>
                                            <th>Issue Date</th>
                                            <th>Status</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($cos_history as $cos): ?>
                                        <tr>
                                            <td><strong><?php echo htmlspecialchars($cos['cos_number']); ?></strong></td>
                                            <td><?php echo htmlspecialchars($cos['cos_route'] ?? '-'); ?></td>
                                            <td><?php echo ucfirst($cos['cos_type'] ?? '-'); ?></td>
                                            <td><?php echo $cos['issue_date'] ? date('d/m/Y', strtotime($cos['issue_date'])) : '-'; ?></td>
                                            <td>
                                                <?php
                                                $status_colors = [
                                                    'assigned' => 'info',
                                                    'used' => 'success',
                                                    'withdrawn' => 'warning',
                                                    'expired' => 'danger'
                                                ];
                                                $color = $status_colors[$cos['status']] ?? 'secondary';
                                                ?>
                                                <span class="badge bg-<?php echo $color; ?>"><?php echo ucfirst($cos['status'] ?? '-'); ?></span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- RTW Tab -->
                <div class="tab-pane fade" id="rtw">
                    <div class="modern-card">
                        <div class="card-header-modern d-flex justify-content-between align-items-center">
                            <h5 class="card-title-modern mb-0"><i class="bi bi-file-earmark-check"></i> Right to Work Checks</h5>
                            <a href="../rtw/add_check.php?worker_id=<?php echo $worker_id; ?>" class="btn btn-sm btn-primary">
                                <i class="bi bi-plus-circle"></i> Add Check
                            </a>
                        </div>
                        <div class="card-body-modern">
                            <?php if (empty($rtw_checks)): ?>
                                <div class="text-center py-5">
                                    <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                                    <p class="text-muted mt-3">No Right to Work checks recorded</p>
                                    <a href="../rtw/add_check.php?worker_id=<?php echo $worker_id; ?>" class="btn btn-primary">
                                        <i class="bi bi-plus-circle"></i> Add First Check
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Check Date</th>
                                                <th>Type</th>
                                                <th>Documents</th>
                                                <th>Outcome</th>
                                                <th>Next Check Due</th>
                                                <th>Checked By</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($rtw_checks as $check): ?>
                                            <tr>
                                                <td><?php echo date('d/m/Y', strtotime($check['check_date'])); ?></td>
                                                <td><?php echo ucfirst(str_replace('_', ' ', $check['check_type'] ?? '-')); ?></td>
                                                <td><?php echo htmlspecialchars(substr($check['documents_seen'] ?? '-', 0, 50)); ?></td>
                                                <td>
                                                    <?php
                                                    $outcome_colors = ['pass' => 'success', 'fail' => 'danger', 'refer' => 'warning'];
                                                    $color = $outcome_colors[$check['outcome']] ?? 'secondary';
                                                    ?>
                                                    <span class="badge bg-<?php echo $color; ?>"><?php echo ucfirst($check['outcome'] ?? '-'); ?></span>
                                                </td>
                                                <td><?php echo $check['next_check_due'] ? date('d/m/Y', strtotime($check['next_check_due'])) : '-'; ?></td>
                                                <td><?php echo htmlspecialchars($check['checked_by_name'] ?? '-'); ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Documents Tab -->
                <div class="tab-pane fade" id="documents">
                    <div class="modern-card">
                        <div class="card-header-modern d-flex justify-content-between align-items-center">
                            <h5 class="card-title-modern mb-0"><i class="bi bi-files"></i> Worker Documents</h5>
                            <a href="../documents/upload.php?worker_id=<?php echo $worker_id; ?>" class="btn btn-sm btn-primary">
                                <i class="bi bi-upload"></i> Upload Document
                            </a>
                        </div>
                        <div class="card-body-modern">
                            <?php if (empty($documents)): ?>
                                <div class="text-center py-5">
                                    <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                                    <p class="text-muted mt-3">No documents uploaded</p>
                                    <a href="../documents/upload.php?worker_id=<?php echo $worker_id; ?>" class="btn btn-primary">
                                        <i class="bi bi-upload"></i> Upload First Document
                                    </a>
                                </div>
                            <?php else: ?>
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Type</th>
                                                <th>Name</th>
                                                <th>Issue Date</th>
                                                <th>Expiry Date</th>
                                                <th>Status</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($documents as $doc): ?>
                                            <tr>
                                                <td>
                                                    <i class="bi bi-file-earmark-pdf text-danger"></i>
                                                    <?php echo ucfirst(str_replace('_', ' ', $doc['document_type'] ?? 'other')); ?>
                                                </td>
                                                <td><?php echo htmlspecialchars($doc['document_name']); ?></td>
                                                <td><?php echo $doc['issue_date'] ? date('d/m/Y', strtotime($doc['issue_date'])) : '-'; ?></td>
                                                <td>
                                                    <?php if ($doc['expiry_date']): ?>
                                                        <?php
                                                        $expiry_days = (strtotime($doc['expiry_date']) - time()) / 86400;
                                                        echo date('d/m/Y', strtotime($doc['expiry_date']));
                                                        if ($expiry_days < 0): ?>
                                                            <span class="badge bg-danger ms-2">Expired</span>
                                                        <?php elseif ($expiry_days <= 30): ?>
                                                            <span class="badge bg-warning ms-2"><?php echo floor($expiry_days); ?> days</span>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($doc['verified']): ?>
                                                        <span class="badge bg-success">
                                                            <i class="bi bi-check-circle"></i> Verified
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-secondary">Pending</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <a href="<?php echo htmlspecialchars($doc['file_path']); ?>" class="btn btn-sm btn-outline-primary" target="_blank">
                                                        <i class="bi bi-eye"></i>
                                                    </a>
                                                    <a href="<?php echo htmlspecialchars($doc['file_path']); ?>" class="btn btn-sm btn-outline-success" download>
                                                        <i class="bi bi-download"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Notes Tab -->
                <div class="tab-pane fade" id="notes">
                    <div class="modern-card">
                        <div class="card-header-modern d-flex justify-content-between align-items-center">
                            <h5 class="card-title-modern mb-0"><i class="bi bi-journal-text"></i> Worker Notes</h5>
                            <button class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#addNoteModal">
                                <i class="bi bi-plus-circle"></i> Add Note
                            </button>
                        </div>
                        <div class="card-body-modern">
                            <?php if (empty($notes)): ?>
                                <div class="text-center py-5">
                                    <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                                    <p class="text-muted mt-3">No notes recorded</p>
                                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addNoteModal">
                                        <i class="bi bi-plus-circle"></i> Add First Note
                                    </button>
                                </div>
                            <?php else: ?>
                                <?php foreach ($notes as $note): ?>
                                <div class="note-item mb-3 p-3 border rounded">
                                    <div class="d-flex justify-content-between align-items-start mb-2">
                                        <div>
                                            <span class="badge bg-<?php
                                                $type_colors = [
                                                    'general' => 'secondary',
                                                    'compliance' => 'warning',
                                                    'hr' => 'info',
                                                    'disciplinary' => 'danger',
                                                    'performance' => 'success',
                                                    'ho_contact' => 'primary'
                                                ];
                                                echo $type_colors[$note['note_type']] ?? 'secondary';
                                            ?>"><?php echo ucfirst(str_replace('_', ' ', $note['note_type'])); ?></span>
                                            <?php if ($note['is_compliance_related']): ?>
                                                <span class="badge bg-warning ms-1"><i class="bi bi-shield-check"></i> Compliance</span>
                                            <?php endif; ?>
                                            <?php if ($note['is_confidential']): ?>
                                                <span class="badge bg-danger ms-1"><i class="bi bi-lock"></i> Confidential</span>
                                            <?php endif; ?>
                                        </div>
                                        <small class="text-muted"><?php echo date('d/m/Y H:i', strtotime($note['created_at'])); ?></small>
                                    </div>
                                    <p class="mb-1"><?php echo nl2br(htmlspecialchars($note['note'])); ?></p>
                                    <small class="text-muted">By <?php echo htmlspecialchars($note['created_by_name']); ?></small>
                                </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Add Note Modal -->
<div class="modal fade" id="addNoteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Add Note</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="add_note.php">
                <div class="modal-body">
                    <input type="hidden" name="worker_id" value="<?php echo $worker_id; ?>">

                    <div class="mb-3">
                        <label class="form-label">Note Type</label>
                        <select name="note_type" class="form-select" required>
                            <option value="general">General</option>
                            <option value="compliance">Compliance</option>
                            <option value="hr">HR</option>
                            <option value="disciplinary">Disciplinary</option>
                            <option value="performance">Performance</option>
                            <option value="ho_contact">Home Office Contact</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Note</label>
                        <textarea name="note" class="form-control" rows="5" required></textarea>
                    </div>

                    <div class="form-check mb-2">
                        <input type="checkbox" class="form-check-input" name="is_compliance_related" value="1" id="isCompliance">
                        <label class="form-check-label" for="isCompliance">
                            Compliance Related
                        </label>
                    </div>

                    <div class="form-check">
                        <input type="checkbox" class="form-check-input" name="is_confidential" value="1" id="isConfidential">
                        <label class="form-check-label" for="isConfidential">
                            Confidential
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Note</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.info-row {
    display: flex;
    justify-content: space-between;
    padding: 0.75rem 0;
    border-bottom: 1px solid #eee;
}

.info-row:last-child {
    border-bottom: none;
}

.info-label {
    font-weight: 600;
    color: #666;
}

.info-value {
    text-align: right;
    color: #333;
}

.modern-tabs {
    border-bottom: 2px solid #dee2e6;
}

.modern-tabs .nav-link {
    color: #6c757d;
    border: none;
    padding: 1rem 1.5rem;
    transition: all 0.2s;
}

.modern-tabs .nav-link:hover {
    color: #667eea;
    background: rgba(102, 126, 234, 0.05);
}

.modern-tabs .nav-link.active {
    color: #667eea;
    background: transparent;
    border-bottom: 2px solid #667eea;
    margin-bottom: -2px;
}

.note-item {
    background: #f8f9fa;
    transition: all 0.2s;
}

.note-item:hover {
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}
</style>

<?php include '../../includes/footer.php'; ?>
