<?php
/**
 * View Worker Profile - Comprehensive UK Home Office Compliant View
 * Integrates: RTW, CoS, Documents, Contact History, Employment History, Notes
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

$page_title = 'Worker Profile';
$current_page = 'workers';

$worker_id = $_GET['id'] ?? '';

if (empty($worker_id)) {
    header('Location: list.php');
    exit;
}

// Fetch worker details
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$worker) {
    die('Worker not found');
}

// Parse full_name into parts for display
$name_parts = explode(' ', $worker['full_name']);
$display_first = $name_parts[0] ?? '';
$display_last = count($name_parts) > 1 ? end($name_parts) : '';
$display_middle = '';
if (count($name_parts) > 2) {
    $display_middle = implode(' ', array_slice($name_parts, 1, -1));
}

// Fetch RTW Checks
$stmt = $pdo->prepare("SELECT r.*, CONCAT(u.first_name, ' ', u.last_name) as checked_by_name
                       FROM " . DB_PREFIX . "worker_rtw_checks r
                       LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
                       WHERE r.worker_id = ?
                       ORDER BY r.check_date DESC");
$stmt->execute([$worker_id]);
$rtw_checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch CoS History
$stmt = $pdo->prepare("SELECT c.*, CONCAT(u.first_name, ' ', u.last_name) as created_by_name
                       FROM " . DB_PREFIX . "worker_cos_history c
                       LEFT JOIN " . DB_PREFIX . "users u ON c.created_by = u.id
                       WHERE c.worker_id = ?
                       ORDER BY c.created_at DESC");
$stmt->execute([$worker_id]);
$cos_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Documents
$stmt = $pdo->prepare("SELECT d.*, CONCAT(u.first_name, ' ', u.last_name) as uploaded_by_name, CONCAT(v.first_name, ' ', v.last_name) as verified_by_name
                       FROM " . DB_PREFIX . "worker_documents d
                       LEFT JOIN " . DB_PREFIX . "users u ON d.uploaded_by = u.id
                       LEFT JOIN " . DB_PREFIX . "users v ON d.verified_by = v.id
                       WHERE d.worker_id = ? AND d.deleted_at IS NULL
                       ORDER BY d.uploaded_at DESC");
$stmt->execute([$worker_id]);
$documents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Contact History
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_contact_history
                       WHERE worker_id = ?
                       ORDER BY valid_from DESC");
$stmt->execute([$worker_id]);
$contact_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Employment History
$stmt = $pdo->prepare("SELECT e.*, w.name as location_name, CONCAT(u.first_name, ' ', u.last_name) as created_by_name
                       FROM " . DB_PREFIX . "worker_employment_history e
                       LEFT JOIN " . DB_PREFIX . "work_locations w ON e.work_location_id = w.id
                       LEFT JOIN " . DB_PREFIX . "users u ON e.created_by = u.id
                       WHERE e.worker_id = ?
                       ORDER BY e.start_date DESC");
$stmt->execute([$worker_id]);
$employment_history = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Notes
$stmt = $pdo->prepare("SELECT n.*, CONCAT(u.first_name, ' ', u.last_name) as created_by_name
                       FROM " . DB_PREFIX . "worker_notes n
                       LEFT JOIN " . DB_PREFIX . "users u ON n.created_by = u.id
                       WHERE n.worker_id = ?
                       ORDER BY n.created_at DESC");
$stmt->execute([$worker_id]);
$notes = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calculate visa expiry status
$days_to_visa_expiry = null;
$visa_status = '';
if (!empty($worker['visa_expiry'])) {
    $days_to_visa_expiry = (strtotime($worker['visa_expiry']) - time()) / 86400;
    if ($days_to_visa_expiry < 0) {
        $visa_status = 'expired';
    } elseif ($days_to_visa_expiry <= 30) {
        $visa_status = 'urgent';
    } elseif ($days_to_visa_expiry <= 90) {
        $visa_status = 'warning';
    } else {
        $visa_status = 'valid';
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <!-- Header Section -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-person-circle"></i> <?php echo htmlspecialchars($worker['full_name']); ?>
            </h1>
            <p class="welcome-subtitle">
                Employee #<?php echo htmlspecialchars($worker['employee_number']); ?> |
                <?php echo htmlspecialchars($worker['job_title'] ?? 'No Title'); ?>
                <?php if ($worker['status'] === 'active'): ?>
                    <span class="badge bg-success ms-2">Active</span>
                <?php else: ?>
                    <span class="badge bg-secondary ms-2"><?php echo ucfirst($worker['status'] ?? ''); ?></span>
                <?php endif; ?>
            </p>
        </div>
        <div class="welcome-actions">
            <a href="edit.php?id=<?php echo $worker_id; ?>" class="btn btn-primary">
                <i class="bi bi-pencil"></i> Edit Profile
            </a>
            <a href="list.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
    </div>

    <!-- Compliance Alerts -->
    <?php if ($visa_status === 'expired' || $visa_status === 'urgent'): ?>
    <div class="alert alert-danger modern-alert">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>Visa Compliance Alert:</strong>
        <?php if ($visa_status === 'expired'): ?>
            Visa expired <?php echo abs(floor($days_to_visa_expiry)); ?> days ago. Immediate action required.
        <?php else: ?>
            Visa expires in <?php echo floor($days_to_visa_expiry); ?> days. Urgent attention needed.
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <div class="row g-4">
        <!-- Left Column - Core Information -->
        <div class="col-lg-4">
            <!-- Photo & Basic Info Card -->
            <div class="modern-card mb-4">
                <div class="card-body-modern text-center">
                    <?php if (!empty($worker['photo'])): ?>
                        <img src="../../uploads/workers/<?php echo htmlspecialchars($worker['photo']); ?>"
                             alt="Photo" class="rounded-circle mb-3" style="width: 150px; height: 150px; object-fit: cover;">
                    <?php else: ?>
                        <div class="rounded-circle bg-light d-inline-flex align-items-center justify-content-center mb-3"
                             style="width: 150px; height: 150px;">
                            <i class="bi bi-person-fill" style="font-size: 4rem; color: #ccc;"></i>
                        </div>
                    <?php endif; ?>

                    <h4><?php echo htmlspecialchars($worker['full_name']); ?></h4>
                    <p class="text-muted mb-3"><?php echo htmlspecialchars($worker['job_title'] ?? 'No Title'); ?></p>

                    <div class="d-grid gap-2">
                        <a href="mailto:<?php echo htmlspecialchars($worker['email'] ?? ''); ?>" class="btn btn-outline-primary btn-sm">
                            <i class="bi bi-envelope"></i> Email
                        </a>
                        <?php if (!empty($worker['phone'])): ?>
                        <a href="tel:<?php echo htmlspecialchars($worker['phone']); ?>" class="btn btn-outline-success btn-sm">
                            <i class="bi bi-telephone"></i> Call
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Personal Details Card -->
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-person-badge"></i> Personal Details</h5>
                </div>
                <div class="card-body-modern">
                    <div class="info-row">
                        <span class="info-label">First Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($display_first); ?></span>
                    </div>
                    <?php if (!empty($display_middle)): ?>
                    <div class="info-row">
                        <span class="info-label">Middle Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($display_middle); ?></span>
                    </div>
                    <?php endif; ?>
                    <div class="info-row">
                        <span class="info-label">Last Name:</span>
                        <span class="info-value"><?php echo htmlspecialchars($display_last); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Date of Birth:</span>
                        <span class="info-value"><?php echo $worker['date_of_birth'] ? date('d/m/Y', strtotime($worker['date_of_birth'])) : '-'; ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Nationality:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['nationality'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">NI Number:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['ni_number'] ?? '-'); ?></span>
                    </div>
                </div>
            </div>

            <!-- Contact Details Card -->
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-envelope"></i> Contact Details</h5>
                </div>
                <div class="card-body-modern">
                    <div class="info-row">
                        <span class="info-label">Email:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['email'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Phone:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['phone'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Address:</span>
                        <span class="info-value"><?php echo nl2br(htmlspecialchars($worker['address'] ?? '-')); ?></span>
                    </div>
                    <hr>
                    <div class="info-row">
                        <span class="info-label">Emergency Contact:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['emergency_contact'] ?? '-'); ?></span>
                    </div>
                    <div class="info-row">
                        <span class="info-label">Emergency Phone:</span>
                        <span class="info-value"><?php echo htmlspecialchars($worker['emergency_phone'] ?? '-'); ?></span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Right Column - Detailed Information Tabs -->
        <div class="col-lg-8">
            <!-- Navigation Tabs -->
            <ul class="nav nav-tabs modern-tabs mb-4" role="tablist">
                <li class="nav-item">
                    <a class="nav-link active" data-bs-toggle="tab" href="#employment">
                        <i class="bi bi-briefcase"></i> Employment
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#sponsorship">
                        <i class="bi bi-shield-check"></i> Sponsorship
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#rtw">
                        <i class="bi bi-file-earmark-check"></i> Right to Work
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#documents">
                        <i class="bi bi-files"></i> Documents (<?php echo count($documents); ?>)
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#history">
                        <i class="bi bi-clock-history"></i> History
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#notes">
                        <i class="bi bi-journal-text"></i> Notes (<?php echo count($notes); ?>)
                    </a>
                </li>
            </ul>

            <!-- Tab Content -->
            <div class="tab-content">
                <!-- Employment Tab -->
                <div class="tab-pane fade show active" id="employment">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <h5 class="card-title-modern"><i class="bi bi-briefcase"></i> Current Employment</h5>
                        </div>
                        <div class="card-body-modern">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Job Title:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['job_title'] ?? '-'); ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Department:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['department'] ?? '-'); ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Contract Type:</span>
                                        <span class="info-value"><?php echo htmlspecialchars($worker['contract_type'] ?? '-'); ?></span>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-row">
                                        <span class="info-label">Start Date:</span>
                                        <span class="info-value"><?php echo $worker['start_date'] ? date('d/m/Y', strtotime($worker['start_date'])) : '-'; ?></span>
                                    </div>
                                    <div class="info-row">
                                        <span class="info-label">Status:</span>
                                        <span class="info-value">
                                            <?php if ($worker['status'] === 'active'): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary"><?php echo ucfirst($worker['status'] ?? ''); ?></span>
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <?php if (!empty($employment_history)): ?>
                            <hr>
                            <h6 class="mb-3"><i class="bi bi-clock-history"></i> Employment History</h6>
                            <div class="table-responsive">
                                <table class="table table-sm table-hover">
                                    <thead>
                                        <tr>
                                            <th>Period</th>
                                            <th>Job Title</th>
                                            <th>Department</th>
                                            <th>Location</th>
                                            <th>Change Reason</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($employment_history as $emp): ?>
                                        <tr>
                                            <td>
                                                <?php echo date('d/m/Y', strtotime($emp['start_date'])); ?>
                                                <?php if ($emp['end_date']): ?>
                                                    - <?php echo date('d/m/Y', strtotime($emp['end_date'])); ?>
                                                <?php else: ?>
                                                    <span class="badge bg-success">Current</span>
                                                <?php endif; ?>
                                            </td>
                                            <td><?php echo htmlspecialchars($emp['job_title']); ?></td>
                                            <td><?php echo htmlspecialchars($emp['department']); ?></td>
                                            <td><?php echo htmlspecialchars($emp['location_name'] ?? '-'); ?></td>
                                            <td><?php echo ucfirst(str_replace('_', ' ', $emp['change_reason'])); ?></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            
            <div class="row mb-4">
                <div class="col-md-6">
                    <table class="table table-sm">
                        <tr>
                            <td width="40%"><strong>Employee Number:</strong></td>
                            <td><?= htmlspecialchars($worker['employee_number']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Date of Birth:</strong></td>
                            <td><?= format_date($worker['date_of_birth']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Gender:</strong></td>
                            <td><?= htmlspecialchars($worker['gender'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Nationality:</strong></td>
                            <td><?= htmlspecialchars($worker['nationality'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Email:</strong></td>
                            <td><?= htmlspecialchars($worker['email'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Phone:</strong></td>
                            <td><?= htmlspecialchars($worker['phone'] ?? '-') ?></td>
                        </tr>
                    </table>
                </div>
                
                <div class="col-md-6">
                    <table class="table table-sm">
                        <tr>
                            <td width="40%"><strong>Job Title:</strong></td>
                            <td><?= htmlspecialchars($worker['job_title'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Department:</strong></td>
                            <td><?= htmlspecialchars($worker['department'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Employment Status:</strong></td>
                            <td>
                                <span class="badge <?= $worker['employment_status'] === 'Active' ? 'bg-success' : 'bg-secondary' ?>">
                                    <?= htmlspecialchars($worker['employment_status']) ?>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <td><strong>Contract Type:</strong></td>
                            <td><?= htmlspecialchars($worker['contract_type'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Start Date:</strong></td>
                            <td><?= format_date($worker['employment_start_date']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Salary:</strong></td>
                            <td><?= $worker['salary'] ? '£' . number_format($worker['salary'], 2) : '-' ?></td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <hr>
            
            <h6 class="text-primary mb-3">Immigration Information</h6>
            <div class="row">
                <div class="col-md-6">
                    <table class="table table-sm">
                        <tr>
                            <td width="40%"><strong>Visa Type:</strong></td>
                            <td><?= htmlspecialchars($worker['visa_type'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Visa Number:</strong></td>
                            <td><?= htmlspecialchars($worker['visa_number'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Visa Issue Date:</strong></td>
                            <td><?= format_date($worker['visa_issue_date']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>Visa Expiry Date:</strong></td>
                            <td>
                                <?= format_date($worker['visa_expiry_date']) ?>
                                <?php if ($worker['days_to_expiry'] !== null): ?>
                                    <?php if ($worker['days_to_expiry'] < 0): ?>
                                        <span class="badge bg-danger">Expired</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 30): ?>
                                        <span class="badge bg-danger"><?= $worker['days_to_expiry'] ?> days left</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 90): ?>
                                        <span class="badge bg-warning"><?= $worker['days_to_expiry'] ?> days left</span>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="col-md-6">
                    <table class="table table-sm">
                        <tr>
                            <td width="40%"><strong>Passport Number:</strong></td>
                            <td><?= htmlspecialchars($worker['passport_number'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>Passport Expiry:</strong></td>
                            <td><?= format_date($worker['passport_expiry_date']) ?></td>
                        </tr>
                        <tr>
                            <td><strong>CoS Number:</strong></td>
                            <td><?= htmlspecialchars($worker['cos_number'] ?? '-') ?></td>
                        </tr>
                        <tr>
                            <td><strong>CoS Issue Date:</strong></td>
                            <td><?= format_date($worker['cos_issue_date']) ?></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Recent Attendance -->
        <div class="stat-card mt-3">
            <h6>Recent Attendance</h6>
            <div class="table-responsive">
                <table class="table table-sm table-hover">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Check In</th>
                            <th>Check Out</th>
                            <th>Hours</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($recent_attendance)): ?>
                        <tr>
                            <td colspan="5" class="text-center text-muted">No attendance records</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($recent_attendance as $att): ?>
                        <tr>
                            <td><?= format_date($att['attendance_date']) ?></td>
                            <td><?= $att['check_in_time'] ?? '-' ?></td>
                            <td><?= $att['check_out_time'] ?? '-' ?></td>
                            <td><?= $att['hours_worked'] ?? '-' ?></td>
                            <td>
                                <span class="badge <?= $att['status'] === 'Present' ? 'bg-success' : 'bg-secondary' ?>">
                                    <?= htmlspecialchars($att['status']) ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <!-- Sidebar -->
    <div class="col-md-4">
        <!-- Quick Actions -->
        <div class="stat-card">
            <h6>Quick Actions</h6>
            <div class="d-grid gap-2">
                <a href="../attendance/mark.php?worker_id=<?= $worker['id'] ?>" class="btn btn-success btn-sm">
                    <i class="bi bi-calendar-check"></i> Mark Attendance
                </a>
                <a href="../documents/upload.php?worker_id=<?= $worker['id'] ?>" class="btn btn-primary btn-sm">
                    <i class="bi bi-file-earmark-arrow-up"></i> Upload Document
                </a>
                <a href="../holidays/request.php?worker_id=<?= $worker['id'] ?>" class="btn btn-warning btn-sm">
                    <i class="bi bi-umbrella"></i> Request Holiday
                </a>
            </div>
        </div>
        
        <!-- Documents -->
        <div class="stat-card mt-3">
            <h6>Documents (<?= count($documents) ?>)</h6>
            <?php if (empty($documents)): ?>
                <p class="text-muted small">No documents uploaded</p>
            <?php else: ?>
                <div class="list-group list-group-flush">
                    <?php foreach (array_slice($documents, 0, 5) as $doc): ?>
                    <a href="../documents/view.php?id=<?= $doc['id'] ?>" class="list-group-item list-group-item-action py-2">
                        <small>
                            <i class="bi bi-file-earmark"></i> 
                            <?= htmlspecialchars($doc['document_name']) ?>
                            <br>
                            <span class="text-muted"><?= $doc['document_category'] ?></span>
                        </small>
                    </a>
                    <?php endforeach; ?>
                </div>
                <?php if (count($documents) > 5): ?>
                <a href="../documents/list.php?worker_id=<?= $worker['id'] ?>" class="btn btn-sm btn-outline-primary w-100 mt-2">
                    View All Documents
                </a>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <!-- Holiday Requests -->
        <div class="stat-card mt-3">
            <h6>Recent Holidays</h6>
            <?php if (empty($holidays)): ?>
                <p class="text-muted small">No holiday requests</p>
            <?php else: ?>
                <?php foreach ($holidays as $holiday): ?>
                <div class="mb-2 p-2 bg-light rounded">
                    <small>
                        <strong><?= format_date($holiday['start_date']) ?> - <?= format_date($holiday['end_date']) ?></strong><br>
                        <?= $holiday['total_days'] ?> days | 
                        <span class="badge <?= $holiday['approval_status'] === 'Approved' ? 'bg-success' : ($holiday['approval_status'] === 'Pending' ? 'bg-warning' : 'bg-danger') ?>">
                            <?= $holiday['approval_status'] ?>
                        </span>
                    </small>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- Record Info -->
        <div class="stat-card mt-3">
            <h6>Record Information</h6>
            <small>
                <strong>Created:</strong> <?= format_datetime($worker['created_at']) ?><br>
                <strong>By:</strong> <?= htmlspecialchars($worker['created_by_name'] ?? 'System') ?><br>
                <strong>Last Updated:</strong> <?= format_datetime($worker['updated_at']) ?>
            </small>
        </div>
    </div>
</div>

<?php require_once '../../includes/footer.php'; ?>