<?php
/**
 * View Worker Profile - Comprehensive UK Home Office Compliant View
 * Worker Module Tabs: Overview, Qualifications, Addresses, Emergency, Notes
 * Module-Dependent Tabs: Documents, RTW, CoS, Attendance, Holidays (if modules active)
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_once '../../includes/helpers.php';
    require_once '../../includes/module_helpers.php';
    require_authentication();
}

// Get database connection
$pdo = get_database_connection();

$page_title = 'Worker Profile';
$current_page = 'workers';

$worker_id = $_GET['id'] ?? '';

if (empty($worker_id)) {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php'));
    exit;
}

// Fetch worker details
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$worker) {
    die('Worker not found');
}

// Calculate visa status for alerts
$visa_status = 'unknown';
$days_to_visa_expiry = null;

if (!empty($worker['visa_expiry'])) {
    $expiry_date = new DateTime($worker['visa_expiry']);
    $today = new DateTime();
    $interval = $today->diff($expiry_date);
    $days_to_visa_expiry = (int)$interval->format('%r%a');

    if ($days_to_visa_expiry < 0) {
        $visa_status = 'expired';
    } elseif ($days_to_visa_expiry <= 30) {
        $visa_status = 'urgent';
    } elseif ($days_to_visa_expiry <= 90) {
        $visa_status = 'warning';
    } else {
        $visa_status = 'valid';
    }
}

// Fetch qualifications
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_qualifications WHERE worker_id = ? ORDER BY year_obtained DESC");
$stmt->execute([$worker_id]);
$qualifications = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch address history
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_addresses WHERE worker_id = ? ORDER BY move_in_date DESC");
$stmt->execute([$worker_id]);
$addresses = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch emergency contacts
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_emergency_contacts WHERE worker_id = ? ORDER BY is_primary DESC, id ASC");
$stmt->execute([$worker_id]);
$emergency_contacts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch notes/history
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_notes WHERE worker_id = ? ORDER BY created_at DESC");
$stmt->execute([$worker_id]);
$notes = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Check which modules are active
$documents_active = is_module_active('documents');
$rtw_active = is_module_active('rtw');
$cos_active = is_module_active('cos');
$attendance_active = is_module_active('attendance');
$holidays_active = is_module_active('holidays');
$locations_active = is_module_active('locations');

// Fetch module data if active (with error handling for missing tables)
$documents = [];
$rtw_checks = [];
$cos_records = [];
$recent_attendance = [];
$worker_holidays = [];

if ($documents_active) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "documents WHERE worker_id = ? ORDER BY uploaded_at DESC");
        $stmt->execute([$worker_id]);
        $documents = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $documents_active = false;
    }
}

if ($rtw_active) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "rtw_checks WHERE worker_id = ? ORDER BY check_date DESC");
        $stmt->execute([$worker_id]);
        $rtw_checks = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $rtw_active = false;
    }
}

if ($cos_active) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "cos WHERE worker_id = ? ORDER BY issue_date DESC");
        $stmt->execute([$worker_id]);
        $cos_records = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $cos_active = false;
    }
}

if ($attendance_active) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "attendance WHERE worker_id = ? ORDER BY attendance_date DESC LIMIT 30");
        $stmt->execute([$worker_id]);
        $recent_attendance = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $attendance_active = false;
    }
}

if ($holidays_active) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_holidays WHERE worker_id = ? ORDER BY start_date DESC");
        $stmt->execute([$worker_id]);
        $worker_holidays = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $holidays_active = false;
    }
}

// Fetch location assignments if locations module is active
$worker_location_assignments = [];
$worker_location = null; // Keep for backward compatibility
if ($locations_active) {
    try {
        // Fetch all location assignments for this worker with shift details
        $stmt = $pdo->prepare("SELECT
            wla.*,
            l.name as location_name,
            l.type as location_type,
            l.address as location_address,
            l.city as location_city,
            l.postcode as location_postcode,
            l.capacity as location_capacity,
            l.status as location_status,
            sp.shift_name,
            sp.shift_type,
            sp.start_time,
            sp.end_time,
            sp.break_duration,
            sp.days_of_week as shift_days_of_week
            FROM " . DB_PREFIX . "worker_location_assignments wla
            LEFT JOIN " . DB_PREFIX . "work_locations l ON wla.location_id = l.id
            LEFT JOIN " . DB_PREFIX . "location_shift_patterns sp ON wla.shift_pattern_id = sp.id
            WHERE wla.worker_id = ?
                AND (wla.end_date IS NULL OR wla.end_date >= CURDATE())
            ORDER BY wla.is_primary DESC, wla.id ASC");
        $stmt->execute([$worker_id]);
        $worker_location_assignments = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Get primary location for backward compatibility
        if (!empty($worker_location_assignments)) {
            $primary = $worker_location_assignments[0];
            $worker_location = [
                'id' => $primary['location_id'],
                'name' => $primary['location_name'],
                'type' => $primary['location_type'],
                'address' => $primary['location_address'],
                'city' => $primary['location_city'],
                'postcode' => $primary['location_postcode'],
                'capacity' => $primary['location_capacity'],
                'status' => $primary['location_status']
            ];
        } elseif (!empty($worker['location_id'])) {
            // Fallback to old single location if no assignments exist
            $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "work_locations WHERE id = ?");
            $stmt->execute([$worker['location_id']]);
            $worker_location = $stmt->fetch(PDO::FETCH_ASSOC);
        }
    } catch (PDOException $e) {
        // Table doesn't exist yet - module not fully installed
        $locations_active = false;
    }
}

// Check for missing required data (for red tab indicators)
$missing_qualifications = empty($qualifications);

// Check if addresses exist in EITHER the separate table OR the main worker table fields
// (the edit form allows saving to both locations)
$has_address_table_data = !empty($addresses);
$has_worker_table_address = false; // We don't have address fields in main worker table currently
$missing_addresses = !$has_address_table_data && !$has_worker_table_address;

// Check if emergency contacts exist in EITHER the separate table OR the main worker table fields
$has_emergency_table_data = !empty($emergency_contacts);
$has_worker_table_emergency = !empty($worker['emergency_contact']) || !empty($worker['emergency_phone']);
$missing_emergency = !$has_emergency_table_data && !$has_worker_table_emergency;

$missing_documents = $documents_active && empty($documents);
$missing_rtw = $rtw_active && empty($rtw_checks);

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<style>
/* Tab Styling */
.nav-tabs {
    border-bottom: 2px solid #dee2e6;
    margin-bottom: 1.5rem;
}

.nav-tabs .nav-link {
    border: none;
    color: #6c757d;
    padding: 0.75rem 1rem;
    margin-right: 0.25rem;
    font-weight: 500;
}

.nav-tabs .nav-link:hover {
    color: #0d6efd;
}

.nav-tabs .nav-link.active {
    color: #0d6efd;
    background: transparent;
    border-bottom: 2px solid #0d6efd;
    font-weight: 600;
}

.tab-danger {
    color: #dc3545 !important;
}

.nav-tabs .nav-link.active.tab-danger {
    border-bottom-color: #dc3545;
}

/* Card Styling */
.modern-card {
    background: #fff;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
    margin-bottom: 1.5rem;
}

.card-header-modern {
    background: #f8f9fa;
    border-bottom: 1px solid #dee2e6;
    padding: 1rem 1.25rem;
    font-weight: 600;
    font-size: 1rem;
}

.card-header-modern i {
    margin-right: 0.5rem;
    color: #0d6efd;
}

.card-body-modern {
    padding: 1.25rem;
}

/* Info Table */
.info-table {
    margin-bottom: 0;
    width: 100%;
}

.info-table td {
    padding: 0.75rem;
    border-bottom: 1px solid #dee2e6;
}

.info-table tr:last-child td {
    border-bottom: none;
}

.info-table td:first-child {
    font-weight: 600;
    width: 180px;
    color: #495057;
}

.info-table td:last-child {
    color: #212529;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 3rem 1rem;
    color: #6c757d;
}

.empty-state i {
    font-size: 3rem;
    margin-bottom: 1rem;
    opacity: 0.5;
}

/* Table Headers */
.table thead {
    background: #f8f9fa;
}

.table thead th {
    border-bottom: 2px solid #dee2e6;
    font-weight: 600;
    color: #495057;
}

.table-hover tbody tr:hover {
    background: #f8f9fa;
}

/* Cards in Tabs */
.card {
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
}
</style>

<div class="dashboard-container">
    <!-- Header Section -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-person-circle"></i> <?php echo htmlspecialchars($worker['full_name']); ?>
            </h1>
            <p class="welcome-subtitle">
                Employee #<?php echo htmlspecialchars($worker['employee_number']); ?> |
                <?php echo htmlspecialchars($worker['job_title'] ?? 'No Title'); ?>
                <?php
                $status_badge_class = 'bg-secondary';
                $status_label = ucfirst(str_replace('_', ' ', $worker['status'] ?? ''));

                if ($worker['status'] === 'active') {
                    $status_badge_class = 'bg-success';
                } elseif ($worker['status'] === 'pre_employment') {
                    $status_badge_class = 'bg-info';
                    $status_label = 'Pre-Employment';
                } elseif ($worker['status'] === 'pending_rtw') {
                    $status_badge_class = 'bg-warning text-dark';
                    $status_label = 'Pending RTW Check';
                } elseif ($worker['status'] === 'terminated') {
                    $status_badge_class = 'bg-danger';
                } elseif ($worker['status'] === 'inactive') {
                    $status_badge_class = 'bg-secondary';
                }
                ?>
                <span class="badge <?= $status_badge_class ?> ms-2"><?= $status_label ?></span>
            </p>
        </div>
        <div class="welcome-actions">
            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker_id : 'edit.php?id=' . $worker_id ?>" class="btn btn-primary">
                <i class="bi bi-pencil"></i> Edit Profile
            </a>
            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php' ?>" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
    </div>

    <!-- Compliance Alerts -->
    <?php if ($visa_status === 'expired' || $visa_status === 'urgent'): ?>
    <div class="alert alert-danger">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>Visa Compliance Alert:</strong>
        <?php if ($visa_status === 'expired'): ?>
            Visa expired <?php echo abs(floor($days_to_visa_expiry)); ?> days ago. Immediate action required.
        <?php else: ?>
            Visa expires in <?php echo floor($days_to_visa_expiry); ?> days. Urgent attention needed.
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <!-- RTW Status Alerts -->
    <?php if ($worker['status'] === 'pending_rtw'): ?>
    <div class="alert alert-warning">
        <i class="bi bi-exclamation-triangle-fill"></i>
        <strong>Right to Work Check Required:</strong> This worker requires a Right to Work check before they can start employment.
        <?php if ($rtw_active): ?>
            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/rtw/add_check?worker_id=' . $worker_id : '../rtw/add_check.php?worker_id=' . $worker_id ?>" class="btn btn-sm btn-warning ms-2">
                <i class="bi bi-plus-circle"></i> Perform RTW Check
            </a>
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <?php if ($worker['status'] === 'pre_employment'): ?>
    <div class="alert alert-info">
        <i class="bi bi-info-circle-fill"></i>
        <strong>Pre-Employment Status:</strong> This worker is awaiting visa approval. They cannot start employment until visa is granted and RTW check is completed.
    </div>
    <?php endif; ?>

    <!-- Activation Button for Pending Workers -->
    <?php if ($worker['status'] === 'pre_employment' || $worker['status'] === 'pending_rtw'): ?>
    <div class="card border-primary mb-4">
        <div class="card-body">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h5 class="card-title mb-2">
                        <i class="bi bi-person-check-fill text-primary"></i>
                        Worker Activation Required
                    </h5>
                    <p class="card-text mb-0">
                        This worker cannot start employment until activated. Activation will confirm all details and set the employment start date.
                    </p>
                </div>
                <div class="col-md-4 text-end">
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/activate_worker?id=' . $worker_id : 'activate_worker.php?id=' . $worker_id ?>"
                       class="btn btn-lg btn-success"
                       target="_blank">
                        <i class="bi bi-check-circle-fill"></i> Activate Worker
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Activation History (if worker was previously activated) -->
    <?php
    try {
        $activation_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_activation WHERE worker_id = ? ORDER BY activation_date DESC LIMIT 1");
        $activation_stmt->execute([$worker_id]);
        $activation_record = $activation_stmt->fetch();
    } catch (PDOException $e) {
        $activation_record = null;
    }

    if ($activation_record && $worker['status'] === 'active'):
    ?>
    <div class="alert alert-success">
        <div class="row align-items-center">
            <div class="col-md-8">
                <i class="bi bi-check-circle-fill"></i>
                <strong>Worker Activated:</strong>
                Activated on <?= date('d/m/Y H:i', strtotime($activation_record['activation_date'])) ?>
                <?php if ($activation_record['previous_status']): ?>
                    <br><small>Previous status: <?= ucfirst(str_replace('_', ' ', $activation_record['previous_status'])) ?></small>
                <?php endif; ?>
            </div>
            <div class="col-md-4 text-end">
                <small class="text-muted">
                    Employment started: <?= date('d/m/Y', strtotime($activation_record['start_date'])) ?>
                </small>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Navigation Tabs -->
    <ul class="nav nav-tabs mb-4" id="workerTabs" role="tablist">
        <!-- Worker Module Tabs (always visible) -->
        <li class="nav-item" role="presentation">
            <button class="nav-link active" id="overview-tab" data-bs-toggle="tab" data-bs-target="#overview" type="button">
                <i class="bi bi-person"></i> Overview
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $missing_qualifications ? 'tab-danger' : '' ?>" id="qualifications-tab" data-bs-toggle="tab" data-bs-target="#qualifications" type="button">
                <i class="bi bi-mortarboard"></i> Qualifications
                <?= $missing_qualifications ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $missing_addresses ? 'tab-danger' : '' ?>" id="addresses-tab" data-bs-toggle="tab" data-bs-target="#addresses" type="button">
                <i class="bi bi-house"></i> Addresses
                <?= $missing_addresses ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $missing_emergency ? 'tab-danger' : '' ?>" id="emergency-tab" data-bs-toggle="tab" data-bs-target="#emergency" type="button">
                <i class="bi bi-telephone-plus"></i> Emergency Contacts
                <?= $missing_emergency ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="notes-tab" data-bs-toggle="tab" data-bs-target="#notes" type="button">
                <i class="bi bi-journal-text"></i> Notes
            </button>
        </li>

        <!-- Module-Dependent Tabs -->
        <?php if ($locations_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= (!$worker_location) ? 'tab-danger' : '' ?>" id="location-tab" data-bs-toggle="tab" data-bs-target="#location" type="button">
                <i class="bi bi-geo-alt"></i> Location
                <?= (!$worker_location) ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <?php endif; ?>

        <?php if ($documents_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $missing_documents ? 'tab-danger' : '' ?>" id="documents-tab" data-bs-toggle="tab" data-bs-target="#documents" type="button">
                <i class="bi bi-files"></i> Documents
                <?= $missing_documents ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <?php endif; ?>

        <?php if ($rtw_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link <?= $missing_rtw ? 'tab-danger' : '' ?>" id="rtw-tab" data-bs-toggle="tab" data-bs-target="#rtw" type="button">
                <i class="bi bi-file-earmark-check"></i> RTW Checks
                <?= $missing_rtw ? '<i class="bi bi-exclamation-circle ms-1"></i>' : '' ?>
            </button>
        </li>
        <?php endif; ?>

        <?php if ($cos_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="cos-tab" data-bs-toggle="tab" data-bs-target="#cos" type="button">
                <i class="bi bi-file-earmark-medical"></i> CoS
            </button>
        </li>
        <?php endif; ?>

        <?php if ($attendance_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="attendance-tab" data-bs-toggle="tab" data-bs-target="#attendance" type="button">
                <i class="bi bi-calendar-check"></i> Attendance
            </button>
        </li>
        <?php endif; ?>

        <?php if ($holidays_active): ?>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="holidays-tab" data-bs-toggle="tab" data-bs-target="#holidays" type="button">
                <i class="bi bi-umbrella"></i> Holidays
            </button>
        </li>
        <?php endif; ?>
    </ul>

    <!-- Tab Content -->
    <div class="tab-content" id="workerTabsContent">

        <!-- OVERVIEW TAB -->
        <div class="tab-pane fade show active" id="overview" role="tabpanel">
            <div class="row g-4">
                <!-- Personal Information -->
                <div class="col-md-6">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-person"></i> Personal Information
                        </div>
                        <div class="card-body-modern">
                            <table class="info-table w-100">
                                <tr>
                                    <td>Full Name</td>
                                    <td><?= htmlspecialchars($worker['full_name']) ?></td>
                                </tr>
                                <tr>
                                    <td>Date of Birth</td>
                                    <td><?= $worker['date_of_birth'] ? format_date($worker['date_of_birth']) : '-' ?></td>
                                </tr>
                                <tr>
                                    <td>Gender</td>
                                    <td><?= htmlspecialchars($worker['gender'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Place of Birth</td>
                                    <td><?= htmlspecialchars($worker['place_of_birth'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Nationality</td>
                                    <td><?= htmlspecialchars($worker['nationality']) ?></td>
                                </tr>
                                <tr>
                                    <td>NI Number</td>
                                    <td><?= htmlspecialchars($worker['ni_number'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Driving License</td>
                                    <td><?= htmlspecialchars($worker['driving_license'] ?? '-') ?></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Employment Details -->
                <div class="col-md-6">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-briefcase"></i> Employment Details
                        </div>
                        <div class="card-body-modern">
                            <table class="info-table w-100">
                                <tr>
                                    <td>Job Title</td>
                                    <td><?= htmlspecialchars($worker['job_title'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Department</td>
                                    <td><?= htmlspecialchars($worker['department'] ?? '-') ?></td>
                                </tr>
                                <?php if ($locations_active): ?>
                                <tr>
                                    <td>Work Location<?= count($worker_location_assignments) > 1 ? 's' : '' ?></td>
                                    <td>
                                        <?php if (!empty($worker_location_assignments)): ?>
                                            <?php foreach ($worker_location_assignments as $index => $assignment): ?>
                                                <div class="mb-2 <?= $index > 0 ? 'border-top pt-2' : '' ?>">
                                                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/locations/view?id=' . $assignment['location_id'] : '../locations/view.php?id=' . $assignment['location_id'] ?>" class="fw-bold text-decoration-none">
                                                        <i class="bi bi-geo-alt-fill text-primary"></i>
                                                        <?= htmlspecialchars($assignment['location_name']) ?>
                                                    </a>
                                                    <?php if ($assignment['is_primary']): ?>
                                                        <span class="badge bg-success ms-1">Primary</span>
                                                    <?php endif; ?>
                                                    <?php if ($assignment['location_type']): ?>
                                                        <span class="badge bg-secondary ms-1"><?= ucfirst($assignment['location_type']) ?></span>
                                                    <?php endif; ?>
                                                    <br>
                                                    <small class="text-muted">
                                                        <?php if ($assignment['shift_name']): ?>
                                                            <i class="bi bi-clock"></i> <?= htmlspecialchars($assignment['shift_name']) ?>
                                                            <?php if ($assignment['start_time'] && $assignment['end_time']): ?>
                                                                (<?= date('H:i', strtotime($assignment['start_time'])) ?>-<?= date('H:i', strtotime($assignment['end_time'])) ?>)
                                                            <?php endif; ?>
                                                        <?php endif; ?>
                                                        <?php if ($assignment['days_of_week']): ?>
                                                            <br><i class="bi bi-calendar-week"></i> <?= htmlspecialchars($assignment['days_of_week']) ?>
                                                        <?php endif; ?>
                                                    </small>
                                                </div>
                                            <?php endforeach; ?>
                                        <?php else: ?>
                                            <span class="text-danger">
                                                <i class="bi bi-exclamation-circle"></i> Not assigned
                                            </span>
                                            <br>
                                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker_id : 'edit.php?id=' . $worker_id ?>" class="btn btn-sm btn-primary mt-2">
                                                <i class="bi bi-geo-alt"></i> Assign Location
                                            </a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endif; ?>
                                <tr>
                                    <td>Start Date</td>
                                    <td><?= $worker['start_date'] ? format_date($worker['start_date']) : '-' ?></td>
                                </tr>
                                <tr>
                                    <td>Contract Type</td>
                                    <td><?= htmlspecialchars($worker['contract_type'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Probation End Date</td>
                                    <td><?= $worker['probation_end_date'] ? format_date($worker['probation_end_date']) : '-' ?></td>
                                </tr>
                                <tr>
                                    <td>Line Manager</td>
                                    <td>
                                        <?php
                                        if ($worker['manager_id']) {
                                            $mgr_stmt = $pdo->prepare("SELECT full_name FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
                                            $mgr_stmt->execute([$worker['manager_id']]);
                                            $manager = $mgr_stmt->fetch(PDO::FETCH_ASSOC);
                                            echo htmlspecialchars($manager['full_name'] ?? 'Unknown');
                                        } else {
                                            echo '-';
                                        }
                                        ?>
                                    </td>
                                </tr>
                                <tr>
                                    <td>Annual Salary</td>
                                    <td><?= $worker['annual_salary'] ? '£' . number_format($worker['annual_salary'], 2) : '-' ?></td>
                                </tr>
                                <tr>
                                    <td>Employment Status</td>
                                    <td><?= htmlspecialchars($worker['employment_status'] ?? '-') ?></td>
                                </tr>
                                <?php if (!empty($worker['employment_status_notes'])): ?>
                                <tr>
                                    <td>Status Notes</td>
                                    <td><?= htmlspecialchars($worker['employment_status_notes']) ?></td>
                                </tr>
                                <?php endif; ?>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Immigration Status -->
                <div class="col-md-6">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-globe-europe-africa"></i> Immigration Status
                        </div>
                        <div class="card-body-modern">
                            <?php
                            $is_british = (strtolower($worker['nationality']) === 'british');
                            ?>

                            <?php if ($is_british): ?>
                                <!-- British National - Simplified View -->
                                <div class="alert alert-success mb-3">
                                    <i class="bi bi-check-circle-fill"></i> <strong>British National</strong> - No visa required
                                </div>
                                <table class="info-table w-100">
                                    <tr>
                                        <td>Passport Number</td>
                                        <td><?= htmlspecialchars($worker['passport_number'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Passport Issue Date</td>
                                        <td><?= $worker['passport_issue_date'] ? format_date($worker['passport_issue_date']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td>Passport Expiry</td>
                                        <td><?= $worker['passport_expiry'] ? format_date($worker['passport_expiry']) : '-' ?></td>
                                    </tr>
                                </table>
                            <?php else: ?>
                                <!-- Non-British - Full Immigration Details -->
                                <table class="info-table w-100">
                                    <tr>
                                        <td>Passport Number</td>
                                        <td><?= htmlspecialchars($worker['passport_number'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Passport Issue Date</td>
                                        <td><?= $worker['passport_issue_date'] ? format_date($worker['passport_issue_date']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td>Passport Expiry</td>
                                        <td><?= $worker['passport_expiry'] ? format_date($worker['passport_expiry']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td>Visa Type</td>
                                        <td><?= htmlspecialchars($worker['visa_type'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Visa Number</td>
                                        <td><?= htmlspecialchars($worker['visa_number'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Visa Issue Date</td>
                                        <td><?= $worker['visa_issue_date'] ? format_date($worker['visa_issue_date']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td>Visa Expiry</td>
                                        <td>
                                            <?php if ($worker['visa_expiry']): ?>
                                                <?= format_date($worker['visa_expiry']) ?>
                                                <?php if ($visa_status === 'expired'): ?>
                                                    <span class="badge bg-danger ms-2">Expired</span>
                                                <?php elseif ($visa_status === 'urgent'): ?>
                                                    <span class="badge bg-danger ms-2"><?= $days_to_visa_expiry ?> days</span>
                                                <?php elseif ($visa_status === 'warning'): ?>
                                                    <span class="badge bg-warning text-dark ms-2"><?= $days_to_visa_expiry ?> days</span>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>BRP Number</td>
                                        <td><?= htmlspecialchars($worker['brp_number'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>BRP Expiry</td>
                                        <td><?= $worker['brp_expiry'] ? format_date($worker['brp_expiry']) : '-' ?></td>
                                    </tr>
                                    <tr>
                                        <td>Share Code</td>
                                        <td><?= htmlspecialchars($worker['share_code'] ?? '-') ?></td>
                                    </tr>
                                    <tr>
                                        <td>Share Code Expiry</td>
                                        <td><?= $worker['share_code_expiry'] ? format_date($worker['share_code_expiry']) : '-' ?></td>
                                    </tr>
                                </table>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Contact Information -->
                <div class="col-md-6">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-envelope"></i> Contact Information
                        </div>
                        <div class="card-body-modern">
                            <table class="info-table w-100">
                                <tr>
                                    <td>Work Email</td>
                                    <td><?= htmlspecialchars($worker['email'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Personal Email</td>
                                    <td><?= htmlspecialchars($worker['personal_email'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Work Phone</td>
                                    <td><?= htmlspecialchars($worker['phone'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Mobile Phone</td>
                                    <td><?= htmlspecialchars($worker['mobile_phone'] ?? '-') ?></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Banking Information -->
                <div class="col-md-6">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-bank"></i> Banking Information
                        </div>
                        <div class="card-body-modern">
                            <table class="info-table w-100">
                                <tr>
                                    <td>Bank Name</td>
                                    <td><?= htmlspecialchars($worker['bank_name'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Account Number</td>
                                    <td><?= htmlspecialchars($worker['bank_account_number'] ?? '-') ?></td>
                                </tr>
                                <tr>
                                    <td>Sort Code</td>
                                    <td><?= htmlspecialchars($worker['bank_sort_code'] ?? '-') ?></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- Additional Notes -->
                <?php if (!empty($worker['notes'])): ?>
                <div class="col-12">
                    <div class="modern-card">
                        <div class="card-header-modern">
                            <i class="bi bi-sticky"></i> Additional Notes
                        </div>
                        <div class="card-body-modern">
                            <p><?= nl2br(htmlspecialchars($worker['notes'])) ?></p>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- QUALIFICATIONS TAB -->
        <div class="tab-pane fade" id="qualifications" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-mortarboard"></i> Qualifications & Certifications
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/add_qualification?worker_id=' . $worker_id : 'add_qualification.php?worker_id=' . $worker_id ?>" class="btn btn-sm btn-primary float-end">
                        <i class="bi bi-plus-circle"></i> Add Qualification
                    </a>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($qualifications)): ?>
                        <div class="empty-state">
                            <i class="bi bi-mortarboard-fill"></i>
                            <p>No qualifications recorded</p>
                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/add_qualification?worker_id=' . $worker_id : 'add_qualification.php?worker_id=' . $worker_id ?>" class="btn btn-primary">
                                <i class="bi bi-plus-circle"></i> Add First Qualification
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Qualification</th>
                                        <th>Institution</th>
                                        <th>Year</th>
                                        <th>Grade/Result</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($qualifications as $qual): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($qual['qualification_name']) ?></td>
                                        <td><?= htmlspecialchars($qual['institution']) ?></td>
                                        <td><?= htmlspecialchars($qual['year_obtained']) ?></td>
                                        <td><?= htmlspecialchars($qual['grade'] ?? '-') ?></td>
                                        <td>
                                            <a href="delete_qualification.php?id=<?= $qual['id'] ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete this qualification?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- ADDRESSES TAB -->
        <div class="tab-pane fade" id="addresses" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-house"></i> Address History
                </div>
                <div class="card-body-modern">
                    <?php if (empty($addresses)): ?>
                        <div class="empty-state">
                            <i class="bi bi-house-fill"></i>
                            <p>No address history recorded</p>
                            <small class="text-muted">Address history is required for compliance and background checks</small>
                        </div>
                    <?php else: ?>
                        <div class="row g-3">
                            <?php foreach ($addresses as $addr): ?>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-body">
                                        <?php
                                        $is_current = (!$addr['move_out_date'] || $addr['move_out_date'] >= date('Y-m-d'));
                                        if ($is_current):
                                        ?>
                                            <span class="badge bg-success mb-2">Current Address</span>
                                        <?php endif; ?>

                                        <p class="mb-1"><strong><?= htmlspecialchars($addr['address_line1']) ?></strong></p>
                                        <?php if ($addr['address_line2']): ?>
                                            <p class="mb-1"><?= htmlspecialchars($addr['address_line2']) ?></p>
                                        <?php endif; ?>
                                        <p class="mb-1"><?= htmlspecialchars($addr['city']) ?>, <?= htmlspecialchars($addr['postcode']) ?></p>
                                        <p class="mb-2"><?= htmlspecialchars($addr['country']) ?></p>

                                        <small class="text-muted">
                                            <?= format_date($addr['move_in_date']) ?> -
                                            <?= $is_current ? 'Present' : format_date($addr['move_out_date']) ?>
                                        </small>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- EMERGENCY CONTACTS TAB -->
        <div class="tab-pane fade" id="emergency" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-telephone-plus"></i> Emergency Contacts
                </div>
                <div class="card-body-modern">
                    <?php if (empty($emergency_contacts) && !$has_worker_table_emergency): ?>
                        <div class="empty-state">
                            <i class="bi bi-telephone-fill"></i>
                            <p>No emergency contacts recorded</p>
                            <small class="text-muted">Emergency contact information is required for worker safety</small>
                        </div>
                    <?php else: ?>
                        <div class="row g-3">
                            <?php
                            // Display contacts from separate table first
                            foreach ($emergency_contacts as $contact): ?>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-body">
                                        <?php if ($contact['is_primary']): ?>
                                            <span class="badge bg-primary mb-2">Primary Contact</span>
                                        <?php endif; ?>

                                        <h6><?= htmlspecialchars($contact['contact_name']) ?></h6>
                                        <p class="mb-1"><strong>Relationship:</strong> <?= htmlspecialchars($contact['relationship']) ?></p>
                                        <p class="mb-1"><strong>Phone:</strong> <?= htmlspecialchars($contact['contact_phone']) ?></p>
                                        <?php if ($contact['contact_address']): ?>
                                            <p class="mb-0"><strong>Address:</strong> <?= htmlspecialchars($contact['contact_address']) ?></p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>

                            <?php
                            // If no separate table entries, show data from main worker table
                            if (empty($emergency_contacts) && $has_worker_table_emergency): ?>
                                <?php if (!empty($worker['emergency_contact']) || !empty($worker['emergency_phone'])): ?>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-body">
                                            <span class="badge bg-primary mb-2">Primary Contact</span>
                                            <h6><?= htmlspecialchars($worker['emergency_contact'] ?? 'Not specified') ?></h6>
                                            <?php if (!empty($worker['emergency_contact_relationship'])): ?>
                                                <p class="mb-1"><strong>Relationship:</strong> <?= htmlspecialchars($worker['emergency_contact_relationship']) ?></p>
                                            <?php endif; ?>
                                            <?php if (!empty($worker['emergency_phone'])): ?>
                                                <p class="mb-1"><strong>Phone:</strong> <?= htmlspecialchars($worker['emergency_phone']) ?></p>
                                            <?php endif; ?>
                                            <?php if (!empty($worker['emergency_contact_address'])): ?>
                                                <p class="mb-0"><strong>Address:</strong> <?= htmlspecialchars($worker['emergency_contact_address']) ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>

                                <?php if (!empty($worker['secondary_emergency_contact']) || !empty($worker['secondary_emergency_phone'])): ?>
                                <div class="col-md-6">
                                    <div class="card">
                                        <div class="card-body">
                                            <span class="badge bg-secondary mb-2">Secondary Contact</span>
                                            <h6><?= htmlspecialchars($worker['secondary_emergency_contact'] ?? 'Not specified') ?></h6>
                                            <?php if (!empty($worker['secondary_emergency_relationship'])): ?>
                                                <p class="mb-1"><strong>Relationship:</strong> <?= htmlspecialchars($worker['secondary_emergency_relationship']) ?></p>
                                            <?php endif; ?>
                                            <?php if (!empty($worker['secondary_emergency_phone'])): ?>
                                                <p class="mb-1"><strong>Phone:</strong> <?= htmlspecialchars($worker['secondary_emergency_phone']) ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- NOTES TAB -->
        <div class="tab-pane fade" id="notes" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-journal-text"></i> Notes & History
                </div>
                <div class="card-body-modern">
                    <?php if (empty($notes)): ?>
                        <div class="empty-state">
                            <i class="bi bi-journal-text"></i>
                            <p>No notes recorded</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($notes as $note): ?>
                        <div class="card mb-3">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div>
                                        <h6><?= htmlspecialchars($note['title'] ?? 'Note') ?></h6>
                                        <p class="mb-0"><?= nl2br(htmlspecialchars($note['note'])) ?></p>
                                    </div>
                                    <small class="text-muted"><?= format_datetime($note['created_at']) ?></small>
                                </div>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- LOCATION TAB (if module active) -->
        <?php if ($locations_active): ?>
        <div class="tab-pane fade" id="location" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-geo-alt"></i> Work Location Assignment
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker_id : 'edit.php?id=' . $worker_id ?>" class="btn btn-sm btn-primary float-end">
                        <i class="bi bi-pencil"></i> Change Location
                    </a>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($worker_location_assignments)): ?>
                        <!-- Multiple Location Assignments -->
                        <div class="row g-3">
                            <?php foreach ($worker_location_assignments as $index => $assignment): ?>
                            <div class="col-md-12">
                                <div class="card <?= $assignment['is_primary'] ? 'border-success' : '' ?>">
                                    <div class="card-header <?= $assignment['is_primary'] ? 'bg-success text-white' : 'bg-light' ?>">
                                        <strong>
                                            <i class="bi bi-geo-alt-fill"></i>
                                            <?= htmlspecialchars($assignment['location_name']) ?>
                                        </strong>
                                        <?= $assignment['is_primary'] ? '<span class="badge bg-white text-success float-end">Primary</span>' : '' ?>
                                        <?php if ($assignment['location_type']): ?>
                                            <span class="badge bg-secondary ms-2"><?= ucfirst($assignment['location_type']) ?></span>
                                        <?php endif; ?>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <table class="info-table w-100">
                                                    <tr>
                                                        <td width="40%"><strong>Address</strong></td>
                                                        <td>
                                                            <?= htmlspecialchars($assignment['location_address'] ?? '-') ?><br>
                                                            <?= htmlspecialchars($assignment['location_city'] ?? '') ?> <?= htmlspecialchars($assignment['location_postcode'] ?? '') ?>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Shift Name</strong></td>
                                                        <td><?= htmlspecialchars($assignment['shift_name'] ?? 'Not assigned') ?></td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Shift Type</strong></td>
                                                        <td>
                                                            <?php if ($assignment['shift_type']): ?>
                                                                <span class="badge bg-primary"><?= ucfirst($assignment['shift_type']) ?></span>
                                                            <?php else: ?>
                                                                -
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                    <tr>
                                                        <td><strong>Shift Times</strong></td>
                                                        <td>
                                                            <?php if ($assignment['start_time'] && $assignment['end_time']): ?>
                                                                <span class="badge bg-success">
                                                                    <i class="bi bi-clock"></i> <?= date('H:i', strtotime($assignment['start_time'])) ?>
                                                                </span>
                                                                <i class="bi bi-arrow-right mx-1"></i>
                                                                <span class="badge bg-danger">
                                                                    <i class="bi bi-clock-fill"></i> <?= date('H:i', strtotime($assignment['end_time'])) ?>
                                                                </span>
                                                            <?php else: ?>
                                                                -
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </div>
                                            <div class="col-md-6">
                                                <table class="info-table w-100">
                                                    <tr>
                                                        <td width="40%"><strong>Days of Week</strong></td>
                                                        <td>
                                                            <?php
                                                            $days = explode(',', $assignment['days_of_week'] ?? '');
                                                            foreach ($days as $day) {
                                                                if (trim($day)) {
                                                                    echo '<span class="badge bg-secondary me-1">' . htmlspecialchars(trim($day)) . '</span>';
                                                                }
                                                            }
                                                            ?>
                                                        </td>
                                                    </tr>
                                                    <?php if ($assignment['break_duration'] > 0): ?>
                                                    <tr>
                                                        <td><strong>Break Duration</strong></td>
                                                        <td><?= $assignment['break_duration'] ?> minutes</td>
                                                    </tr>
                                                    <?php endif; ?>
                                                    <?php if ($assignment['start_date'] || $assignment['end_date']): ?>
                                                    <tr>
                                                        <td><strong>Assignment Period</strong></td>
                                                        <td>
                                                            <?= $assignment['start_date'] ? format_date($assignment['start_date']) : 'Not set' ?>
                                                            <?php if ($assignment['end_date']): ?>
                                                                - <?= format_date($assignment['end_date']) ?>
                                                            <?php else: ?>
                                                                - Ongoing
                                                            <?php endif; ?>
                                                        </td>
                                                    </tr>
                                                    <?php endif; ?>
                                                    <tr>
                                                        <td colspan="2">
                                                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/locations/view?id=' . $assignment['location_id'] : '../locations/view.php?id=' . $assignment['location_id'] ?>" class="btn btn-sm btn-outline-primary">
                                                                <i class="bi bi-eye"></i> View Location Details
                                                            </a>
                                                        </td>
                                                    </tr>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="bi bi-geo-alt text-danger" style="font-size: 4rem;"></i>
                            <h5 class="text-danger mt-3">No Location Assigned</h5>
                            <p class="text-muted">This worker is not currently assigned to any work location.</p>
                            <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker_id : 'edit.php?id=' . $worker_id ?>" class="btn btn-primary">
                                <i class="bi bi-geo-alt"></i> Assign Work Location
                            </a>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- DOCUMENTS TAB (if module active) -->
        <?php if ($documents_active): ?>
        <div class="tab-pane fade" id="documents" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-files"></i> Documents
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/documents/upload?worker_id=' . $worker_id : '../documents/upload.php?worker_id=' . $worker_id ?>" class="btn btn-sm btn-primary float-end">
                        <i class="bi bi-upload"></i> Upload Document
                    </a>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($documents)): ?>
                        <div class="empty-state">
                            <i class="bi bi-files"></i>
                            <p>No documents uploaded</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Document Name</th>
                                        <th>Category</th>
                                        <th>Uploaded</th>
                                        <th>Expiry</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($documents as $doc): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($doc['document_name']) ?></td>
                                        <td><?= htmlspecialchars($doc['category'] ?? '-') ?></td>
                                        <td><?= format_date($doc['uploaded_at']) ?></td>
                                        <td>
                                            <?php if ($doc['expiry_date']): ?>
                                                <?= format_date($doc['expiry_date']) ?>
                                            <?php else: ?>
                                                -
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="../../<?= htmlspecialchars($doc['file_path']) ?>" class="btn btn-sm btn-outline-primary" download>
                                                <i class="bi bi-download"></i>
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- RTW CHECKS TAB (if module active) -->
        <?php if ($rtw_active): ?>
        <div class="tab-pane fade" id="rtw" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-file-earmark-check"></i> Right to Work Checks
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/rtw/add_check?worker_id=' . $worker_id : '../rtw/add_check.php?worker_id=' . $worker_id ?>" class="btn btn-sm btn-primary float-end">
                        <i class="bi bi-plus-circle"></i> Add RTW Check
                    </a>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($rtw_checks)): ?>
                        <div class="empty-state">
                            <i class="bi bi-file-earmark-check"></i>
                            <p>No RTW checks recorded</p>
                            <small class="text-danger">Right to Work checks are legally required</small>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Check Date</th>
                                        <th>Check Type</th>
                                        <th>Result</th>
                                        <th>Next Check Due</th>
                                        <th>Checked By</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($rtw_checks as $check): ?>
                                    <tr>
                                        <td><?= format_date($check['check_date']) ?></td>
                                        <td><?= htmlspecialchars($check['check_type']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= $check['result'] === 'pass' ? 'success' : 'danger' ?>">
                                                <?= ucfirst($check['result']) ?>
                                            </span>
                                        </td>
                                        <td><?= $check['next_check_due'] ? format_date($check['next_check_due']) : '-' ?></td>
                                        <td><?= htmlspecialchars($check['checked_by']) ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- COS TAB (if module active) -->
        <?php if ($cos_active): ?>
        <div class="tab-pane fade" id="cos" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-file-earmark-medical"></i> Certificate of Sponsorship
                </div>
                <div class="card-body-modern">
                    <?php if (empty($cos_records)): ?>
                        <div class="empty-state">
                            <i class="bi bi-file-earmark-medical"></i>
                            <p>No CoS records found</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>CoS Number</th>
                                        <th>Issue Date</th>
                                        <th>Start Date</th>
                                        <th>End Date</th>
                                        <th>Job Title</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($cos_records as $cos): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($cos['cos_number']) ?></td>
                                        <td><?= format_date($cos['issue_date']) ?></td>
                                        <td><?= format_date($cos['start_date']) ?></td>
                                        <td><?= format_date($cos['end_date']) ?></td>
                                        <td><?= htmlspecialchars($cos['job_title']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= $cos['status'] === 'active' ? 'success' : 'secondary' ?>">
                                                <?= ucfirst($cos['status']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- ATTENDANCE TAB (if module active) -->
        <?php if ($attendance_active): ?>
        <div class="tab-pane fade" id="attendance" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-calendar-check"></i> Recent Attendance (Last 30 Days)
                    <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/attendance/list?worker_id=' . $worker_id : '../attendance/list.php?worker_id=' . $worker_id ?>" class="btn btn-sm btn-primary float-end">
                        View All Attendance
                    </a>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($recent_attendance)): ?>
                        <div class="empty-state">
                            <i class="bi bi-calendar-check"></i>
                            <p>No attendance records found</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover table-sm">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Status</th>
                                        <th>Time In</th>
                                        <th>Time Out</th>
                                        <th>Hours</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_attendance as $att): ?>
                                    <tr>
                                        <td><?= format_date($att['attendance_date']) ?></td>
                                        <td><?= htmlspecialchars($att['status']) ?></td>
                                        <td><?= $att['time_in'] ?? '-' ?></td>
                                        <td><?= $att['time_out'] ?? '-' ?></td>
                                        <td><?= $att['hours_worked'] ?? '-' ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- HOLIDAYS TAB (if module active) -->
        <?php if ($holidays_active): ?>
        <div class="tab-pane fade" id="holidays" role="tabpanel">
            <div class="modern-card">
                <div class="card-header-modern">
                    <i class="bi bi-umbrella"></i> Holiday Requests
                </div>
                <div class="card-body-modern">
                    <?php if (empty($worker_holidays)): ?>
                        <div class="empty-state">
                            <i class="bi bi-umbrella"></i>
                            <p>No holiday requests found</p>
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Start Date</th>
                                        <th>End Date</th>
                                        <th>Days</th>
                                        <th>Type</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($worker_holidays as $holiday): ?>
                                    <tr>
                                        <td><?= format_date($holiday['start_date']) ?></td>
                                        <td><?= format_date($holiday['end_date']) ?></td>
                                        <td><?= $holiday['total_days'] ?></td>
                                        <td><?= htmlspecialchars($holiday['type']) ?></td>
                                        <td>
                                            <span class="badge bg-<?= $holiday['status'] === 'approved' ? 'success' : ($holiday['status'] === 'pending' ? 'warning' : 'danger') ?>">
                                                <?= ucfirst($holiday['status']) ?>
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

    </div>
</div>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
