<?php
/**
 * Process Worker Activation
 * Handles the backend logic for activating workers
 */

require_once '../../config/database.php';
require_once '../../includes/database.php';
session_start();

$pdo = get_database_connection();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Invalid request method');
}

$worker_id = $_POST['worker_id'] ?? null;
$start_date = $_POST['start_date'] ?? null;
$probation_end_date = $_POST['probation_end_date'] ?: null;
$activation_notes = $_POST['activation_notes'] ?: null;
$rtw_check_id = $_POST['rtw_check_id'] ?: null;

if (!$worker_id || !$start_date) {
    die('Missing required fields');
}

try {
    $pdo->beginTransaction();

    // Fetch current worker data
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
    $stmt->execute([$worker_id]);
    $worker = $stmt->fetch();

    if (!$worker) {
        throw new Exception('Worker not found');
    }

    if ($worker['status'] === 'active') {
        throw new Exception('Worker is already active');
    }

    $previous_status = $worker['status'];

    // Store confirmed details as JSON
    $confirmed_details = [
        'employee_number' => $worker['employee_number'],
        'full_name' => $worker['full_name'],
        'nationality' => $worker['nationality'],
        'job_title' => $worker['job_title'],
        'department' => $worker['department'],
        'contract_type' => $worker['contract_type'],
        'email' => $worker['email'],
        'phone' => $worker['phone'],
        'previous_start_date' => $worker['start_date'],
        'activated_at' => date('Y-m-d H:i:s')
    ];

    // Update worker to active status with start date
    $update_stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "sponsored_workers
                                   SET status = 'active',
                                       start_date = ?,
                                       probation_end_date = ?,
                                       awaiting_visa = 0,
                                       updated_at = NOW()
                                   WHERE id = ?");
    $update_stmt->execute([$start_date, $probation_end_date, $worker_id]);

    // Record activation in activation history table
    $activation_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_activation
                                       (worker_id, activated_by, start_date, previous_status, new_status,
                                        rtw_check_id, activation_notes, confirmed_details)
                                       VALUES (?, ?, ?, ?, 'active', ?, ?, ?)");

    $activated_by = $_SESSION['user_id'] ?? 1; // Default to admin if no session

    $activation_stmt->execute([
        $worker_id,
        $activated_by,
        $start_date,
        $previous_status,
        $rtw_check_id,
        $activation_notes,
        json_encode($confirmed_details)
    ]);

    $pdo->commit();

    // Determine redirect URL based on referrer or default to view page
    $redirect_url = "view.php?id=$worker_id";
    if (isset($_SERVER['HTTP_REFERER']) && strpos($_SERVER['HTTP_REFERER'], 'list.php') !== false) {
        $redirect_url = "list.php?success=Worker activated successfully";
    } else {
        $redirect_url = "view.php?id=$worker_id&success=Worker activated successfully";
    }

    // Check if this was opened in a modal (has window.opener)
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>Activation Successful</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body>
        <div class="container mt-5">
            <div class="alert alert-success text-center">
                <h4><i class="bi bi-check-circle-fill"></i> Worker Activated Successfully!</h4>
                <p>
                    <strong><?= htmlspecialchars($worker['full_name']) ?></strong> has been activated.<br>
                    Employment start date: <?= date('d/m/Y', strtotime($start_date)) ?>
                </p>
                <p class="mb-0">Redirecting...</p>
            </div>
        </div>

        <script>
            setTimeout(function() {
                // If opened in modal/popup, close and refresh parent
                if (window.opener) {
                    window.opener.location.reload();
                    window.close();
                } else {
                    // Otherwise redirect to view page
                    window.location.href = '<?= $redirect_url ?>';
                }
            }, 2000);
        </script>
    </body>
    </html>
    <?php

} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }

    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>Activation Failed</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    </head>
    <body>
        <div class="container mt-5">
            <div class="alert alert-danger text-center">
                <h4><i class="bi bi-x-circle-fill"></i> Activation Failed</h4>
                <p><?= htmlspecialchars($e->getMessage()) ?></p>
                <button onclick="window.history.back()" class="btn btn-primary">Go Back</button>
            </div>
        </div>
    </body>
    </html>
    <?php
}
?>
