<?php
$page_title = 'Workers List';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

// Search and filter
$search = $_GET['search'] ?? '';
$status = $_GET['status'] ?? '';
$department = $_GET['department'] ?? '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 25;

// Build query - EXCLUDE deleted workers by default unless specifically filtered
$where = ["1=1"];
$params = [];

// By default, hide deleted workers unless status filter is explicitly set to "deleted"
if ($status === 'deleted') {
    $where[] = "sw.status = 'deleted'";
} elseif ($status) {
    $where[] = "sw.status = ?";
    $params[] = $status;
} else {
    // No status filter - exclude deleted workers
    $where[] = "sw.status != 'deleted'";
}

if ($search) {
    $where[] = "(sw.full_name LIKE ? OR sw.employee_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
}

if ($department) {
    $where[] = "sw.department = ?";
    $params[] = $department;
}

$where_clause = implode(' AND ', $where);

// Count total
$count_sql = "SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers sw WHERE $where_clause";
$total = $pdo->prepare($count_sql);
$total->execute($params);
$total_records = $total->fetchColumn();

// Calculate pagination
$total_pages = ceil($total_records / $per_page);
$offset = ($page - 1) * $per_page;
$has_prev = $page > 1;
$has_next = $page < $total_pages;

// Fetch workers grouped by department
$sql = "SELECT sw.*,
        DATEDIFF(sw.visa_expiry, CURDATE()) as days_to_expiry
        FROM " . DB_PREFIX . "sponsored_workers sw
        WHERE $where_clause
        ORDER BY
            CASE
                WHEN sw.department IS NULL OR sw.department = '' THEN 1
                ELSE 0
            END,
            sw.department ASC,
            sw.full_name ASC
        LIMIT {$per_page} OFFSET {$offset}";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$all_workers = $stmt->fetchAll();

// Group workers by department
$workers_by_dept = [];
$unassigned = [];
foreach ($all_workers as $worker) {
    $dept = $worker['department'] ?: 'Unassigned';
    if ($dept === 'Unassigned') {
        $unassigned[] = $worker;
    } else {
        if (!isset($workers_by_dept[$dept])) {
            $workers_by_dept[$dept] = [];
        }
        $workers_by_dept[$dept][] = $worker;
    }
}

// Get filter options
$statuses = [
    'active' => 'Active',
    'pre_employment' => 'Pre-Employment',
    'pending_rtw' => 'Pending RTW',
    'inactive' => 'Inactive',
    'terminated' => 'Terminated',
    'deleted' => 'Deleted'
];

// Fetch unique departments for filter
$dept_sql = "SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers WHERE department IS NOT NULL AND department != '' ORDER BY department";
$dept_stmt = $pdo->query($dept_sql);
$departments = [];
while ($row = $dept_stmt->fetch()) {
    $departments[$row['department']] = $row['department'];
}

// Get stats for cards (exclude deleted workers)
$total_workers = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status != 'deleted'")->fetchColumn();
$active_workers = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active'")->fetchColumn();
$expiring_visas = $pdo->query("SELECT COUNT(*) FROM " . DB_PREFIX . "sponsored_workers WHERE status != 'deleted' AND visa_expiry <= DATE_ADD(CURDATE(), INTERVAL 30 DAY) AND visa_expiry >= CURDATE()")->fetchColumn();
$total_departments = $pdo->query("SELECT COUNT(DISTINCT department) FROM " . DB_PREFIX . "sponsored_workers WHERE status != 'deleted' AND department IS NOT NULL AND department != ''")->fetchColumn();

// Function to get status badge class and label
function get_status_badge($status) {
    $badge_class = 'bg-secondary';
    $label = ucfirst(str_replace('_', ' ', $status));

    if ($status === 'active') {
        $badge_class = 'bg-success';
    } elseif ($status === 'pre_employment') {
        $badge_class = 'bg-info';
        $label = 'Pre-Employment';
    } elseif ($status === 'pending_rtw') {
        $badge_class = 'bg-warning text-dark';
        $label = 'Pending RTW';
    } elseif ($status === 'terminated') {
        $badge_class = 'bg-danger';
    } elseif ($status === 'inactive') {
        $badge_class = 'bg-secondary';
    }

    return ['class' => $badge_class, 'label' => $label];
}
?>

<!-- Success Message -->
<?php if (isset($_GET['success'])): ?>
<div class="alert alert-success alert-dismissible fade show">
    <i class="bi bi-check-circle-fill"></i> <strong>Success!</strong> Worker has been added successfully.
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<!-- Quick Stats Grid -->
<div class="quick-actions-grid mb-4">
    <div class="quick-action-btn gradient-blue">
        <div class="quick-action-icon">
            <i class="fas fa-users"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($total_workers); ?> Total Workers</span>
    </div>

    <div class="quick-action-btn gradient-green">
        <div class="quick-action-icon">
            <i class="ri-user-star-fill"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($active_workers); ?> Active</span>
    </div>

    <div class="quick-action-btn gradient-orange">
        <div class="quick-action-icon">
            <span class="material-symbols-outlined" style="font-size: 1.5rem;">assignment_late</span>
        </div>
        <span class="quick-action-label"><?php echo number_format($expiring_visas); ?> Expiring Soon</span>
    </div>

    <div class="quick-action-btn gradient-purple">
        <div class="quick-action-icon">
            <i class="ri-building-fill"></i>
        </div>
        <span class="quick-action-label"><?php echo number_format($total_departments); ?> Departments</span>
    </div>
</div>

<div class="stat-card">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h5><i class="bi bi-people"></i> Workers (<?= $total_records ?>)</h5>
        <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/workers/add' : 'add.php'; ?>" class="btn btn-primary">
            <i class="bi bi-person-plus"></i> Add New Worker
        </a>
    </div>
    
    <!-- Search and Filter -->
    <form method="GET" class="row g-3 mb-4">
        <div class="col-md-4">
            <input type="text" name="search" class="form-control" 
                   placeholder="Search name or employee number..." 
                   value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-3">
            <select name="status" class="form-select">
                <option value="">All Statuses</option>
                <?php foreach ($statuses as $value => $label): ?>
                    <option value="<?= $value ?>" <?= $status === $value ? 'selected' : '' ?>>
                        <?= htmlspecialchars($label) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-3">
            <select name="department" class="form-select">
                <option value="">All Departments</option>
                <?php foreach ($departments as $value => $label): ?>
                    <option value="<?= $value ?>" <?= $department === $value ? 'selected' : '' ?>>
                        <?= htmlspecialchars($label) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="submit" class="btn btn-secondary w-100">
                <i class="bi bi-search"></i> Search
            </button>
        </div>
    </form>
    
    <!-- Workers Table - Grouped by Department -->
    <?php if (empty($all_workers)): ?>
        <div class="alert alert-info text-center">
            <i class="bi bi-inbox" style="font-size: 48px;"></i>
            <p class="mt-2">No workers found</p>
        </div>
    <?php else: ?>
        <?php foreach ($workers_by_dept as $dept_name => $dept_workers): ?>
        <div class="department-section mb-4">
            <div class="department-header bg-primary text-white p-3 rounded-top">
                <h5 class="mb-0">
                    <i class="bi bi-building"></i> <?= htmlspecialchars($dept_name) ?>
                    <span class="badge bg-light text-dark ms-2"><?= count($dept_workers) ?> workers</span>
                </h5>
            </div>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>Employee #</th>
                            <th>Name</th>
                            <th>Job Title</th>
                            <th>Line Manager</th>
                            <th>Visa Status</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($dept_workers as $worker): ?>
                        <tr>
                            <td><?= htmlspecialchars($worker['employee_number']) ?></td>
                            <td>
                                <strong><?= htmlspecialchars($worker['full_name']) ?></strong>
                            </td>
                            <td><?= htmlspecialchars($worker['job_title'] ?: '-') ?></td>
                            <td><?= htmlspecialchars($worker['line_manager'] ?: '-') ?></td>
                            <td>
                                <?php if ($worker['days_to_expiry'] !== null): ?>
                                    <?php if ($worker['days_to_expiry'] < 0): ?>
                                        <span class="badge bg-danger">Expired</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 30): ?>
                                        <span class="badge bg-danger"><?= $worker['days_to_expiry'] ?> days</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 90): ?>
                                        <span class="badge bg-warning text-dark"><?= $worker['days_to_expiry'] ?> days</span>
                                    <?php else: ?>
                                        <span class="badge bg-success">Valid</span>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="badge bg-secondary">N/A</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php $status_info = get_status_badge($worker['status']); ?>
                                <span class="badge <?= $status_info['class'] ?>">
                                    <?= $status_info['label'] ?>
                                </span>
                            </td>
                            <td>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/view?id=' . $worker['id'] : 'view.php?id=' . $worker['id'] ?>" class="btn btn-sm btn-outline-primary" title="View">
                                    <i class="bi bi-eye"></i>
                                </a>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker['id'] : 'edit.php?id=' . $worker['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <?php if ($worker['status'] === 'pre_employment' || $worker['status'] === 'pending_rtw'): ?>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/activate_worker?id=' . $worker['id'] : 'activate_worker.php?id=' . $worker['id'] ?>"
                                   class="btn btn-sm btn-success"
                                   title="Activate Worker"
                                   target="_blank">
                                    <i class="bi bi-check-circle-fill"></i>
                                </a>
                                <?php endif; ?>
                                <?php
                                // Get current user for permission check
                                if (!isset($current_user)) {
                                    $current_user = get_logged_in_user();
                                }

                                // Show delete button based on permission level
                                if ($current_user && $current_user['permission_level'] >= 70):
                                    if ($current_user['permission_level'] >= 100 && $worker['status'] === 'deleted'):
                                        // Admin seeing deleted worker - show PERMANENT DELETE with PIN
                                ?>
                                <button type="button" class="btn btn-sm btn-danger"
                                        onclick="showPermanentDeleteModal(<?= $worker['id'] ?>, '<?= htmlspecialchars($worker['full_name'], ENT_QUOTES) ?>')"
                                        title="Permanently Delete">
                                    <i class="bi bi-trash-fill"></i>
                                </button>
                                <?php elseif ($worker['status'] !== 'deleted'): ?>
                                <!-- Soft delete for all users with permission >= 70 -->
                                <button type="button" class="btn btn-sm btn-outline-danger"
                                        onclick="showSoftDeleteModal(<?= $worker['id'] ?>, '<?= htmlspecialchars($worker['full_name'], ENT_QUOTES) ?>')"
                                        title="Delete">
                                    <i class="bi bi-trash"></i>
                                </button>
                                <?php endif; endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endforeach; ?>

        <!-- Unassigned Workers Section -->
        <?php if (!empty($unassigned)): ?>
        <div class="department-section mb-4">
            <div class="department-header bg-secondary text-white p-3 rounded-top">
                <h5 class="mb-0">
                    <i class="bi bi-question-circle"></i> Unassigned Department
                    <span class="badge bg-light text-dark ms-2"><?= count($unassigned) ?> workers</span>
                </h5>
            </div>
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th>Employee #</th>
                            <th>Name</th>
                            <th>Job Title</th>
                            <th>Line Manager</th>
                            <th>Visa Status</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($unassigned as $worker): ?>
                        <tr>
                            <td><?= htmlspecialchars($worker['employee_number']) ?></td>
                            <td>
                                <strong><?= htmlspecialchars($worker['full_name']) ?></strong>
                            </td>
                            <td><?= htmlspecialchars($worker['job_title'] ?: '-') ?></td>
                            <td><?= htmlspecialchars($worker['line_manager'] ?: '-') ?></td>
                            <td>
                                <?php if ($worker['days_to_expiry'] !== null): ?>
                                    <?php if ($worker['days_to_expiry'] < 0): ?>
                                        <span class="badge bg-danger">Expired</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 30): ?>
                                        <span class="badge bg-danger"><?= $worker['days_to_expiry'] ?> days</span>
                                    <?php elseif ($worker['days_to_expiry'] <= 90): ?>
                                        <span class="badge bg-warning text-dark"><?= $worker['days_to_expiry'] ?> days</span>
                                    <?php else: ?>
                                        <span class="badge bg-success">Valid</span>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="badge bg-secondary">N/A</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php $status_info = get_status_badge($worker['status']); ?>
                                <span class="badge <?= $status_info['class'] ?>">
                                    <?= $status_info['label'] ?>
                                </span>
                            </td>
                            <td>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/view?id=' . $worker['id'] : 'view.php?id=' . $worker['id'] ?>" class="btn btn-sm btn-outline-primary" title="View">
                                    <i class="bi bi-eye"></i>
                                </a>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/edit?id=' . $worker['id'] : 'edit.php?id=' . $worker['id'] ?>" class="btn btn-sm btn-outline-warning" title="Edit">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <?php if ($worker['status'] === 'pre_employment' || $worker['status'] === 'pending_rtw'): ?>
                                <a href="<?= defined('MODULE_EMBEDDED') ? '../../m/workers/activate_worker?id=' . $worker['id'] : 'activate_worker.php?id=' . $worker['id'] ?>"
                                   class="btn btn-sm btn-success"
                                   title="Activate Worker"
                                   target="_blank">
                                    <i class="bi bi-check-circle-fill"></i>
                                </a>
                                <?php endif; ?>
                                <?php
                                // Get current user for permission check
                                if (!isset($current_user)) {
                                    $current_user = get_logged_in_user();
                                }

                                // Show delete button based on permission level
                                if ($current_user && $current_user['permission_level'] >= 70):
                                    if ($current_user['permission_level'] >= 100 && $worker['status'] === 'deleted'):
                                        // Admin seeing deleted worker - show PERMANENT DELETE with PIN
                                ?>
                                <button type="button" class="btn btn-sm btn-danger"
                                        onclick="showPermanentDeleteModal(<?= $worker['id'] ?>, '<?= htmlspecialchars($worker['full_name'], ENT_QUOTES) ?>')"
                                        title="Permanently Delete">
                                    <i class="bi bi-trash-fill"></i>
                                </button>
                                <?php elseif ($worker['status'] !== 'deleted'): ?>
                                <!-- Soft delete for all users with permission >= 70 -->
                                <button type="button" class="btn btn-sm btn-outline-danger"
                                        onclick="showSoftDeleteModal(<?= $worker['id'] ?>, '<?= htmlspecialchars($worker['full_name'], ENT_QUOTES) ?>')"
                                        title="Delete">
                                    <i class="bi bi-trash"></i>
                                </button>
                                <?php endif; endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php endif; ?>
    <?php endif; ?>
    
    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
    <nav>
        <ul class="pagination justify-content-center">
            <?php if ($has_prev): ?>
            <li class="page-item">
                <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Previous
                </a>
            </li>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $total_pages; $i++): ?>
            <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    <?= $i ?>
                </a>
            </li>
            <?php endfor; ?>

            <?php if ($has_next): ?>
            <li class="page-item">
                <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Next
                </a>
            </li>
            <?php endif; ?>
        </ul>
    </nav>
    <?php endif; ?>
</div>

<!-- Soft Delete Confirmation Modal -->
<div class="modal fade" id="softDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-warning text-dark">
                <h5 class="modal-title">
                    <i class="bi bi-exclamation-triangle-fill"></i> Confirm Delete
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="<?= defined('MODULE_EMBEDDED') ? '../../modules/workers/delete.php' : 'delete.php' ?>" id="softDeleteForm">
                <div class="modal-body">
                    <input type="hidden" name="id" id="soft_delete_worker_id">
                    <input type="hidden" name="delete_type" value="soft">

                    <p>Are you sure you want to delete this worker?</p>
                    <p><strong>Worker:</strong> <span id="soft_delete_worker_name"></span></p>

                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> This will mark the worker as deleted. Administrators can restore deleted workers later if needed.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle"></i> Cancel
                    </button>
                    <button type="submit" class="btn btn-warning">
                        <i class="bi bi-trash"></i> Delete Worker
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- PIN Modal for Permanent Delete -->
<div class="modal fade" id="permanentDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">
                    <i class="bi bi-exclamation-triangle-fill"></i> Permanent Delete Warning
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="<?= defined('MODULE_EMBEDDED') ? '../../modules/workers/delete.php' : 'delete.php' ?>" id="permanentDeleteForm">
                <div class="modal-body">
                    <div class="alert alert-danger">
                        <strong>⚠️ THIS ACTION CANNOT BE UNDONE!</strong>
                        <p class="mb-0 mt-2">You are about to <strong>PERMANENTLY DELETE</strong> this worker from the database. All records will be lost forever.</p>
                    </div>

                    <input type="hidden" name="id" id="delete_worker_id">
                    <input type="hidden" name="delete_type" value="permanent">

                    <p><strong>Worker:</strong> <span id="delete_worker_name"></span></p>

                    <div class="mb-3">
                        <label class="form-label"><strong>Enter your 4-digit PIN to confirm:</strong></label>
                        <input type="password"
                               name="pin"
                               id="delete_pin"
                               class="form-control form-control-lg text-center"
                               maxlength="4"
                               pattern="[0-9]{4}"
                               placeholder="••••"
                               required
                               autocomplete="off">
                        <small class="text-muted">Your admin PIN is required for permanent deletions</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="bi bi-x-circle"></i> Cancel
                    </button>
                    <button type="submit" class="btn btn-danger">
                        <i class="bi bi-trash-fill"></i> Permanently Delete
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
let softDeleteModal;
let permanentDeleteModal;

document.addEventListener('DOMContentLoaded', function() {
    softDeleteModal = new bootstrap.Modal(document.getElementById('softDeleteModal'));
    permanentDeleteModal = new bootstrap.Modal(document.getElementById('permanentDeleteModal'));
});

function showSoftDeleteModal(workerId, workerName) {
    document.getElementById('soft_delete_worker_id').value = workerId;
    document.getElementById('soft_delete_worker_name').textContent = workerName;
    softDeleteModal.show();
}

function showPermanentDeleteModal(workerId, workerName) {
    document.getElementById('delete_worker_id').value = workerId;
    document.getElementById('delete_worker_name').textContent = workerName;
    document.getElementById('delete_pin').value = '';
    permanentDeleteModal.show();

    // Focus on PIN input
    setTimeout(() => {
        document.getElementById('delete_pin').focus();
    }, 500);
}
</script>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>