<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();
$page_title = 'Edit Worker';
$current_page = 'workers';
$errors = [];

// Load nationalities
$nationalities_data = include '../../includes/nationalities.php';
$all_nationalities = $nationalities_data['all'];
$eu_eea_countries = $nationalities_data['eu_eea'];
$no_visa_countries = $nationalities_data['no_visa_required'];

// Fetch departments and job roles from centralized tables
$departments = [];
$job_roles = [];
$existing_staff = [];
try {
    $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "departments WHERE is_active = 1 ORDER BY name");
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "job_roles WHERE is_active = 1 ORDER BY title");
    $job_roles = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch all existing staff for line manager dropdown (hierarchy building)
    $stmt = $pdo->query("SELECT id, full_name, job_title, department, employee_number
                         FROM " . DB_PREFIX . "sponsored_workers
                         WHERE status = 'active'
                         ORDER BY full_name");
    $existing_staff = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Silently fail if tables don't exist yet
}

$worker_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

// Fetch worker
$stmt = $pdo->prepare("SELECT * FROM scms_sponsored_workers WHERE id = ?");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch();

if (!$worker) {
    header('Location: list.php');
    exit;
}

// Split full_name for display
$name_parts = explode(' ', $worker['full_name'], 3);
$first_name = $name_parts[0] ?? '';
$middle_name = count($name_parts) === 3 ? $name_parts[1] : '';
$last_name = count($name_parts) === 3 ? $name_parts[2] : ($name_parts[1] ?? '');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $first_name_post = trim($_POST['first_name'] ?? '');
    $middle_name_post = trim($_POST['middle_name'] ?? '');
    $last_name_post = trim($_POST['last_name'] ?? '');
    $nationality = trim($_POST['nationality'] ?? '');

    if (empty($first_name_post)) $errors[] = "First name required";
    if (empty($last_name_post)) $errors[] = "Last name required";
    if (empty($nationality)) $errors[] = "Nationality required";

    if (empty($errors)) {
        // Build full name
        $full_name = $first_name_post;
        if (!empty($middle_name_post)) $full_name .= ' ' . $middle_name_post;
        $full_name .= ' ' . $last_name_post;

        try {
            $stmt = $pdo->prepare("UPDATE scms_sponsored_workers SET
                full_name = ?, date_of_birth = ?, gender = ?, place_of_birth = ?, nationality = ?,
                passport_number = ?, passport_issue_date = ?, passport_expiry = ?, passport_country = ?,
                ni_number = ?, driving_license = ?,
                brp_number = ?, brp_expiry = ?, share_code = ?, share_code_expiry = ?,
                visa_type = ?, visa_number = ?, visa_issue_date = ?, visa_expiry = ?, visa_status = ?,
                indefinite_leave = ?, settled_status = ?, pre_settled_status = ?,
                cos_number = ?, cos_issue_date = ?, cos_start_date = ?, cos_end_date = ?,
                right_to_work_checked = ?, right_to_work_check_date = ?, right_to_work_expiry = ?,
                email = ?, personal_email = ?, phone = ?, mobile_phone = ?,
                address = ?, city = ?, postcode = ?, country = ?,
                emergency_contact = ?, emergency_phone = ?, emergency_contact_relationship = ?, emergency_contact_address = ?,
                secondary_emergency_contact = ?, secondary_emergency_phone = ?, secondary_emergency_relationship = ?,
                job_title = ?, department = ?, line_manager = ?, contract_type = ?,
                start_date = ?, probation_end_date = ?, annual_salary = ?, hourly_rate = ?, work_hours_per_week = ?,
                bank_name = ?, bank_account_number = ?, bank_sort_code = ?, tax_code = ?,
                notes = ?
                WHERE id = ?");

            $stmt->execute([
                $full_name,
                $_POST['date_of_birth'] ?: null,
                $_POST['gender'] ?: null,
                $_POST['place_of_birth'] ?: null,
                $nationality,
                $_POST['passport_number'] ?: null,
                $_POST['passport_issue_date'] ?: null,
                $_POST['passport_expiry'] ?: null,
                $_POST['passport_country'] ?: null,
                $_POST['ni_number'] ?: null,
                $_POST['driving_license'] ?: null,
                $_POST['brp_number'] ?: null,
                $_POST['brp_expiry'] ?: null,
                $_POST['share_code'] ?: null,
                $_POST['share_code_expiry'] ?: null,
                $_POST['visa_type'] ?: null,
                $_POST['visa_number'] ?: null,
                $_POST['visa_issue_date'] ?: null,
                $_POST['visa_expiry'] ?: null,
                $_POST['visa_status'] ?: null,
                isset($_POST['indefinite_leave']) ? 1 : 0,
                isset($_POST['settled_status']) ? 1 : 0,
                isset($_POST['pre_settled_status']) ? 1 : 0,
                $_POST['cos_number'] ?: null,
                $_POST['cos_issue_date'] ?: null,
                $_POST['cos_start_date'] ?: null,
                $_POST['cos_end_date'] ?: null,
                isset($_POST['right_to_work_checked']) ? 1 : 0,
                $_POST['right_to_work_check_date'] ?: null,
                $_POST['right_to_work_expiry'] ?: null,
                $_POST['email'] ?: null,
                $_POST['personal_email'] ?: null,
                $_POST['phone'] ?: null,
                $_POST['mobile_phone'] ?: null,
                $_POST['address'] ?: null,
                $_POST['city'] ?: null,
                $_POST['postcode'] ?: null,
                $_POST['country'] ?: 'United Kingdom',
                $_POST['emergency_contact'] ?: null,
                $_POST['emergency_phone'] ?: null,
                $_POST['emergency_contact_relationship'] ?: null,
                $_POST['emergency_contact_address'] ?: null,
                $_POST['secondary_emergency_contact'] ?: null,
                $_POST['secondary_emergency_phone'] ?: null,
                $_POST['secondary_emergency_relationship'] ?: null,
                $_POST['job_title'] ?: null,
                $_POST['department'] ?: null,
                $_POST['line_manager'] ?: null,
                $_POST['contract_type'] ?: null,
                $_POST['start_date'] ?: null,
                $_POST['probation_end_date'] ?: null,
                $_POST['annual_salary'] ?: null,
                $_POST['hourly_rate'] ?: null,
                $_POST['work_hours_per_week'] ?: null,
                $_POST['bank_name'] ?: null,
                $_POST['bank_account_number'] ?: null,
                $_POST['bank_sort_code'] ?: null,
                $_POST['tax_code'] ?: null,
                $_POST['notes'] ?: null,
                $worker_id
            ]);

            header("Location: list.php?success=1");
            exit;

        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }

    // If there were errors, update display values from POST
    $first_name = $first_name_post;
    $middle_name = $middle_name_post;
    $last_name = $last_name_post;
}

include '../../includes/header.php';

// Helper function for checked state
function is_checked($field, $worker) {
    return !empty($worker[$field]) && $worker[$field] == 1;
}

// Helper function for selected option
function is_selected($value, $current) {
    return $value === $current ? 'selected' : '';
}
?>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.conditional-field {
    display: none;
}

.conditional-field.show {
    display: block;
}

.info-badge {
    background: #e3f2fd;
    color: #1976d2;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    font-size: 0.875rem;
    margin-bottom: 1rem;
}
</style>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title"><i class="bi bi-pencil-fill"></i> Edit Worker</h1>
            <p class="welcome-subtitle">Update worker information</p>
        </div>
        <div class="welcome-actions">
            <a href="list.php" class="btn btn-secondary"><i class="bi bi-arrow-left"></i> Back to List</a>
        </div>
    </div>

    <?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <ul class="mb-0">
            <?php foreach ($errors as $e): ?>
                <li><?php echo htmlspecialchars($e); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST" id="workerForm">
                <!-- Personal Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-person"></i> Personal Information
                    </h6>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>First Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="first_name"
                                   value="<?= htmlspecialchars($first_name) ?>" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Middle Name</label>
                            <input type="text" class="form-control" name="middle_name"
                                   value="<?= htmlspecialchars($middle_name) ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Last Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="last_name"
                                   value="<?= htmlspecialchars($last_name) ?>" required>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Date of Birth</label>
                            <input type="date" class="form-control" name="date_of_birth"
                                   value="<?= htmlspecialchars($worker['date_of_birth'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Gender</label>
                            <select class="form-select" name="gender">
                                <option value="">Select...</option>
                                <option value="Male" <?= is_selected('Male', $worker['gender'] ?? '') ?>>Male</option>
                                <option value="Female" <?= is_selected('Female', $worker['gender'] ?? '') ?>>Female</option>
                                <option value="Other" <?= is_selected('Other', $worker['gender'] ?? '') ?>>Other</option>
                                <option value="Prefer not to say" <?= is_selected('Prefer not to say', $worker['gender'] ?? '') ?>>Prefer not to say</option>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Place of Birth</label>
                            <input type="text" class="form-control" name="place_of_birth"
                                   value="<?= htmlspecialchars($worker['place_of_birth'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Nationality</strong> <span class="text-danger">*</span></label>
                            <select class="form-select" name="nationality" id="nationality" required>
                                <option value="">Select nationality...</option>
                                <?php foreach ($all_nationalities as $nat): ?>
                                    <option value="<?= htmlspecialchars($nat) ?>"
                                        <?= is_selected($nat, $worker['nationality'] ?? '') ?>>
                                        <?= htmlspecialchars($nat) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Determines immigration requirements</small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">National Insurance Number</label>
                            <input type="text" class="form-control" name="ni_number"
                                   value="<?= htmlspecialchars($worker['ni_number'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Driving License Number</label>
                            <input type="text" class="form-control" name="driving_license"
                                   value="<?= htmlspecialchars($worker['driving_license'] ?? '') ?>">
                        </div>
                    </div>
                </div>

                <!-- Passport Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-passport"></i> Passport Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Number</label>
                            <input type="text" class="form-control" name="passport_number"
                                   value="<?= htmlspecialchars($worker['passport_number'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Country</label>
                            <input type="text" class="form-control" name="passport_country"
                                   value="<?= htmlspecialchars($worker['passport_country'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Issue Date</label>
                            <input type="date" class="form-control" name="passport_issue_date"
                                   value="<?= htmlspecialchars($worker['passport_issue_date'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Expiry Date</label>
                            <input type="date" class="form-control" name="passport_expiry"
                                   value="<?= htmlspecialchars($worker['passport_expiry'] ?? '') ?>">
                        </div>
                    </div>
                </div>

                <!-- Employment Status Section -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-briefcase-fill"></i> Employment Status
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Employment Status</strong> <span class="text-danger">*</span></label>
                            <select class="form-select" name="employment_status" id="employment_status" required>
                                <option value="">Select status...</option>
                                <option value="new_hire" <?= is_selected('new_hire', $worker['employment_status'] ?? 'new_hire') ?>>New Hire</option>
                                <option value="pre_employment" <?= is_selected('pre_employment', $worker['employment_status'] ?? '') ?>>Pre-Employment (CoS Issued, Awaiting Visa)</option>
                            </select>
                            <small class="text-muted">
                                <strong>New Hire:</strong> Standard new employee |
                                <strong>Pre-Employment:</strong> For sponsored workers awaiting visa approval
                            </small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Status Start Date</label>
                            <input type="date" class="form-control" name="employment_status_date"
                                   value="<?= htmlspecialchars($worker['employment_status_date'] ?? '') ?>">
                        </div>
                    </div>

                    <!-- Pre-Employment Specific Fields -->
                    <div id="preEmploymentFields" class="conditional-field <?= ($worker['employment_status'] ?? '') === 'pre_employment' ? 'show' : '' ?>">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> <strong>Pre-Employment Mode:</strong> Use this for workers who have been offered a job and CoS issued, but are awaiting visa approval.
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Location</label>
                                <select class="form-select" name="cos_location" id="cos_location">
                                    <option value="">Select...</option>
                                    <option value="in_country" <?= is_selected('in_country', $worker['cos_location'] ?? '') ?>>In-Country Application</option>
                                    <option value="out_of_country" <?= is_selected('out_of_country', $worker['cos_location'] ?? '') ?>>Out of Country Application</option>
                                </select>
                                <small class="text-muted">Determines which verification checks can be performed</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="form-check mt-4">
                                    <input class="form-check-input" type="checkbox" name="awaiting_visa" id="awaiting_visa" value="1" <?= !empty($worker['awaiting_visa']) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="awaiting_visa">
                                        <strong>Awaiting Visa Approval</strong>
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="alert alert-warning" id="inCountryNotice" style="display:none;">
                            <i class="bi bi-exclamation-triangle"></i> <strong>In-Country Application:</strong> Right to Work checks and NI number verification should be completed.
                        </div>

                        <div class="alert alert-warning" id="outCountryNotice" style="display:none;">
                            <i class="bi bi-exclamation-triangle"></i> <strong>Out of Country Application:</strong> Right to Work checks and some verifications cannot be completed until worker arrives in UK.
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Employment Status Notes</label>
                            <textarea class="form-control" name="employment_status_notes" rows="2"><?= htmlspecialchars($worker['employment_status_notes'] ?? '') ?></textarea>
                        </div>
                    </div>
                </div>

                <!-- Immigration Status -->
                <div class="form-section immigration-section" id="immigrationSection">
                    <h6 class="form-section-title">
                        <i class="bi bi-globe-europe-africa"></i> Immigration Status
                    </h6>

                    <div class="info-badge mb-3" id="britishNotice" style="display: none;">
                        <i class="bi bi-info-circle"></i> British and Irish nationals have automatic right to work in the UK.
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label"><strong>Visa/Immigration Status</strong></label>
                            <select class="form-select" name="visa_type" id="visa_type">
                                <option value="">Select immigration status...</option>
                                <optgroup label="No Visa Required">
                                    <option value="British Citizen" <?= is_selected('British Citizen', $worker['visa_type'] ?? '') ?>>British Citizen</option>
                                    <option value="Irish Citizen" <?= is_selected('Irish Citizen', $worker['visa_type'] ?? '') ?>>Irish Citizen</option>
                                    <option value="Indefinite Leave to Remain" <?= is_selected('Indefinite Leave to Remain', $worker['visa_type'] ?? '') ?>>Indefinite Leave to Remain (ILR)</option>
                                </optgroup>
                                <optgroup label="EU Settlement Scheme">
                                    <option value="EU Settled Status" <?= is_selected('EU Settled Status', $worker['visa_type'] ?? '') ?>>EU Settled Status</option>
                                    <option value="EU Pre-Settled Status" <?= is_selected('EU Pre-Settled Status', $worker['visa_type'] ?? '') ?>>EU Pre-Settled Status</option>
                                </optgroup>
                                <optgroup label="Skilled Worker Route">
                                    <option value="Skilled Worker" <?= is_selected('Skilled Worker', $worker['visa_type'] ?? '') ?>>Skilled Worker Visa</option>
                                    <option value="Health and Care Worker" <?= is_selected('Health and Care Worker', $worker['visa_type'] ?? '') ?>>Health and Care Worker Visa</option>
                                    <option value="Senior or Specialist Worker" <?= is_selected('Senior or Specialist Worker', $worker['visa_type'] ?? '') ?>>Senior or Specialist Worker Visa</option>
                                </optgroup>
                                <optgroup label="Student & Graduate">
                                    <option value="Graduate" <?= is_selected('Graduate', $worker['visa_type'] ?? '') ?>>Graduate Visa</option>
                                    <option value="Student" <?= is_selected('Student', $worker['visa_type'] ?? '') ?>>Student Visa</option>
                                </optgroup>
                                <optgroup label="Family & Dependant">
                                    <option value="Spouse Visa" <?= is_selected('Spouse Visa', $worker['visa_type'] ?? '') ?>>Spouse/Partner Visa</option>
                                    <option value="Dependant" <?= is_selected('Dependant', $worker['visa_type'] ?? '') ?>>Dependant Visa</option>
                                    <option value="Family" <?= is_selected('Family', $worker['visa_type'] ?? '') ?>>Family Visa</option>
                                </optgroup>
                                <optgroup label="Other Routes">
                                    <option value="Global Talent" <?= is_selected('Global Talent', $worker['visa_type'] ?? '') ?>>Global Talent Visa</option>
                                    <option value="Innovator Founder" <?= is_selected('Innovator Founder', $worker['visa_type'] ?? '') ?>>Innovator Founder Visa</option>
                                    <option value="Start-up" <?= is_selected('Start-up', $worker['visa_type'] ?? '') ?>>Start-up Visa</option>
                                    <option value="Minister of Religion" <?= is_selected('Minister of Religion', $worker['visa_type'] ?? '') ?>>Minister of Religion Visa</option>
                                    <option value="Sportsperson" <?= is_selected('Sportsperson', $worker['visa_type'] ?? '') ?>>Sportsperson Visa</option>
                                    <option value="Youth Mobility" <?= is_selected('Youth Mobility', $worker['visa_type'] ?? '') ?>>Youth Mobility Scheme</option>
                                    <option value="Other" <?= is_selected('Other', $worker['visa_type'] ?? '') ?>>Other</option>
                                </optgroup>
                            </select>
                        </div>
                    </div>

                    <!-- Settlement checkboxes -->
                    <div class="row mb-3">
                        <div class="col-md-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="settled_status" id="settled_status" <?= is_checked('settled_status', $worker) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="settled_status">EU Settled Status</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="pre_settled_status" id="pre_settled_status" <?= is_checked('pre_settled_status', $worker) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="pre_settled_status">EU Pre-Settled Status</label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="indefinite_leave" id="indefinite_leave" <?= is_checked('indefinite_leave', $worker) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="indefinite_leave">Indefinite Leave to Remain (ILR)</label>
                            </div>
                        </div>
                    </div>

                    <!-- BRP -->
                    <div class="conditional-field" id="brpFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Number</label>
                                <input type="text" class="form-control" name="brp_number"
                                       value="<?= htmlspecialchars($worker['brp_number'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Expiry</label>
                                <input type="date" class="form-control" name="brp_expiry"
                                       value="<?= htmlspecialchars($worker['brp_expiry'] ?? '') ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Share Code -->
                    <div class="conditional-field" id="shareCodeFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code</label>
                                <input type="text" class="form-control" name="share_code"
                                       value="<?= htmlspecialchars($worker['share_code'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code Expiry</label>
                                <input type="date" class="form-control" name="share_code_expiry"
                                       value="<?= htmlspecialchars($worker['share_code_expiry'] ?? '') ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Visa Details -->
                    <div class="conditional-field" id="visaFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Number</label>
                                <input type="text" class="form-control" name="visa_number"
                                       value="<?= htmlspecialchars($worker['visa_number'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Status</label>
                                <select class="form-select" name="visa_status">
                                    <option value="">Select...</option>
                                    <option value="Valid" <?= is_selected('Valid', $worker['visa_status'] ?? '') ?>>Valid</option>
                                    <option value="Pending" <?= is_selected('Pending', $worker['visa_status'] ?? '') ?>>Pending</option>
                                    <option value="Expired" <?= is_selected('Expired', $worker['visa_status'] ?? '') ?>>Expired</option>
                                    <option value="Cancelled" <?= is_selected('Cancelled', $worker['visa_status'] ?? '') ?>>Cancelled</option>
                                </select>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Issue Date</label>
                                <input type="date" class="form-control" name="visa_issue_date"
                                       value="<?= htmlspecialchars($worker['visa_issue_date'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Expiry Date</label>
                                <input type="date" class="form-control" name="visa_expiry"
                                       value="<?= htmlspecialchars($worker['visa_expiry'] ?? '') ?>">
                            </div>
                        </div>
                    </div>

                    <!-- CoS -->
                    <div class="conditional-field" id="cosFields">
                        <h6 class="mt-3 mb-3"><i class="bi bi-file-earmark-text"></i> Certificate of Sponsorship</h6>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Number</label>
                                <input type="text" class="form-control" name="cos_number"
                                       value="<?= htmlspecialchars($worker['cos_number'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Issue Date</label>
                                <input type="date" class="form-control" name="cos_issue_date"
                                       value="<?= htmlspecialchars($worker['cos_issue_date'] ?? '') ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Start Date</label>
                                <input type="date" class="form-control" name="cos_start_date"
                                       value="<?= htmlspecialchars($worker['cos_start_date'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS End Date</label>
                                <input type="date" class="form-control" name="cos_end_date"
                                       value="<?= htmlspecialchars($worker['cos_end_date'] ?? '') ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Right to Work -->
                    <div class="row mt-3">
                        <div class="col-md-12">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" name="right_to_work_checked" id="right_to_work_checked" <?= is_checked('right_to_work_checked', $worker) ? 'checked' : '' ?>>
                                <label class="form-check-label" for="right_to_work_checked">
                                    <strong>Right to Work Check Completed</strong>
                                </label>
                            </div>
                        </div>
                    </div>

                    <div class="conditional-field" id="rtwFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check Date</label>
                                <input type="date" class="form-control" name="right_to_work_check_date"
                                       value="<?= htmlspecialchars($worker['right_to_work_check_date'] ?? '') ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Next Check Due</label>
                                <input type="date" class="form-control" name="right_to_work_expiry"
                                       value="<?= htmlspecialchars($worker['right_to_work_expiry'] ?? '') ?>">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Contact Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-envelope"></i> Contact Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Email</label>
                            <input type="email" class="form-control" name="email"
                                   value="<?= htmlspecialchars($worker['email'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Personal Email</label>
                            <input type="email" class="form-control" name="personal_email"
                                   value="<?= htmlspecialchars($worker['personal_email'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Phone</label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?= htmlspecialchars($worker['phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Mobile Phone</label>
                            <input type="tel" class="form-control" name="mobile_phone"
                                   value="<?= htmlspecialchars($worker['mobile_phone'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Address</label>
                            <textarea class="form-control" name="address" rows="2"><?= htmlspecialchars($worker['address'] ?? '') ?></textarea>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">City</label>
                            <input type="text" class="form-control" name="city"
                                   value="<?= htmlspecialchars($worker['city'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Postcode</label>
                            <input type="text" class="form-control" name="postcode"
                                   value="<?= htmlspecialchars($worker['postcode'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Country</label>
                            <input type="text" class="form-control" name="country"
                                   value="<?= htmlspecialchars($worker['country'] ?? 'United Kingdom') ?>">
                        </div>
                    </div>
                </div>

                <!-- Employment Details -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-briefcase"></i> Employment Details
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Job Title</label>
                            <select class="form-select" name="job_title">
                                <option value="">Select job role...</option>
                                <?php foreach ($job_roles as $role): ?>
                                    <option value="<?= htmlspecialchars($role['title']) ?>"
                                        <?= is_selected($role['title'], $worker['job_title'] ?? '') ?>>
                                        <?= htmlspecialchars($role['title']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Or <a href="../../settings.php?tab=departments" target="_blank">add new role</a></small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Department</label>
                            <select class="form-select" name="department">
                                <option value="">Select department...</option>
                                <?php foreach ($departments as $dept): ?>
                                    <option value="<?= htmlspecialchars($dept['name']) ?>"
                                        <?= is_selected($dept['name'], $worker['department'] ?? '') ?>>
                                        <?= htmlspecialchars($dept['name']) ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Or <a href="../../settings.php?tab=departments" target="_blank">add new department</a></small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Line Manager</label>
                            <select class="form-select" name="line_manager" id="line_manager">
                                <option value="">Select line manager...</option>
                                <?php foreach ($existing_staff as $staff): ?>
                                    <option value="<?= htmlspecialchars($staff['full_name']) ?>"
                                        <?= is_selected($staff['full_name'], $worker['line_manager'] ?? '') ?>>
                                        <?= htmlspecialchars($staff['full_name']) ?>
                                        <?php if (!empty($staff['job_title'])): ?>
                                            - <?= htmlspecialchars($staff['job_title']) ?>
                                        <?php endif; ?>
                                        <?php if (!empty($staff['employee_number'])): ?>
                                            (<?= htmlspecialchars($staff['employee_number']) ?>)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Required for company hierarchy reports</small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Contract Type</label>
                            <select class="form-select" name="contract_type">
                                <option value="">Select...</option>
                                <option value="Full-time" <?= is_selected('Full-time', $worker['contract_type'] ?? '') ?>>Full-time</option>
                                <option value="Part-time" <?= is_selected('Part-time', $worker['contract_type'] ?? '') ?>>Part-time</option>
                                <option value="Fixed Term" <?= is_selected('Fixed Term', $worker['contract_type'] ?? '') ?>>Fixed Term</option>
                                <option value="Temporary" <?= is_selected('Temporary', $worker['contract_type'] ?? '') ?>>Temporary</option>
                                <option value="Zero Hours" <?= is_selected('Zero Hours', $worker['contract_type'] ?? '') ?>>Zero Hours</option>
                            </select>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Start Date</label>
                            <input type="date" class="form-control" name="start_date"
                                   value="<?= htmlspecialchars($worker['start_date'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Probation End Date</label>
                            <input type="date" class="form-control" name="probation_end_date"
                                   value="<?= htmlspecialchars($worker['probation_end_date'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Annual Salary (£)</label>
                            <input type="number" class="form-control" name="annual_salary" step="0.01"
                                   value="<?= htmlspecialchars($worker['annual_salary'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hourly Rate (£)</label>
                            <input type="number" class="form-control" name="hourly_rate" step="0.01"
                                   value="<?= htmlspecialchars($worker['hourly_rate'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hours per Week</label>
                            <input type="number" class="form-control" name="work_hours_per_week" step="0.5"
                                   value="<?= htmlspecialchars($worker['work_hours_per_week'] ?? '') ?>">
                        </div>
                    </div>
                </div>

                <!-- Banking & Tax -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-bank"></i> Banking & Tax Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Bank Name</label>
                            <input type="text" class="form-control" name="bank_name"
                                   value="<?= htmlspecialchars($worker['bank_name'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tax Code</label>
                            <input type="text" class="form-control" name="tax_code"
                                   value="<?= htmlspecialchars($worker['tax_code'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Account Number</label>
                            <input type="text" class="form-control" name="bank_account_number"
                                   value="<?= htmlspecialchars($worker['bank_account_number'] ?? '') ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Sort Code</label>
                            <input type="text" class="form-control" name="bank_sort_code"
                                   value="<?= htmlspecialchars($worker['bank_sort_code'] ?? '') ?>">
                        </div>
                    </div>
                </div>

                <!-- Emergency Contact -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-telephone-plus"></i> Emergency Contact
                    </h6>

                    <h6 class="mb-3">Primary Contact</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="emergency_contact"
                                   value="<?= htmlspecialchars($worker['emergency_contact'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="emergency_phone"
                                   value="<?= htmlspecialchars($worker['emergency_phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="emergency_contact_relationship"
                                   value="<?= htmlspecialchars($worker['emergency_contact_relationship'] ?? '') ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Contact Address</label>
                            <textarea class="form-control" name="emergency_contact_address" rows="2"><?= htmlspecialchars($worker['emergency_contact_address'] ?? '') ?></textarea>
                        </div>
                    </div>

                    <h6 class="mb-3 mt-4">Secondary Contact (Optional)</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="secondary_emergency_contact"
                                   value="<?= htmlspecialchars($worker['secondary_emergency_contact'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="secondary_emergency_phone"
                                   value="<?= htmlspecialchars($worker['secondary_emergency_phone'] ?? '') ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="secondary_emergency_relationship"
                                   value="<?= htmlspecialchars($worker['secondary_emergency_relationship'] ?? '') ?>">
                        </div>
                    </div>
                </div>

                <!-- Additional Notes -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-journal-text"></i> Additional Notes
                    </h6>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" rows="4"><?= htmlspecialchars($worker['notes'] ?? '') ?></textarea>
                        </div>
                    </div>
                </div>

                <div class="text-end mt-4">
                    <a href="list.php" class="btn btn-secondary me-2"><i class="bi bi-x-circle"></i> Cancel</a>
                    <button type="submit" class="btn btn-primary"><i class="bi bi-check-circle"></i> Save Changes</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Define nationality categories
const euEeaCountries = <?php echo json_encode($eu_eea_countries); ?>;
const noVisaCountries = <?php echo json_encode($no_visa_countries); ?>;

// Nationality change handler with progressive logic
document.getElementById('nationality').addEventListener('change', function() {
    const nationality = this.value.trim();
    const britishNotice = document.getElementById('britishNotice');
    const immigrationSection = document.querySelector('.immigration-section');

    // British or Irish - hide immigration section entirely
    if (noVisaCountries.includes(nationality)) {
        if (britishNotice) britishNotice.style.display = 'block';
        if (immigrationSection) immigrationSection.style.display = 'none';
        document.getElementById('visa_type').required = false;
    }
    // EU/EEA nationals - show EU-specific options
    else if (euEeaCountries.includes(nationality)) {
        if (britishNotice) britishNotice.style.display = 'none';
        if (immigrationSection) immigrationSection.style.display = 'block';
        document.getElementById('visa_type').required = true;
    }
    // All other nationals - show full visa options
    else if (nationality) {
        if (britishNotice) britishNotice.style.display = 'none';
        if (immigrationSection) immigrationSection.style.display = 'block';
        document.getElementById('visa_type').required = true;
    }
});

document.getElementById('visa_type').addEventListener('change', function() {
    const visaType = this.value;
    const brpFields = document.getElementById('brpFields');
    const shareCodeFields = document.getElementById('shareCodeFields');
    const visaFields = document.getElementById('visaFields');
    const cosFields = document.getElementById('cosFields');

    brpFields.classList.remove('show');
    shareCodeFields.classList.remove('show');
    visaFields.classList.remove('show');
    cosFields.classList.remove('show');

    const noVisaNeeded = ['British Citizen', 'Irish Citizen', 'Indefinite Leave to Remain'];

    if (noVisaNeeded.includes(visaType)) {
        if (visaType === 'Indefinite Leave to Remain') brpFields.classList.add('show');
        return;
    }

    if (visaType === 'EU Settled Status' || visaType === 'EU Pre-Settled Status') {
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        return;
    }

    const sponsoredVisas = ['Skilled Worker', 'Health and Care Worker', 'Senior or Specialist Worker'];
    if (sponsoredVisas.includes(visaType)) {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        cosFields.classList.add('show');
        return;
    }

    if (visaType && visaType !== '') {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
    }
});

document.getElementById('right_to_work_checked').addEventListener('change', function() {
    const rtwFields = document.getElementById('rtwFields');
    rtwFields.classList.toggle('show', this.checked);
});

// Initialize on load
document.getElementById('visa_type').dispatchEvent(new Event('change'));
document.getElementById('nationality').dispatchEvent(new Event('input'));
if (document.getElementById('right_to_work_checked').checked) {
    document.getElementById('right_to_work_checked').dispatchEvent(new Event('change'));
}
</script>

<?php include '../../includes/footer.php'; ?>
