<?php
session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();
$page_title = 'Add New Worker';
$current_page = 'workers';
$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $employee_number = trim($_POST['employee_number'] ?? '');
    $first_name = trim($_POST['first_name'] ?? '');
    $middle_name = trim($_POST['middle_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $nationality = trim($_POST['nationality'] ?? '');

    // Validate required fields
    if (empty($employee_number)) $errors[] = "Employee number required";
    if (empty($first_name)) $errors[] = "First name required";
    if (empty($last_name)) $errors[] = "Last name required";
    if (empty($nationality)) $errors[] = "Nationality required";

    // Nationality-based validation
    $is_british = (strtolower($nationality) === 'british' || strtolower($nationality) === 'irish');

    if (!$is_british) {
        if (empty($_POST['visa_type'])) $errors[] = "Visa/immigration status required for non-British/Irish nationals";
    }

    // Check duplicate employee number
    if (!empty($employee_number) && empty($errors)) {
        $check = $pdo->prepare("SELECT id FROM scms_sponsored_workers WHERE employee_number = ?");
        $check->execute([$employee_number]);
        if ($check->fetch()) $errors[] = "Employee number already exists";
    }

    if (empty($errors)) {
        // Build full name
        $full_name = $first_name;
        if (!empty($middle_name)) $full_name .= ' ' . $middle_name;
        $full_name .= ' ' . $last_name;

        try {
            $stmt = $pdo->prepare("INSERT INTO scms_sponsored_workers
                (employee_number, full_name, date_of_birth, gender, place_of_birth, nationality,
                 passport_number, passport_issue_date, passport_expiry, passport_country,
                 ni_number, driving_license,
                 brp_number, brp_expiry, share_code, share_code_expiry,
                 visa_type, visa_number, visa_issue_date, visa_expiry, visa_status,
                 indefinite_leave, settled_status, pre_settled_status,
                 cos_number, cos_issue_date, cos_start_date, cos_end_date,
                 right_to_work_checked, right_to_work_check_date, right_to_work_expiry,
                 email, personal_email, phone, mobile_phone,
                 address, city, postcode, country,
                 emergency_contact, emergency_phone, emergency_contact_relationship, emergency_contact_address,
                 secondary_emergency_contact, secondary_emergency_phone, secondary_emergency_relationship,
                 job_title, department, line_manager, contract_type,
                 start_date, probation_end_date, annual_salary, hourly_rate, work_hours_per_week,
                 bank_name, bank_account_number, bank_sort_code, tax_code,
                 notes, status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())");

            $stmt->execute([
                $employee_number,
                $full_name,
                $_POST['date_of_birth'] ?: null,
                $_POST['gender'] ?: null,
                $_POST['place_of_birth'] ?: null,
                $nationality,
                $_POST['passport_number'] ?: null,
                $_POST['passport_issue_date'] ?: null,
                $_POST['passport_expiry'] ?: null,
                $_POST['passport_country'] ?: null,
                $_POST['ni_number'] ?: null,
                $_POST['driving_license'] ?: null,
                $_POST['brp_number'] ?: null,
                $_POST['brp_expiry'] ?: null,
                $_POST['share_code'] ?: null,
                $_POST['share_code_expiry'] ?: null,
                $_POST['visa_type'] ?: null,
                $_POST['visa_number'] ?: null,
                $_POST['visa_issue_date'] ?: null,
                $_POST['visa_expiry'] ?: null,
                $_POST['visa_status'] ?: null,
                isset($_POST['indefinite_leave']) ? 1 : 0,
                isset($_POST['settled_status']) ? 1 : 0,
                isset($_POST['pre_settled_status']) ? 1 : 0,
                $_POST['cos_number'] ?: null,
                $_POST['cos_issue_date'] ?: null,
                $_POST['cos_start_date'] ?: null,
                $_POST['cos_end_date'] ?: null,
                isset($_POST['right_to_work_checked']) ? 1 : 0,
                $_POST['right_to_work_check_date'] ?: null,
                $_POST['right_to_work_expiry'] ?: null,
                $_POST['email'] ?: null,
                $_POST['personal_email'] ?: null,
                $_POST['phone'] ?: null,
                $_POST['mobile_phone'] ?: null,
                $_POST['address'] ?: null,
                $_POST['city'] ?: null,
                $_POST['postcode'] ?: null,
                $_POST['country'] ?: 'United Kingdom',
                $_POST['emergency_contact'] ?: null,
                $_POST['emergency_phone'] ?: null,
                $_POST['emergency_contact_relationship'] ?: null,
                $_POST['emergency_contact_address'] ?: null,
                $_POST['secondary_emergency_contact'] ?: null,
                $_POST['secondary_emergency_phone'] ?: null,
                $_POST['secondary_emergency_relationship'] ?: null,
                $_POST['job_title'] ?: null,
                $_POST['department'] ?: null,
                $_POST['line_manager'] ?: null,
                $_POST['contract_type'] ?: null,
                $_POST['start_date'] ?: null,
                $_POST['probation_end_date'] ?: null,
                $_POST['annual_salary'] ?: null,
                $_POST['hourly_rate'] ?: null,
                $_POST['work_hours_per_week'] ?: null,
                $_POST['bank_name'] ?: null,
                $_POST['bank_account_number'] ?: null,
                $_POST['bank_sort_code'] ?: null,
                $_POST['tax_code'] ?: null,
                $_POST['notes'] ?: null
            ]);

            header("Location: list.php?success=1");
            exit;

        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

include '../../includes/header.php';
?>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.conditional-field {
    display: none;
}

.conditional-field.show {
    display: block;
}

.info-badge {
    background: #e3f2fd;
    color: #1976d2;
    padding: 0.5rem 1rem;
    border-radius: 6px;
    font-size: 0.875rem;
    margin-bottom: 1rem;
}
</style>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title"><i class="bi bi-person-plus-fill"></i> Add New Worker</h1>
            <p class="welcome-subtitle">Complete HR and immigration information</p>
        </div>
        <div class="welcome-actions">
            <a href="list.php" class="btn btn-secondary"><i class="bi bi-arrow-left"></i> Back</a>
        </div>
    </div>

    <?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <ul class="mb-0">
            <?php foreach ($errors as $e): ?>
                <li><?php echo htmlspecialchars($e); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST" id="workerForm">
                <!-- Employee Number -->
                <div class="form-section">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Employee Number</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="employee_number"
                                   value="<?php echo htmlspecialchars($_POST['employee_number'] ?? ''); ?>" required>
                            <small class="text-muted">Unique identifier (e.g., EMP001, SW001)</small>
                        </div>
                    </div>
                </div>

                <!-- Personal Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-person"></i> Personal Information
                    </h6>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>First Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="first_name"
                                   value="<?php echo htmlspecialchars($_POST['first_name'] ?? ''); ?>" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Middle Name</label>
                            <input type="text" class="form-control" name="middle_name"
                                   value="<?php echo htmlspecialchars($_POST['middle_name'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Last Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="last_name"
                                   value="<?php echo htmlspecialchars($_POST['last_name'] ?? ''); ?>" required>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Date of Birth</label>
                            <input type="date" class="form-control" name="date_of_birth"
                                   value="<?php echo htmlspecialchars($_POST['date_of_birth'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Gender</label>
                            <select class="form-select" name="gender">
                                <option value="">Select...</option>
                                <option value="Male">Male</option>
                                <option value="Female">Female</option>
                                <option value="Other">Other</option>
                                <option value="Prefer not to say">Prefer not to say</option>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Place of Birth</label>
                            <input type="text" class="form-control" name="place_of_birth"
                                   placeholder="City, Country"
                                   value="<?php echo htmlspecialchars($_POST['place_of_birth'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Nationality</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="nationality" id="nationality"
                                   placeholder="e.g., British, Irish, Indian, Nigerian"
                                   value="<?php echo htmlspecialchars($_POST['nationality'] ?? ''); ?>" required>
                            <small class="text-muted">Required for all workers</small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">National Insurance Number</label>
                            <input type="text" class="form-control" name="ni_number"
                                   placeholder="AB123456C" pattern="[A-Z]{2}[0-9]{6}[A-D]"
                                   value="<?php echo htmlspecialchars($_POST['ni_number'] ?? ''); ?>">
                            <small class="text-muted">Format: 2 letters, 6 numbers, 1 letter</small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Driving License Number</label>
                            <input type="text" class="form-control" name="driving_license"
                                   value="<?php echo htmlspecialchars($_POST['driving_license'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Passport Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-passport"></i> Passport Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Number</label>
                            <input type="text" class="form-control" name="passport_number"
                                   value="<?php echo htmlspecialchars($_POST['passport_number'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Country of Issue</label>
                            <input type="text" class="form-control" name="passport_country"
                                   value="<?php echo htmlspecialchars($_POST['passport_country'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Issue Date</label>
                            <input type="date" class="form-control" name="passport_issue_date"
                                   value="<?php echo htmlspecialchars($_POST['passport_issue_date'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Expiry Date</label>
                            <input type="date" class="form-control" name="passport_expiry"
                                   value="<?php echo htmlspecialchars($_POST['passport_expiry'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Immigration Status (Non-British/Irish) -->
                <div class="form-section" id="immigrationSection">
                    <h6 class="form-section-title">
                        <i class="bi bi-globe-europe-africa"></i> Immigration Status
                    </h6>

                    <div class="info-badge mb-3" id="britishNotice" style="display: none;">
                        <i class="bi bi-info-circle"></i> British and Irish nationals have automatic right to work in the UK. Immigration details are optional.
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label"><strong>Visa/Immigration Status</strong> <span class="text-danger" id="visaRequired">*</span></label>
                            <select class="form-select" name="visa_type" id="visa_type">
                                <option value="">Select immigration status...</option>
                                <optgroup label="No Visa Required">
                                    <option value="British Citizen">British Citizen</option>
                                    <option value="Irish Citizen">Irish Citizen</option>
                                    <option value="Indefinite Leave to Remain">Indefinite Leave to Remain (ILR)</option>
                                </optgroup>
                                <optgroup label="EU Settlement Scheme">
                                    <option value="EU Settled Status">EU Settled Status</option>
                                    <option value="EU Pre-Settled Status">EU Pre-Settled Status</option>
                                </optgroup>
                                <optgroup label="Skilled Worker Route">
                                    <option value="Skilled Worker">Skilled Worker Visa</option>
                                    <option value="Health and Care Worker">Health and Care Worker Visa</option>
                                    <option value="Senior or Specialist Worker">Senior or Specialist Worker Visa</option>
                                </optgroup>
                                <optgroup label="Student & Graduate">
                                    <option value="Graduate">Graduate Visa</option>
                                    <option value="Student">Student Visa</option>
                                </optgroup>
                                <optgroup label="Family & Dependant">
                                    <option value="Spouse Visa">Spouse/Partner Visa</option>
                                    <option value="Dependant">Dependant Visa</option>
                                    <option value="Family">Family Visa</option>
                                </optgroup>
                                <optgroup label="Other Routes">
                                    <option value="Global Talent">Global Talent Visa</option>
                                    <option value="Innovator Founder">Innovator Founder Visa</option>
                                    <option value="Start-up">Start-up Visa</option>
                                    <option value="Minister of Religion">Minister of Religion Visa</option>
                                    <option value="Sportsperson">Sportsperson Visa</option>
                                    <option value="Youth Mobility">Youth Mobility Scheme</option>
                                    <option value="Other">Other</option>
                                </optgroup>
                            </select>
                        </div>
                    </div>

                    <!-- Settlement Status Checkboxes -->
                    <div class="row mb-3" id="settlementFields" style="display: none;">
                        <div class="col-md-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="settled_status" id="settled_status">
                                <label class="form-check-label" for="settled_status">
                                    EU Settled Status
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="pre_settled_status" id="pre_settled_status">
                                <label class="form-check-label" for="pre_settled_status">
                                    EU Pre-Settled Status
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="indefinite_leave" id="indefinite_leave">
                                <label class="form-check-label" for="indefinite_leave">
                                    Indefinite Leave to Remain (ILR)
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- BRP Information -->
                    <div class="conditional-field" id="brpFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Number</label>
                                <input type="text" class="form-control" name="brp_number"
                                       placeholder="12-digit number"
                                       value="<?php echo htmlspecialchars($_POST['brp_number'] ?? ''); ?>">
                                <small class="text-muted">Biometric Residence Permit</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Expiry Date</label>
                                <input type="date" class="form-control" name="brp_expiry"
                                       value="<?php echo htmlspecialchars($_POST['brp_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Share Code -->
                    <div class="conditional-field" id="shareCodeFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code</label>
                                <input type="text" class="form-control" name="share_code"
                                       placeholder="9-character code"
                                       value="<?php echo htmlspecialchars($_POST['share_code'] ?? ''); ?>">
                                <small class="text-muted">From gov.uk Right to Work check</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code Expiry</label>
                                <input type="date" class="form-control" name="share_code_expiry"
                                       value="<?php echo htmlspecialchars($_POST['share_code_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Visa Details (for visa holders) -->
                    <div class="conditional-field" id="visaFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Number</label>
                                <input type="text" class="form-control" name="visa_number"
                                       value="<?php echo htmlspecialchars($_POST['visa_number'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Status</label>
                                <select class="form-select" name="visa_status">
                                    <option value="">Select...</option>
                                    <option value="Valid">Valid</option>
                                    <option value="Pending">Pending</option>
                                    <option value="Expired">Expired</option>
                                    <option value="Cancelled">Cancelled</option>
                                </select>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Issue Date</label>
                                <input type="date" class="form-control" name="visa_issue_date"
                                       value="<?php echo htmlspecialchars($_POST['visa_issue_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Expiry Date</label>
                                <input type="date" class="form-control" name="visa_expiry"
                                       value="<?php echo htmlspecialchars($_POST['visa_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Certificate of Sponsorship (for sponsored workers) -->
                    <div class="conditional-field" id="cosFields">
                        <h6 class="mt-3 mb-3"><i class="bi bi-file-earmark-text"></i> Certificate of Sponsorship</h6>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Number</label>
                                <input type="text" class="form-control" name="cos_number"
                                       placeholder="e.g., ABC123456789"
                                       value="<?php echo htmlspecialchars($_POST['cos_number'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Issue Date</label>
                                <input type="date" class="form-control" name="cos_issue_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_issue_date'] ?? ''); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Start Date</label>
                                <input type="date" class="form-control" name="cos_start_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_start_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS End Date</label>
                                <input type="date" class="form-control" name="cos_end_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_end_date'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Right to Work Check -->
                    <div class="row mt-3">
                        <div class="col-md-12">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" name="right_to_work_checked" id="right_to_work_checked">
                                <label class="form-check-label" for="right_to_work_checked">
                                    <strong>Right to Work Check Completed</strong>
                                </label>
                            </div>
                        </div>
                    </div>

                    <div class="conditional-field" id="rtwFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check Date</label>
                                <input type="date" class="form-control" name="right_to_work_check_date"
                                       value="<?php echo htmlspecialchars($_POST['right_to_work_check_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Next Check Due</label>
                                <input type="date" class="form-control" name="right_to_work_expiry"
                                       value="<?php echo htmlspecialchars($_POST['right_to_work_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Contact Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-envelope"></i> Contact Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Email</label>
                            <input type="email" class="form-control" name="email"
                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Personal Email</label>
                            <input type="email" class="form-control" name="personal_email"
                                   value="<?php echo htmlspecialchars($_POST['personal_email'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Phone</label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Mobile Phone</label>
                            <input type="tel" class="form-control" name="mobile_phone"
                                   value="<?php echo htmlspecialchars($_POST['mobile_phone'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Address</label>
                            <textarea class="form-control" name="address" rows="2"><?php echo htmlspecialchars($_POST['address'] ?? ''); ?></textarea>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">City</label>
                            <input type="text" class="form-control" name="city"
                                   value="<?php echo htmlspecialchars($_POST['city'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Postcode</label>
                            <input type="text" class="form-control" name="postcode"
                                   value="<?php echo htmlspecialchars($_POST['postcode'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Country</label>
                            <input type="text" class="form-control" name="country"
                                   value="<?php echo htmlspecialchars($_POST['country'] ?? 'United Kingdom'); ?>">
                        </div>
                    </div>
                </div>

                <!-- Employment Details -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-briefcase"></i> Employment Details
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Job Title</label>
                            <input type="text" class="form-control" name="job_title"
                                   value="<?php echo htmlspecialchars($_POST['job_title'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Department</label>
                            <input type="text" class="form-control" name="department"
                                   value="<?php echo htmlspecialchars($_POST['department'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Line Manager</label>
                            <input type="text" class="form-control" name="line_manager"
                                   value="<?php echo htmlspecialchars($_POST['line_manager'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Contract Type</label>
                            <select class="form-select" name="contract_type">
                                <option value="">Select...</option>
                                <option value="Full-time">Full-time</option>
                                <option value="Part-time">Part-time</option>
                                <option value="Fixed Term">Fixed Term</option>
                                <option value="Temporary">Temporary</option>
                                <option value="Zero Hours">Zero Hours</option>
                            </select>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Start Date</label>
                            <input type="date" class="form-control" name="start_date"
                                   value="<?php echo htmlspecialchars($_POST['start_date'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Probation End Date</label>
                            <input type="date" class="form-control" name="probation_end_date"
                                   value="<?php echo htmlspecialchars($_POST['probation_end_date'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Annual Salary (£)</label>
                            <input type="number" class="form-control" name="annual_salary" step="0.01"
                                   value="<?php echo htmlspecialchars($_POST['annual_salary'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hourly Rate (£)</label>
                            <input type="number" class="form-control" name="hourly_rate" step="0.01"
                                   value="<?php echo htmlspecialchars($_POST['hourly_rate'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hours per Week</label>
                            <input type="number" class="form-control" name="work_hours_per_week" step="0.5"
                                   value="<?php echo htmlspecialchars($_POST['work_hours_per_week'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Banking & Tax -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-bank"></i> Banking & Tax Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Bank Name</label>
                            <input type="text" class="form-control" name="bank_name"
                                   value="<?php echo htmlspecialchars($_POST['bank_name'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Tax Code</label>
                            <input type="text" class="form-control" name="tax_code"
                                   placeholder="e.g., 1257L"
                                   value="<?php echo htmlspecialchars($_POST['tax_code'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Account Number</label>
                            <input type="text" class="form-control" name="bank_account_number"
                                   pattern="[0-9]{8}"
                                   value="<?php echo htmlspecialchars($_POST['bank_account_number'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Sort Code</label>
                            <input type="text" class="form-control" name="bank_sort_code"
                                   placeholder="12-34-56" pattern="[0-9]{2}-[0-9]{2}-[0-9]{2}"
                                   value="<?php echo htmlspecialchars($_POST['bank_sort_code'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Emergency Contact -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-telephone-plus"></i> Emergency Contact
                    </h6>

                    <h6 class="mb-3">Primary Contact</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="emergency_contact"
                                   value="<?php echo htmlspecialchars($_POST['emergency_contact'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="emergency_phone"
                                   value="<?php echo htmlspecialchars($_POST['emergency_phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="emergency_contact_relationship"
                                   placeholder="e.g., Spouse, Parent, Sibling"
                                   value="<?php echo htmlspecialchars($_POST['emergency_contact_relationship'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Contact Address</label>
                            <textarea class="form-control" name="emergency_contact_address" rows="2"><?php echo htmlspecialchars($_POST['emergency_contact_address'] ?? ''); ?></textarea>
                        </div>
                    </div>

                    <h6 class="mb-3 mt-4">Secondary Contact (Optional)</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="secondary_emergency_contact"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_contact'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="secondary_emergency_phone"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="secondary_emergency_relationship"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_relationship'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Additional Notes -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-journal-text"></i> Additional Notes
                    </h6>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" rows="4"><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                            <small class="text-muted">Any additional information about the worker</small>
                        </div>
                    </div>
                </div>

                <div class="text-end mt-4">
                    <a href="list.php" class="btn btn-secondary me-2"><i class="bi bi-x-circle"></i> Cancel</a>
                    <button type="submit" class="btn btn-primary"><i class="bi bi-check-circle"></i> Add Worker</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
// Nationality change handler
document.getElementById('nationality').addEventListener('input', function() {
    const nationality = this.value.toLowerCase().trim();
    const isBritishIrish = nationality === 'british' || nationality === 'irish';
    const britishNotice = document.getElementById('britishNotice');
    const visaRequired = document.getElementById('visaRequired');

    if (isBritishIrish) {
        britishNotice.style.display = 'block';
        visaRequired.style.display = 'none';
        document.getElementById('visa_type').required = false;
    } else {
        britishNotice.style.display = 'none';
        visaRequired.style.display = 'inline';
        document.getElementById('visa_type').required = true;
    }
});

// Visa type change handler
document.getElementById('visa_type').addEventListener('change', function() {
    const visaType = this.value;
    const brpFields = document.getElementById('brpFields');
    const shareCodeFields = document.getElementById('shareCodeFields');
    const visaFields = document.getElementById('visaFields');
    const cosFields = document.getElementById('cosFields');

    // Hide all conditional fields
    brpFields.classList.remove('show');
    shareCodeFields.classList.remove('show');
    visaFields.classList.remove('show');
    cosFields.classList.remove('show');

    // Visa types that don't need additional immigration fields
    const noVisaNeeded = ['British Citizen', 'Irish Citizen', 'Indefinite Leave to Remain'];

    if (noVisaNeeded.includes(visaType)) {
        // Show only BRP for ILR
        if (visaType === 'Indefinite Leave to Remain') {
            brpFields.classList.add('show');
        }
        return;
    }

    // EU Settlement Scheme
    if (visaType === 'EU Settled Status' || visaType === 'EU Pre-Settled Status') {
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        return;
    }

    // Sponsored worker visas
    const sponsoredVisas = ['Skilled Worker', 'Health and Care Worker', 'Senior or Specialist Worker'];
    if (sponsoredVisas.includes(visaType)) {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        cosFields.classList.add('show');
        return;
    }

    // All other visas show standard fields
    if (visaType && visaType !== '') {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
    }
});

// Right to Work check handler
document.getElementById('right_to_work_checked').addEventListener('change', function() {
    const rtwFields = document.getElementById('rtwFields');
    if (this.checked) {
        rtwFields.classList.add('show');
    } else {
        rtwFields.classList.remove('show');
    }
});

// Restore state if form was submitted with errors
<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['visa_type'])): ?>
document.getElementById('visa_type').dispatchEvent(new Event('change'));
<?php endif; ?>

<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['nationality'])): ?>
document.getElementById('nationality').dispatchEvent(new Event('input'));
<?php endif; ?>

<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['right_to_work_checked'])): ?>
document.getElementById('right_to_work_checked').checked = true;
document.getElementById('right_to_work_checked').dispatchEvent(new Event('change'));
<?php endif; ?>
</script>

<?php include '../../includes/footer.php'; ?>
