<?php
/**
 * Delete Worker (Soft Delete with PIN Verification)
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(90); // Only high-level users can delete

$page_title = 'Delete Worker';
$current_page = 'workers';

$worker_id = $_GET['id'] ?? '';

if (empty($worker_id)) {
    header('Location: list.php');
    exit;
}

// Fetch worker details
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ? AND deleted_at IS NULL");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$worker) {
    $_SESSION['error'] = 'Worker not found or already deleted';
    header('Location: list.php');
    exit;
}

$errors = [];

// Get current user permission level
$current_user = get_logged_in_user();
$is_system_admin = $current_user['permission_level'] >= 100;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $pin = $_POST['pin'] ?? '';
    $delete_reason = trim($_POST['delete_reason'] ?? '');
    $delete_type = $_POST['delete_type'] ?? 'soft'; // 'soft' or 'permanent'

    // Only system admins can perform permanent delete
    if ($delete_type === 'permanent' && !$is_system_admin) {
        $errors[] = "You do not have permission to permanently delete workers";
    }

    // Validate PIN
    if (empty($pin)) {
        $errors[] = "PIN is required for deletion";
    } else {
        // Verify PIN against current user
        $stmt = $pdo->prepare("SELECT pin_hash FROM " . DB_PREFIX . "users WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user || !password_verify($pin, $user['pin_hash'])) {
            $errors[] = "Invalid PIN";
        }
    }

    // Validate reason
    if (empty($delete_reason)) {
        $errors[] = "Deletion reason is required";
    }

    // If no errors, perform delete
    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            if ($delete_type === 'permanent') {
                // Permanent delete - remove from database
                $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
                $stmt->execute([$worker_id]);

                $action_message = "Permanently deleted worker: {$worker['full_name']} (ID: {$worker_id})";
                $success_message = 'Worker permanently deleted from database';
            } else {
                // Soft delete worker
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "sponsored_workers
                    SET deleted_at = NOW(),
                        deleted_by = ?,
                        delete_reason = ?,
                        status = 'deleted'
                    WHERE id = ?
                ");
                $stmt->execute([$_SESSION['user_id'], $delete_reason, $worker_id]);

                $action_message = "Soft deleted worker: {$worker['full_name']} (ID: {$worker_id})";
                $success_message = 'Worker deleted successfully';
            }

            // Log the deletion
            $stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "audit_logs
                (user_id, action, details, created_at)
                VALUES (?, ?, ?, NOW())
            ");
            $stmt->execute([
                $_SESSION['user_id'],
                'Worker Deleted',
                $action_message . ". Reason: {$delete_reason}"
            ]);

            $pdo->commit();

            $_SESSION['success'] = $success_message;
            header('Location: list.php');
            exit;

        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Delete failed: " . $e->getMessage();
        }
    }
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-6">
            <div class="modern-card">
                <div class="card-header-modern bg-danger text-white">
                    <h5 class="card-title-modern mb-0">
                        <i class="bi bi-exclamation-triangle"></i> Delete Worker
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($is_system_admin): ?>
                        <div class="alert alert-danger">
                            <i class="bi bi-shield-exclamation"></i>
                            <strong>System Administrator:</strong> You have the option to permanently delete this record or soft-delete it.
                        </div>

                        <div class="mb-4">
                            <label class="form-label"><strong>Deletion Type</strong> <span class="text-danger">*</span></label>
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="radio" name="delete_type" value="soft" id="deleteSoft" checked>
                                <label class="form-check-label" for="deleteSoft">
                                    <strong>Soft Delete (Recommended)</strong><br>
                                    <small class="text-muted">Mark as deleted but keep data for audit purposes</small>
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="radio" name="delete_type" value="permanent" id="deletePermanent">
                                <label class="form-check-label" for="deletePermanent">
                                    <strong class="text-danger">Permanent Delete</strong><br>
                                    <small class="text-danger">Remove from database completely (cannot be undone)</small>
                                </label>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle-fill"></i>
                            <strong>Warning:</strong> This action will soft-delete the worker record. The data will be marked as deleted but preserved for audit purposes.
                        </div>
                        <input type="hidden" name="delete_type" value="soft">
                    <?php endif; ?>

                    <div class="mb-4 p-3 bg-light rounded">
                        <h6 class="mb-3">Worker Details:</h6>
                        <div class="row">
                            <div class="col-6">
                                <strong>Name:</strong><br>
                                <?php echo htmlspecialchars($worker['full_name']); ?>
                            </div>
                            <div class="col-6">
                                <strong>Employee #:</strong><br>
                                <?php echo htmlspecialchars($worker['employee_number']); ?>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-6">
                                <strong>Job Title:</strong><br>
                                <?php echo htmlspecialchars($worker['job_title'] ?? '-'); ?>
                            </div>
                            <div class="col-6">
                                <strong>Department:</strong><br>
                                <?php echo htmlspecialchars($worker['department'] ?? '-'); ?>
                            </div>
                        </div>
                    </div>

                    <form method="POST">
                        <div class="mb-3">
                            <label class="form-label">
                                <strong>Deletion Reason</strong> <span class="text-danger">*</span>
                            </label>
                            <textarea name="delete_reason" class="form-control" rows="4" required
                                      placeholder="Please provide a detailed reason for deletion..."></textarea>
                            <small class="text-muted">This will be recorded in the audit log</small>
                        </div>

                        <div class="mb-4">
                            <label class="form-label">
                                <strong>Your PIN</strong> <span class="text-danger">*</span>
                            </label>
                            <input type="password" name="pin" class="form-control" required
                                   placeholder="Enter your PIN to confirm deletion" maxlength="6">
                            <small class="text-muted">Enter your 4-6 digit PIN to authorize this deletion</small>
                        </div>

                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i>
                            <strong>What happens when you delete:</strong>
                            <ul class="mb-0 mt-2">
                                <li>Worker record is marked as deleted (soft delete)</li>
                                <li>Worker will not appear in active lists</li>
                                <li>All historical data is preserved</li>
                                <li>Deletion is logged in audit trail</li>
                                <li>Related documents remain accessible</li>
                            </ul>
                        </div>

                        <hr>

                        <div class="d-flex justify-content-between">
                            <a href="view.php?id=<?php echo $worker_id; ?>" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-danger">
                                <i class="bi bi-trash"></i> Delete Worker
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <div class="modern-card mt-3">
                <div class="card-header-modern">
                    <h6 class="card-title-modern mb-0">
                        <i class="bi bi-info-circle"></i> Alternative Actions
                    </h6>
                </div>
                <div class="card-body-modern">
                    <p class="mb-2">Instead of deleting, you might want to:</p>
                    <ul>
                        <li><strong>Change Status:</strong> Set worker to "inactive" or "suspended" instead</li>
                        <li><strong>Set End Date:</strong> Add an employment end date to mark departure</li>
                        <li><strong>Archive:</strong> Keep record active but mark as departed</li>
                    </ul>
                    <a href="edit.php?id=<?php echo $worker_id; ?>" class="btn btn-outline-primary btn-sm">
                        <i class="bi bi-pencil"></i> Edit Worker Instead
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include '../../includes/footer.php'; ?>
