<?php
/**
 * Workers Module - Delete Worker
 * Role-based deletion:
 * - Permission 70-89: Can soft delete only (sets status to 'deleted')
 * - Permission 90-99: Can soft delete + restore
 * - Permission 100+: Can soft delete, restore, or PERMANENT delete (requires PIN)
 */

// Helper function to handle redirects
function safe_redirect($url) {
    if (defined('MODULE_EMBEDDED')) {
        echo '<script>window.location.href = "' . htmlspecialchars($url) . '";</script>';
        echo '<noscript><meta http-equiv="refresh" content="0;url=' . htmlspecialchars($url) . '"></noscript>';
    } else {
        header('Location: ' . $url);
    }
    exit;
}

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        safe_redirect('../../login.php');
    }
} else {
    if (!isset($current_user)) {
        $current_user = get_logged_in_user();
    }
}

// Check minimum permission - delete requires level 70
if (!isset($current_user['permission_level']) || $current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete workers.';
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
}

$worker_id = (int)($_POST['id'] ?? 0);
$delete_type = $_POST['delete_type'] ?? 'soft'; // 'soft' or 'permanent'
$pin = $_POST['pin'] ?? '';

if ($worker_id <= 0) {
    $_SESSION['error'] = 'Invalid worker ID.';
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if worker exists and get current status
    $stmt = $pdo->prepare("SELECT id, full_name, employee_number, status FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
    $stmt->execute([$worker_id]);
    $worker = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$worker) {
        $_SESSION['error'] = 'Worker not found.';
        safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
    }

    // Determine what type of delete to perform based on role and request
    $permission_level = $current_user['permission_level'];
    $current_status = $worker['status'];

    // CASE 1: Non-admin users (70-89) - Can only soft delete active workers
    if ($permission_level >= 70 && $permission_level < 90) {
        if ($current_status === 'deleted') {
            $_SESSION['error'] = 'This worker is already deleted. Only admins can permanently remove deleted workers.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php'));
            exit;
        }

        // Soft delete
        $stmt = $pdo->prepare("
            UPDATE " . DB_PREFIX . "sponsored_workers
            SET status = 'deleted',
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$worker_id]);
        $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been deleted.';
    }

    // CASE 2: Senior users (90-99) - Can soft delete and restore
    elseif ($permission_level >= 90 && $permission_level < 100) {
        if ($delete_type === 'restore' || $current_status === 'deleted') {
            // Restore deleted worker
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "sponsored_workers
                SET status = 'active',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$worker_id]);
            $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been restored.';
        } else {
            // Soft delete
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "sponsored_workers
                SET status = 'deleted',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$worker_id]);
            $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been deleted.';
        }
    }

    // CASE 3: Admins (100+) - Can do everything including permanent delete
    elseif ($permission_level >= 100) {
        // If explicitly requesting restore
        if ($delete_type === 'restore') {
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "sponsored_workers
                SET status = 'active',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$worker_id]);
            $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been restored.';
        }
        // If requesting permanent delete
        elseif ($delete_type === 'permanent') {
            // REQUIRE PIN for permanent delete
            if (empty($pin)) {
                $_SESSION['error'] = 'PIN is required for permanent deletion.';
                safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
            }

            // Verify PIN (check against user's PIN)
            if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
                $_SESSION['error'] = 'Invalid PIN. Permanent deletion cancelled.';
                safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
            }

            // Log the permanent deletion for audit trail
            $audit_stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "audit_log
                (user_id, action, table_name, record_id, details, ip_address, created_at)
                VALUES (?, 'PERMANENT_DELETE', 'sponsored_workers', ?, ?, ?, NOW())
            ");
            $details = json_encode([
                'worker_name' => $worker['full_name'],
                'employee_number' => $worker['employee_number'],
                'reason' => 'Admin permanent deletion with PIN verification'
            ]);
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $audit_stmt->execute([$current_user['id'], $worker_id, $details, $ip]);

            // PERMANENT DELETE - Remove from database
            $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
            $stmt->execute([$worker_id]);

            $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" (ID: ' . $worker['employee_number'] . ') has been PERMANENTLY DELETED from the database. This action cannot be undone.';
        } else {
            // Soft delete or restore
            if ($current_status === 'deleted') {
                // Restore
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "sponsored_workers
                    SET status = 'active',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$worker_id]);
                $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been restored.';
            } else {
                // Soft delete
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "sponsored_workers
                    SET status = 'deleted',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$worker_id]);
                $_SESSION['success'] = 'Worker "' . htmlspecialchars($worker['full_name']) . '" has been deleted (soft delete).';
            }
        }
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting worker: ' . $e->getMessage();
}

// Redirect back to list
safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/workers/list' : 'list.php');
?>
