-- ============================================================================
-- SCMS Workers Module - Database Schema
-- UK Home Office Sponsored Workers Compliance System
-- ============================================================================

-- Main Sponsored Workers Table
-- Stores core worker information required by UK Home Office
CREATE TABLE IF NOT EXISTS `scms_sponsored_workers` (
  `id` int(11) NOT NULL AUTO_INCREMENT,

  -- Employee Information
  `employee_number` varchar(50) NOT NULL UNIQUE,
  `title` enum('Mr','Mrs','Miss','Ms','Dr','Prof','Rev','Other') DEFAULT 'Mr',
  `first_name` varchar(100) NOT NULL,
  `middle_names` varchar(200) DEFAULT NULL,
  `last_name` varchar(100) NOT NULL,
  `preferred_name` varchar(100) DEFAULT NULL,
  `date_of_birth` date NOT NULL,
  `gender` enum('male','female','other','prefer_not_to_say') DEFAULT NULL,
  `nationality` varchar(100) DEFAULT NULL,
  `place_of_birth` varchar(200) DEFAULT NULL,

  -- Contact Information
  `email` varchar(200) DEFAULT NULL,
  `phone` varchar(50) DEFAULT NULL,
  `mobile` varchar(50) DEFAULT NULL,
  `emergency_contact_name` varchar(200) DEFAULT NULL,
  `emergency_contact_phone` varchar(50) DEFAULT NULL,
  `emergency_contact_relationship` varchar(100) DEFAULT NULL,

  -- UK Address (Current Residence - REQUIRED by Home Office)
  `address_line1` varchar(255) DEFAULT NULL,
  `address_line2` varchar(255) DEFAULT NULL,
  `city` varchar(100) DEFAULT NULL,
  `county` varchar(100) DEFAULT NULL,
  `postcode` varchar(20) DEFAULT NULL,
  `country` varchar(100) DEFAULT 'United Kingdom',

  -- Passport & Immigration Details (CRITICAL for Home Office)
  `passport_number` varchar(50) DEFAULT NULL,
  `passport_country` varchar(100) DEFAULT NULL,
  `passport_issue_date` date DEFAULT NULL,
  `passport_expiry_date` date DEFAULT NULL,

  -- Visa & Immigration Status (REQUIRED by Home Office)
  `visa_status` enum('skilled_worker','health_care','graduate','senior_specialist','global_talent','indefinite_leave','settled_status','pre_settled','other') DEFAULT 'skilled_worker',
  `visa_number` varchar(100) DEFAULT NULL COMMENT 'BRP/eVisa number',
  `visa_issue_date` date DEFAULT NULL,
  `visa_expiry_date` date DEFAULT NULL,
  `immigration_status` enum('sponsored','settled','pre_settled','indefinite_leave') DEFAULT 'sponsored',

  -- Certificate of Sponsorship (CoS) - REQUIRED for sponsored workers
  `cos_number` varchar(50) DEFAULT NULL COMMENT 'Certificate of Sponsorship number',
  `cos_issue_date` date DEFAULT NULL,
  `cos_start_date` date DEFAULT NULL COMMENT 'Employment start date on CoS',
  `cos_job_title` varchar(255) DEFAULT NULL,
  `cos_soc_code` varchar(20) DEFAULT NULL COMMENT 'Standard Occupational Classification code',
  `cos_salary` decimal(10,2) DEFAULT NULL COMMENT 'Salary stated on CoS',

  -- Right to Work Checks (UK Home Office requirement)
  `rtw_check_date` date DEFAULT NULL COMMENT 'Last Right to Work check date',
  `rtw_checked_by` int(11) DEFAULT NULL COMMENT 'User ID who performed check',
  `rtw_expiry_date` date DEFAULT NULL COMMENT 'When next RTW check is due',
  `rtw_documents_verified` tinyint(1) DEFAULT 0,
  `share_code` varchar(20) DEFAULT NULL COMMENT 'Home Office online RTW share code',
  `share_code_expiry` datetime DEFAULT NULL,

  -- Employment Details
  `job_title` varchar(255) DEFAULT NULL,
  `department` varchar(200) DEFAULT NULL,
  `start_date` date DEFAULT NULL,
  `end_date` date DEFAULT NULL,
  `contract_type` enum('permanent','fixed_term','temporary','contractor') DEFAULT 'permanent',
  `employment_status` enum('full_time','part_time','casual') DEFAULT 'full_time',
  `working_hours` decimal(5,2) DEFAULT 37.50 COMMENT 'Hours per week',
  `salary` decimal(10,2) DEFAULT NULL,
  `salary_period` enum('hourly','weekly','monthly','annually') DEFAULT 'annually',

  -- UK Compliance
  `ni_number` varchar(13) DEFAULT NULL COMMENT 'National Insurance Number',
  `tax_code` varchar(20) DEFAULT NULL,
  `pension_opt_in` tinyint(1) DEFAULT 1,
  `student_loan` enum('none','plan_1','plan_2','plan_4','postgrad') DEFAULT 'none',

  -- Bank Details for Payroll
  `bank_name` varchar(200) DEFAULT NULL,
  `bank_account_number` varchar(20) DEFAULT NULL,
  `bank_sort_code` varchar(10) DEFAULT NULL,
  `bank_account_name` varchar(200) DEFAULT NULL,

  -- Work Location
  `work_location_id` int(11) DEFAULT NULL,
  `assigned_manager` int(11) DEFAULT NULL COMMENT 'Manager user ID',

  -- Home Office Reporting Requirements
  `absence_reported` tinyint(1) DEFAULT 0 COMMENT 'Absence over 10 days reported to HO',
  `last_attendance_date` date DEFAULT NULL,
  `no_show_count` int(11) DEFAULT 0 COMMENT 'Consecutive days of absence',
  `ho_report_date` date DEFAULT NULL COMMENT 'Last report to Home Office',
  `ho_report_reason` text DEFAULT NULL,

  -- Compliance Flags
  `compliance_alert` tinyint(1) DEFAULT 0,
  `compliance_notes` text DEFAULT NULL,
  `requires_attention` tinyint(1) DEFAULT 0,
  `attention_reason` varchar(500) DEFAULT NULL,

  -- Status
  `status` enum('active','suspended','terminated','on_leave','pending') DEFAULT 'active',
  `termination_date` date DEFAULT NULL,
  `termination_reason` text DEFAULT NULL,
  `exit_interview_completed` tinyint(1) DEFAULT 0,

  -- Photo/Documents
  `photo_path` varchar(500) DEFAULT NULL,
  `documents_folder` varchar(500) DEFAULT NULL,

  -- Audit Trail
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_by` int(11) DEFAULT NULL,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  `deleted_at` timestamp NULL DEFAULT NULL COMMENT 'Soft delete',

  PRIMARY KEY (`id`),
  KEY `idx_employee_number` (`employee_number`),
  KEY `idx_status` (`status`),
  KEY `idx_visa_expiry` (`visa_expiry_date`),
  KEY `idx_passport` (`passport_number`),
  KEY `idx_cos` (`cos_number`),
  KEY `idx_ni_number` (`ni_number`),
  KEY `idx_name` (`first_name`, `last_name`),
  KEY `idx_visa_status` (`visa_status`),
  KEY `idx_rtw_expiry` (`rtw_expiry_date`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker Documents Table
-- Stores scanned documents for each worker (passports, visas, certificates)
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_documents` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `document_type` enum('passport','visa','brp','cos','rtw_check','ni_letter','bank_statement','degree','qualification','contract','other') NOT NULL,
  `document_name` varchar(255) NOT NULL,
  `file_path` varchar(500) NOT NULL,
  `file_size` int(11) DEFAULT NULL COMMENT 'Size in bytes',
  `mime_type` varchar(100) DEFAULT NULL,
  `issue_date` date DEFAULT NULL,
  `expiry_date` date DEFAULT NULL,
  `document_number` varchar(100) DEFAULT NULL,
  `verified` tinyint(1) DEFAULT 0,
  `verified_by` int(11) DEFAULT NULL,
  `verified_date` timestamp NULL DEFAULT NULL,
  `notes` text DEFAULT NULL,
  `uploaded_by` int(11) NOT NULL,
  `uploaded_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `deleted_at` timestamp NULL DEFAULT NULL,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_document_type` (`document_type`),
  KEY `idx_expiry` (`expiry_date`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker CoS History Table
-- Track all Certificates of Sponsorship issued to a worker
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_cos_history` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `cos_number` varchar(50) NOT NULL,
  `cos_type` enum('defined','undefined') DEFAULT 'defined',
  `cos_route` varchar(100) DEFAULT 'Skilled Worker',
  `issue_date` date NOT NULL,
  `start_date` date NOT NULL,
  `end_date` date DEFAULT NULL,
  `job_title` varchar(255) NOT NULL,
  `soc_code` varchar(20) DEFAULT NULL,
  `annual_salary` decimal(10,2) NOT NULL,
  `hours_per_week` decimal(5,2) DEFAULT 37.50,
  `sponsor_license` varchar(50) DEFAULT NULL,
  `assigned_date` date DEFAULT NULL COMMENT 'Date CoS assigned to worker',
  `used_date` date DEFAULT NULL COMMENT 'Date worker used CoS for visa',
  `status` enum('assigned','used','withdrawn','expired') DEFAULT 'assigned',
  `notes` text DEFAULT NULL,
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_cos_number` (`cos_number`),
  KEY `idx_status` (`status`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker Contact Details History
-- Track address changes for Home Office reporting
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_contact_history` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `address_line1` varchar(255) NOT NULL,
  `address_line2` varchar(255) DEFAULT NULL,
  `city` varchar(100) NOT NULL,
  `county` varchar(100) DEFAULT NULL,
  `postcode` varchar(20) NOT NULL,
  `country` varchar(100) DEFAULT 'United Kingdom',
  `phone` varchar(50) DEFAULT NULL,
  `mobile` varchar(50) DEFAULT NULL,
  `email` varchar(200) DEFAULT NULL,
  `valid_from` date NOT NULL,
  `valid_to` date DEFAULT NULL,
  `is_current` tinyint(1) DEFAULT 1,
  `reported_to_ho` tinyint(1) DEFAULT 0 COMMENT 'Address change reported to Home Office',
  `ho_report_date` date DEFAULT NULL,
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_current` (`is_current`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker Employment History
-- Track job changes and promotions
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_employment_history` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `job_title` varchar(255) NOT NULL,
  `department` varchar(200) DEFAULT NULL,
  `soc_code` varchar(20) DEFAULT NULL,
  `salary` decimal(10,2) NOT NULL,
  `working_hours` decimal(5,2) DEFAULT NULL,
  `work_location_id` int(11) DEFAULT NULL,
  `start_date` date NOT NULL,
  `end_date` date DEFAULT NULL,
  `is_current` tinyint(1) DEFAULT 1,
  `change_reason` enum('new_hire','promotion','transfer','salary_increase','role_change','other') DEFAULT NULL,
  `reported_to_ho` tinyint(1) DEFAULT 0 COMMENT 'Change reported to Home Office',
  `ho_report_date` date DEFAULT NULL,
  `notes` text DEFAULT NULL,
  `created_by` int(11) DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_current` (`is_current`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker RTW Check History
-- Track all Right to Work checks performed
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_rtw_checks` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `check_date` date NOT NULL,
  `check_type` enum('initial','follow_up','annual') NOT NULL,
  `documents_seen` text NOT NULL COMMENT 'List of documents verified',
  `share_code_used` varchar(20) DEFAULT NULL,
  `online_check` tinyint(1) DEFAULT 0,
  `outcome` enum('pass','fail','refer') DEFAULT 'pass',
  `next_check_due` date DEFAULT NULL,
  `checked_by` int(11) NOT NULL COMMENT 'User ID who performed check',
  `notes` text DEFAULT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_next_check` (`next_check_due`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`checked_by`) REFERENCES `scms_users` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================================================
-- Worker Notes
-- Compliance notes and general comments about workers
-- ============================================================================
CREATE TABLE IF NOT EXISTS `scms_worker_notes` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL,
  `note_type` enum('general','compliance','hr','disciplinary','performance','ho_contact') DEFAULT 'general',
  `note` text NOT NULL,
  `is_confidential` tinyint(1) DEFAULT 0,
  `is_compliance_related` tinyint(1) DEFAULT 0,
  `created_by` int(11) NOT NULL,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

  PRIMARY KEY (`id`),
  KEY `idx_worker` (`worker_id`),
  KEY `idx_type` (`note_type`),
  FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE,
  FOREIGN KEY (`created_by`) REFERENCES `scms_users` (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
