<?php
/**
 * Create Worker Location Assignments Table
 * Allows workers to be assigned to multiple locations with different shift patterns
 * Run this file once to create the database table
 */

require_once '../../config/database.php';
require_once '../../includes/database.php';

$pdo = get_database_connection();

try {
    // Create worker location assignments table
    $sql = "CREATE TABLE IF NOT EXISTS " . DB_PREFIX . "worker_location_assignments (
        id INT(11) AUTO_INCREMENT PRIMARY KEY,
        worker_id INT(11) NOT NULL,
        location_id INT(11) NOT NULL,
        shift_pattern_id INT(11) DEFAULT NULL COMMENT 'Specific shift pattern from location_shift_patterns',
        days_of_week VARCHAR(50) DEFAULT NULL COMMENT 'Comma-separated: Mon,Tue,Wed,Thu,Fri,Sat,Sun',
        is_primary TINYINT(1) DEFAULT 0 COMMENT 'Primary/default location for this worker',
        start_date DATE DEFAULT NULL COMMENT 'When this assignment starts',
        end_date DATE DEFAULT NULL COMMENT 'When this assignment ends (NULL = ongoing)',
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_worker_id (worker_id),
        INDEX idx_location_id (location_id),
        INDEX idx_shift_pattern_id (shift_pattern_id),
        INDEX idx_is_primary (is_primary),
        INDEX idx_dates (start_date, end_date),
        UNIQUE KEY unique_worker_location_shift_days (worker_id, location_id, shift_pattern_id, days_of_week)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $pdo->exec($sql);
    echo "✓ Table " . DB_PREFIX . "worker_location_assignments created successfully<br>";

    // Migrate existing worker location data
    echo "<br>Migrating existing worker location assignments...<br>";

    $migrate_sql = "SELECT id, location_id FROM " . DB_PREFIX . "sponsored_workers WHERE location_id IS NOT NULL";
    $stmt = $pdo->query($migrate_sql);
    $workers_with_locations = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $migrated = 0;
    foreach ($workers_with_locations as $worker) {
        // Check if assignment already exists
        $check = $pdo->prepare("SELECT COUNT(*) FROM " . DB_PREFIX . "worker_location_assignments WHERE worker_id = ? AND location_id = ?");
        $check->execute([$worker['id'], $worker['location_id']]);

        if ($check->fetchColumn() == 0) {
            // Get default shift pattern for this location
            $shift_stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ? AND is_default = 1 LIMIT 1");
            $shift_stmt->execute([$worker['location_id']]);
            $default_shift = $shift_stmt->fetch(PDO::FETCH_ASSOC);

            // Insert the assignment
            $insert = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_location_assignments
                (worker_id, location_id, shift_pattern_id, days_of_week, is_primary, start_date)
                VALUES (?, ?, ?, 'Mon,Tue,Wed,Thu,Fri', 1, CURDATE())");

            $insert->execute([
                $worker['id'],
                $worker['location_id'],
                $default_shift['id'] ?? null
            ]);

            $migrated++;
        }
    }

    echo "✓ Migrated $migrated existing worker location assignments<br>";

    // Add location_context column to attendance table if not exists
    echo "<br>Updating attendance table...<br>";

    $attendance_cols = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "attendance LIKE 'location_id'");
    if ($attendance_cols->rowCount() == 0) {
        $pdo->exec("ALTER TABLE " . DB_PREFIX . "attendance
            ADD COLUMN location_id INT(11) DEFAULT NULL AFTER worker_id,
            ADD COLUMN shift_pattern_id INT(11) DEFAULT NULL AFTER location_id,
            ADD INDEX idx_location_id (location_id),
            ADD INDEX idx_shift_pattern_id (shift_pattern_id)");
        echo "✓ Added location and shift context to attendance table<br>";
    } else {
        echo "✓ Attendance table already has location context<br>";
    }

    // Verify table structure
    echo "<br>Worker Location Assignments table structure:<br>";
    $columns = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "worker_location_assignments");
    echo "<table border='1' cellpadding='5'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
    while ($col = $columns->fetch(PDO::FETCH_ASSOC)) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($col['Field']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Type']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Null']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Key']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Default'] ?? '') . "</td>";
        echo "</tr>";
    }
    echo "</table>";

    echo "<br><br>✓ All done! Workers can now be assigned to multiple locations with different shift patterns.";
    echo "<br><br><a href='list.php'>Go to Workers List</a>";

} catch (PDOException $e) {
    echo "Error: " . $e->getMessage();
}
