// Address History Management with Multi-Step Modal
let addressHistoryCount = 0;
let addressesArray = [];
let addressHistoryType = ''; // 'current', '3', or '5'
let requiredYears = 0;
let yearsCovered = 0;

function openAddressModal() {
    // Show Bootstrap modal
    const modal = new bootstrap.Modal(document.getElementById('addressModal'));
    modal.show();

    // Reset to step 1
    resetAddressModal();
}

function resetAddressModal() {
    // Show type selection, hide form
    document.getElementById('addressTypeSelection').style.display = 'block';
    document.getElementById('addressEntryForm').style.display = 'none';
    document.getElementById('saveAddressBtn').style.display = 'none';
    document.getElementById('progressIndicator').style.display = 'none';

    // Reset radio buttons
    const radios = document.querySelectorAll('input[name="addressHistoryType"]');
    radios.forEach(radio => radio.checked = false);

    // Clear form
    document.getElementById('addressForm').reset();
    document.getElementById('currentAddressCheck').checked = false;
    document.getElementById('modal_move_out_date').disabled = false;
    document.getElementById('modal_move_out_date').required = true;

    // Reset tracking variables for new session
    addressHistoryType = '';
    requiredYears = 0;
    yearsCovered = 0;
}

function startAddressEntry() {
    // Get selected type
    const selectedRadio = document.querySelector('input[name="addressHistoryType"]:checked');
    if (!selectedRadio) {
        alert('Please select an address history option');
        return;
    }

    addressHistoryType = selectedRadio.value;

    // Set required years
    if (addressHistoryType === 'current') {
        requiredYears = 0;
    } else if (addressHistoryType === '3') {
        requiredYears = 3;
    } else if (addressHistoryType === '5') {
        requiredYears = 5;
    }

    // For current address only, check the checkbox automatically
    if (addressHistoryType === 'current') {
        document.getElementById('currentAddressCheck').checked = true;
        toggleMoveOutDate();
    }

    // Hide type selection, show form
    document.getElementById('addressTypeSelection').style.display = 'none';
    document.getElementById('addressEntryForm').style.display = 'block';
    document.getElementById('saveAddressBtn').style.display = 'block';

    // Show progress indicator for multi-year history
    if (requiredYears > 0) {
        document.getElementById('progressIndicator').style.display = 'block';
        updateProgressIndicator();
    }
}

function toggleMoveOutDate() {
    const checkbox = document.getElementById('currentAddressCheck');
    const moveOutField = document.getElementById('modal_move_out_date');
    const moveOutLabel = document.getElementById('moveOutLabel');

    if (checkbox.checked) {
        // Current address - disable move out date and set to today
        moveOutField.value = new Date().toISOString().split('T')[0];
        moveOutField.disabled = true;
        moveOutField.required = false;
        moveOutLabel.innerHTML = 'Move Out Date';
    } else {
        // Previous address - enable move out date
        moveOutField.value = '';
        moveOutField.disabled = false;
        moveOutField.required = true;
        moveOutLabel.innerHTML = 'Move Out Date <span class="text-danger">*</span>';
    }
}

function saveAddress() {
    // Get form values
    const addressLine1 = document.getElementById('modal_address_line1').value.trim();
    const addressLine2 = document.getElementById('modal_address_line2').value.trim();
    const city = document.getElementById('modal_city').value.trim();
    const postcode = document.getElementById('modal_postcode').value.trim();
    const country = document.getElementById('modal_country').value.trim();
    const moveInDate = document.getElementById('modal_move_in_date').value;
    const moveOutDate = document.getElementById('modal_move_out_date').value;
    const isCurrent = document.getElementById('currentAddressCheck').checked;

    // Validate required fields
    if (!addressLine1 || !moveInDate) {
        alert('Please fill in all required fields');
        return;
    }

    if (!isCurrent && !moveOutDate) {
        alert('Please provide Move Out Date or check "This is my current address"');
        return;
    }

    // Validate dates
    const moveIn = new Date(moveInDate);
    const moveOut = new Date(moveOutDate);

    if (moveOut < moveIn) {
        alert('Move Out Date cannot be before Move In Date');
        return;
    }

    // Check for overlapping addresses
    const overlap = checkForOverlap(moveInDate, moveOutDate);
    if (overlap) {
        alert('This address period overlaps with an existing address. Please adjust the dates.\n\n' + overlap);
        return;
    }

    // Create address object
    addressHistoryCount++;
    const address = {
        id: addressHistoryCount,
        addressLine1,
        addressLine2,
        city,
        postcode,
        country,
        moveInDate,
        moveOutDate,
        isCurrent
    };

    // Add to array
    addressesArray.push(address);

    // Calculate years covered
    yearsCovered = calculateTotalYearsCovered();

    // Create address card on main page
    addAddressCard(address);

    // Check if we need more addresses
    if (addressHistoryType === 'current') {
        // Just one address - close modal
        closeModal();
    } else if (yearsCovered >= requiredYears) {
        // Completed required period - close modal
        alert(`Address history complete! You've provided ${yearsCovered.toFixed(1)} years of address history.`);
        closeModal();
    } else {
        // Need more addresses - clear form for next entry
        clearFormForNextAddress();
        updateProgressIndicator();

        // Update button text
        document.getElementById('saveButtonText').textContent = 'Add Next Address';
    }
}

function checkForOverlap(newMoveIn, newMoveOut) {
    const newStart = new Date(newMoveIn);
    const newEnd = new Date(newMoveOut);

    for (const addr of addressesArray) {
        const existingStart = new Date(addr.moveInDate);
        const existingEnd = new Date(addr.moveOutDate);

        // Check if periods overlap
        if (newStart <= existingEnd && newEnd >= existingStart) {
            return `Overlaps with: ${addr.addressLine1} (${formatDate(addr.moveInDate)} - ${formatDate(addr.moveOutDate)})`;
        }
    }

    return null;
}

function calculateTotalYearsCovered() {
    if (addressesArray.length === 0) return 0;

    // Sort addresses by move in date (oldest first)
    const sorted = [...addressesArray].sort((a, b) =>
        new Date(a.moveInDate) - new Date(b.moveInDate)
    );

    // Find the earliest move in date
    const earliest = new Date(sorted[0].moveInDate);

    // Find the latest move out date (should be today if current address included)
    const latest = new Date(sorted[sorted.length - 1].moveOutDate);

    // Calculate years difference
    const diffTime = Math.abs(latest - earliest);
    const diffYears = diffTime / (1000 * 60 * 60 * 24 * 365.25);

    return diffYears;
}

function updateProgressIndicator() {
    const remaining = Math.max(0, requiredYears - yearsCovered);
    const progressText = document.getElementById('progressText');

    if (yearsCovered === 0) {
        progressText.innerHTML = `You need to provide <strong>${requiredYears} years</strong> of address history.`;
    } else if (remaining > 0) {
        progressText.innerHTML = `<strong>${yearsCovered.toFixed(1)} years</strong> covered. <strong>${remaining.toFixed(1)} years</strong> remaining.`;
    } else {
        progressText.innerHTML = `<strong>Complete!</strong> ${yearsCovered.toFixed(1)} years covered.`;
    }
}

function clearFormForNextAddress() {
    // Clear form fields but keep country as United Kingdom
    document.getElementById('modal_address_line1').value = '';
    document.getElementById('modal_address_line2').value = '';
    document.getElementById('modal_city').value = '';
    document.getElementById('modal_postcode').value = '';
    document.getElementById('modal_country').value = 'United Kingdom';
    document.getElementById('modal_move_in_date').value = '';
    document.getElementById('modal_move_out_date').value = '';

    // Uncheck current address checkbox
    document.getElementById('currentAddressCheck').checked = false;
    toggleMoveOutDate();
}

function closeModal() {
    const modal = bootstrap.Modal.getInstance(document.getElementById('addressModal'));
    modal.hide();
    resetAddressModal();
}

function addAddressCard(address) {
    const container = document.getElementById('addressHistoryBoxes');

    // Determine if this is current address
    const today = new Date().toISOString().split('T')[0];
    const isCurrentAddr = address.isCurrent || address.moveOutDate === today;
    const dateRangeText = isCurrentAddr ?
        `${formatDate(address.moveInDate)} - Present` :
        `${formatDate(address.moveInDate)} - ${formatDate(address.moveOutDate)}`;

    // Create the address card HTML
    const cardHTML = `
        <div class="address-card-box" id="address-card-${address.id}" data-move-out="${address.moveOutDate}">
            <button type="button" class="btn btn-danger btn-sm remove-btn" onclick="removeAddress(${address.id})">
                <i class="bi bi-x"></i>
            </button>

            ${isCurrentAddr ? '<span class="badge bg-success mb-2">Current Address</span>' : ''}

            <strong>${address.addressLine1}</strong>
            ${address.addressLine2 ? `<small>${address.addressLine2}</small>` : ''}
            <small>${address.city ? address.city + ', ' : ''}${address.postcode}</small>
            <small>${address.country}</small>

            <span class="date-range">
                ${dateRangeText}
            </span>

            <!-- Hidden inputs for form submission -->
            <input type="hidden" name="address_history[${address.id}][address_line1]" value="${escapeHtml(address.addressLine1)}">
            <input type="hidden" name="address_history[${address.id}][address_line2]" value="${escapeHtml(address.addressLine2)}">
            <input type="hidden" name="address_history[${address.id}][city]" value="${escapeHtml(address.city)}">
            <input type="hidden" name="address_history[${address.id}][postcode]" value="${escapeHtml(address.postcode)}">
            <input type="hidden" name="address_history[${address.id}][country]" value="${escapeHtml(address.country)}">
            <input type="hidden" name="address_history[${address.id}][move_in_date]" value="${address.moveInDate}">
            <input type="hidden" name="address_history[${address.id}][move_out_date]" value="${address.moveOutDate}">
        </div>
    `;

    // Find the plus button
    const plusButton = container.querySelector('.address-add-box');

    // Get all existing cards
    const existingCards = Array.from(container.querySelectorAll('.address-card-box'));

    // Add new card to array
    existingCards.push({ html: cardHTML, moveOutDate: new Date(address.moveOutDate) });

    // Sort by move out date (newest first)
    existingCards.sort((a, b) => b.moveOutDate - a.moveOutDate);

    // Remove all existing cards
    container.querySelectorAll('.address-card-box').forEach(card => card.remove());

    // Re-insert cards in sorted order
    existingCards.forEach(card => {
        if (card.html) {
            plusButton.insertAdjacentHTML('beforebegin', card.html);
        } else {
            plusButton.insertAdjacentElement('beforebegin', card);
        }
    });
}

function removeAddress(id) {
    // Remove from array
    addressesArray = addressesArray.filter(addr => addr.id !== id);

    // Remove card from DOM
    const card = document.getElementById(`address-card-${id}`);
    if (card) {
        card.remove();
    }

    // Recalculate years covered
    yearsCovered = calculateTotalYearsCovered();
}

function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    return `${months[date.getMonth()]} ${date.getFullYear()}`;
}

function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}
