<?php
// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
}

$page_title = 'Add New Worker';
$current_page = 'workers';
$errors = [];

// Get database connection
$pdo = get_database_connection();

// Get module settings for employee number generation
$employee_number_format = 'manual'; // default
$employee_number_prefix = 'ER'; // default
$employee_number_padding = 3; // default
$generated_employee_number = '';

// Load module settings
$settings_file = __DIR__ . '/../../includes/get_module_settings.php';
if (file_exists($settings_file)) {
    require_once $settings_file;
    $worker_settings = get_module_settings('workers');
    $employee_number_format = $worker_settings['employee_number_format'] ?? 'manual';
    $employee_number_prefix = $worker_settings['employee_number_prefix'] ?? 'ER';
    $employee_number_padding = (int)($worker_settings['employee_number_padding'] ?? 3);
}

// Generate next employee number if auto-generation is enabled
if ($employee_number_format !== 'manual' && empty($_POST['employee_number'])) {
    // Get the highest existing employee number
    $stmt = $pdo->query("SELECT employee_number FROM " . DB_PREFIX . "sponsored_workers ORDER BY id DESC LIMIT 100");
    $existing_numbers = $stmt->fetchAll(PDO::FETCH_COLUMN);

    $next_number = 1;

    if ($employee_number_format === 'auto_prefix' && !empty($employee_number_prefix)) {
        // Extract numbers from existing employee IDs that match the prefix
        foreach ($existing_numbers as $emp_num) {
            if (strpos($emp_num, $employee_number_prefix) === 0) {
                $num_part = (int)substr($emp_num, strlen($employee_number_prefix));
                if ($num_part >= $next_number) {
                    $next_number = $num_part + 1;
                }
            }
        }
        $generated_employee_number = $employee_number_prefix . str_pad($next_number, $employee_number_padding, '0', STR_PAD_LEFT);
    } elseif ($employee_number_format === 'auto_numeric') {
        // Pure numeric - get highest number
        foreach ($existing_numbers as $emp_num) {
            if (is_numeric($emp_num)) {
                $num = (int)$emp_num;
                if ($num >= $next_number) {
                    $next_number = $num + 1;
                }
            }
        }
        $generated_employee_number = str_pad($next_number, 5, '0', STR_PAD_LEFT);
    } elseif ($employee_number_format === 'auto_alpha') {
        // EMP prefix - extract numbers from EMPxxx format
        foreach ($existing_numbers as $emp_num) {
            if (preg_match('/EMP(\d+)/', $emp_num, $matches)) {
                $num = (int)$matches[1];
                if ($num >= $next_number) {
                    $next_number = $num + 1;
                }
            }
        }
        $generated_employee_number = 'EMP' . str_pad($next_number, 3, '0', STR_PAD_LEFT);
    }
}

// Load nationalities - use absolute path for reliability
$nationalities_file = __DIR__ . '/../../includes/nationalities.php';
if (file_exists($nationalities_file)) {
    $nationalities_data = include $nationalities_file;
    $all_nationalities = $nationalities_data['all'] ?? [];
    $eu_eea_countries = $nationalities_data['eu_eea'] ?? [];
    $no_visa_countries = $nationalities_data['no_visa_required'] ?? [];
} else {
    // Fallback if file not found
    $all_nationalities = ['British', 'Irish', 'American', 'Australian', 'Canadian', 'Chinese', 'French', 'German', 'Indian', 'Nigerian', 'Pakistani', 'Polish', 'Spanish'];
    $eu_eea_countries = ['British', 'Irish'];
    $no_visa_countries = ['British', 'Irish'];
}

// Country names (for passport country of issue)
$all_countries = [
    'Afghanistan', 'Albania', 'Algeria', 'Andorra', 'Angola', 'Antigua and Barbuda', 'Argentina', 'Armenia', 'Australia', 'Austria',
    'Azerbaijan', 'Bahamas', 'Bahrain', 'Bangladesh', 'Barbados', 'Belarus', 'Belgium', 'Belize', 'Benin', 'Bhutan',
    'Bolivia', 'Bosnia and Herzegovina', 'Botswana', 'Brazil', 'Brunei', 'Bulgaria', 'Burkina Faso', 'Burundi',
    'Cambodia', 'Cameroon', 'Canada', 'Cape Verde', 'Central African Republic', 'Chad', 'Chile', 'China', 'Colombia', 'Comoros',
    'Congo', 'Costa Rica', 'Croatia', 'Cuba', 'Cyprus', 'Czech Republic',
    'Denmark', 'Djibouti', 'Dominica', 'Dominican Republic',
    'Ecuador', 'Egypt', 'El Salvador', 'Equatorial Guinea', 'Eritrea', 'Estonia', 'Eswatini', 'Ethiopia',
    'Fiji', 'Finland', 'France',
    'Gabon', 'Gambia', 'Georgia', 'Germany', 'Ghana', 'Greece', 'Grenada', 'Guatemala', 'Guinea', 'Guinea-Bissau', 'Guyana',
    'Haiti', 'Honduras', 'Hungary',
    'Iceland', 'India', 'Indonesia', 'Iran', 'Iraq', 'Ireland', 'Israel', 'Italy',
    'Jamaica', 'Japan', 'Jordan',
    'Kazakhstan', 'Kenya', 'Kiribati', 'Kosovo', 'Kuwait', 'Kyrgyzstan',
    'Laos', 'Latvia', 'Lebanon', 'Lesotho', 'Liberia', 'Libya', 'Liechtenstein', 'Lithuania', 'Luxembourg',
    'Madagascar', 'Malawi', 'Malaysia', 'Maldives', 'Mali', 'Malta', 'Marshall Islands', 'Mauritania', 'Mauritius', 'Mexico',
    'Micronesia', 'Moldova', 'Monaco', 'Mongolia', 'Montenegro', 'Morocco', 'Mozambique', 'Myanmar',
    'Namibia', 'Nauru', 'Nepal', 'Netherlands', 'New Zealand', 'Nicaragua', 'Niger', 'Nigeria', 'North Korea', 'North Macedonia', 'Norway',
    'Oman',
    'Pakistan', 'Palau', 'Palestine', 'Panama', 'Papua New Guinea', 'Paraguay', 'Peru', 'Philippines', 'Poland', 'Portugal',
    'Qatar',
    'Romania', 'Russia', 'Rwanda',
    'Saint Lucia', 'Samoa', 'San Marino', 'Sao Tome and Principe', 'Saudi Arabia', 'Senegal', 'Serbia', 'Seychelles',
    'Sierra Leone', 'Singapore', 'Slovakia', 'Slovenia', 'Solomon Islands', 'Somalia', 'South Africa', 'South Korea',
    'South Sudan', 'Spain', 'Sri Lanka', 'Sudan', 'Suriname', 'Sweden', 'Switzerland', 'Syria',
    'Taiwan', 'Tajikistan', 'Tanzania', 'Thailand', 'Timor-Leste', 'Togo', 'Tonga', 'Trinidad and Tobago', 'Tunisia',
    'Turkey', 'Turkmenistan', 'Tuvalu',
    'Uganda', 'Ukraine', 'United Arab Emirates', 'United Kingdom', 'United States', 'Uruguay', 'Uzbekistan',
    'Vanuatu', 'Vatican City', 'Venezuela', 'Vietnam',
    'Yemen',
    'Zambia', 'Zimbabwe'
];

// Fetch departments and job roles from centralized tables
$departments = [];
$job_roles = [];
$existing_staff = [];
try {
    $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "departments WHERE is_active = 1 ORDER BY name");
    $departments = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $stmt = $pdo->query("SELECT * FROM " . DB_PREFIX . "job_roles WHERE is_active = 1 ORDER BY title");
    $job_roles = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Fetch all existing staff for line manager dropdown (hierarchy building)
    $stmt = $pdo->query("SELECT id, full_name, job_title, department, employee_number
                         FROM " . DB_PREFIX . "sponsored_workers
                         WHERE status = 'active'
                         ORDER BY full_name");
    $existing_staff = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Silently fail if tables don't exist yet
}

// Check if locations module is active
$locations_module_active = false;
$work_locations = [];
try {
    $stmt = $pdo->query("SELECT status FROM " . DB_PREFIX . "modules_registry WHERE module_id = 'locations' LIMIT 1");
    $module = $stmt->fetch(PDO::FETCH_ASSOC);
    $locations_module_active = ($module && $module['status'] === 'active');

    // Fetch work locations only if module is active
    if ($locations_module_active) {
        $stmt = $pdo->query("SELECT id, name, type FROM " . DB_PREFIX . "work_locations WHERE status = 'active' ORDER BY name");
        $work_locations = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Fetch all shift patterns grouped by location for multi-location assignments
        $stmt = $pdo->query("SELECT id, location_id, shift_name, shift_type, start_time, end_time, is_default
            FROM " . DB_PREFIX . "location_shift_patterns
            WHERE is_active = 1
            ORDER BY location_id, is_default DESC, id ASC");
        $all_shift_patterns = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Group shift patterns by location_id
        $shift_patterns_by_location = [];
        foreach ($all_shift_patterns as $pattern) {
            $shift_patterns_by_location[$pattern['location_id']][] = $pattern;
        }
    }
} catch (PDOException $e) {
    // Module registry doesn't exist or locations not installed
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $employee_number = trim($_POST['employee_number'] ?? '');
    $first_name = trim($_POST['first_name'] ?? '');
    $middle_name = trim($_POST['middle_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $nationality = trim($_POST['nationality'] ?? '');

    $error_fields = []; // Track which fields have errors

    // Validate required fields
    if (empty($employee_number)) {
        $errors[] = "Employee number required";
        $error_fields[] = 'employee_number';
    }
    if (empty($first_name)) {
        $errors[] = "First name required";
        $error_fields[] = 'first_name';
    }
    if (empty($last_name)) {
        $errors[] = "Last name required";
        $error_fields[] = 'last_name';
    }
    if (empty($nationality)) {
        $errors[] = "Nationality required";
        $error_fields[] = 'nationality';
    }

    // Nationality-based validation
    $is_british = (strtolower($nationality) === 'british' || strtolower($nationality) === 'irish');

    if (!$is_british) {
        if (empty($_POST['visa_type'])) {
            $errors[] = "Visa/immigration status required for non-British/Irish nationals";
            $error_fields[] = 'visa_type';
        }
    }

    // Check duplicate employee number
    if (!empty($employee_number) && empty($errors)) {
        $check = $pdo->prepare("SELECT id FROM scms_sponsored_workers WHERE employee_number = ?");
        $check->execute([$employee_number]);
        if ($check->fetch()) $errors[] = "Employee number already exists";
    }

    if (empty($errors)) {
        // Build full name
        $full_name = $first_name;
        if (!empty($middle_name)) $full_name .= ' ' . $middle_name;
        $full_name .= ' ' . $last_name;

        // Determine worker status based on nationality and visa status
        $awaiting_visa = isset($_POST['awaiting_visa']) ? 1 : 0;
        $worker_status = 'active'; // Default for British/Irish

        if (!$is_british) {
            // Non-British/Irish nationals
            if ($awaiting_visa == 1) {
                // Awaiting visa = pre_employment status
                $worker_status = 'pre_employment';
            } else {
                // In UK but needs RTW check = pending_rtw status
                $worker_status = 'pending_rtw';
            }
        }

        // Start date logic: Only active workers can have a start date
        // Pre-employment and pending_rtw workers get NULL start_date until activated
        $start_date = $_POST['start_date'] ?: null;
        if ($worker_status !== 'active') {
            $start_date = null; // Cannot set start date until worker is activated
        }

        try {
            // Map to actual database columns - COMPLETE field list
            $stmt = $pdo->prepare("INSERT INTO scms_sponsored_workers
                (employee_number, first_name, last_name, full_name,
                 date_of_birth, gender, place_of_birth, nationality,
                 passport_number, passport_issue_date, passport_expiry, passport_country,
                 ni_number, driving_license,
                 visa_type, visa_number, visa_issue_date, visa_expiry,
                 brp_number, brp_expiry, share_code, share_code_expiry,
                 cos_number, cos_issued_date, cos_location,
                 email, personal_email, phone, mobile_phone,
                 emergency_contact, emergency_phone, emergency_contact_relationship, emergency_contact_address,
                 secondary_emergency_contact, secondary_emergency_phone, secondary_emergency_relationship,
                 job_title, department, location_id, manager_id, line_manager, contract_type,
                 start_date, probation_end_date, annual_salary, hourly_rate, work_hours_per_week,
                 employment_status, employment_status_date, employment_status_notes,
                 awaiting_visa,
                 bank_name, bank_account_number, bank_sort_code,
                 notes,
                 status, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

            $stmt->execute([
                $employee_number,
                $first_name,
                $last_name,
                $full_name,
                $_POST['date_of_birth'] ?: null,
                $_POST['gender'] ?: null,
                $_POST['place_of_birth'] ?: null,
                $nationality,
                $_POST['passport_number'] ?: null,
                $_POST['passport_issue_date'] ?: null,
                $_POST['passport_expiry'] ?: null,
                $_POST['passport_country'] ?: null,
                $_POST['ni_number'] ?: null,
                $_POST['driving_license'] ?: null,
                $_POST['visa_type'] ?: null,
                $_POST['visa_number'] ?: null,
                $_POST['visa_issue_date'] ?: null,
                $_POST['visa_expiry'] ?: null,
                $_POST['brp_number'] ?: null,
                $_POST['brp_expiry'] ?: null,
                $_POST['share_code'] ?: null,
                $_POST['share_code_expiry'] ?: null,
                $_POST['cos_number'] ?: null,
                $_POST['cos_issued_date'] ?: null,
                $_POST['cos_location'] ?: null,
                $_POST['email'] ?: null,
                $_POST['personal_email'] ?: null,
                $_POST['phone'] ?: null,
                $_POST['mobile_phone'] ?: null,
                $_POST['emergency_contact'] ?: null,
                $_POST['emergency_phone'] ?: null,
                $_POST['emergency_contact_relationship'] ?: null,
                $_POST['emergency_contact_address'] ?: null,
                $_POST['secondary_emergency_contact'] ?: null,
                $_POST['secondary_emergency_phone'] ?: null,
                $_POST['secondary_emergency_relationship'] ?: null,
                $_POST['job_title'] ?: null,
                $_POST['department'] ?: null,
                $_POST['location_id'] ?: null,
                $_POST['manager_id'] ?: null,
                $_POST['line_manager'] ?: null,
                $_POST['contract_type'] ?: null,
                $start_date,
                $_POST['probation_end_date'] ?: null,
                $_POST['annual_salary'] ?: null,
                $_POST['hourly_rate'] ?: null,
                $_POST['work_hours_per_week'] ?: null,
                $_POST['employment_status'] ?: 'new_hire',
                $_POST['employment_status_date'] ?: null,
                $_POST['employment_status_notes'] ?: null,
                isset($_POST['awaiting_visa']) ? 1 : 0,
                $_POST['bank_name'] ?: null,
                $_POST['bank_account_number'] ?: null,
                $_POST['bank_sort_code'] ?: null,
                $_POST['notes'] ?: null,
                $worker_status
            ]);

            $worker_id = $pdo->lastInsertId();

            // Handle location assignments if locations module is active
            if ($locations_module_active && isset($_POST['location_assignments'])) {
                $location_assignments = json_decode($_POST['location_assignments'], true);

                if (!empty($location_assignments)) {
                    $assignment_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_location_assignments
                        (worker_id, location_id, shift_pattern_id, days_of_week, is_primary, start_date, end_date, notes)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)");

                    foreach ($location_assignments as $index => $assignment) {
                        // Clean dates - convert empty or invalid dates to null
                        $start_date = $assignment['start_date'] ?? null;
                        if ($start_date === '' || $start_date === '0000-00-00' || $start_date === 'null' || $start_date === null) {
                            $start_date = null;
                        }

                        $end_date = $assignment['end_date'] ?? null;
                        if ($end_date === '' || $end_date === '0000-00-00' || $end_date === 'null' || $end_date === null) {
                            $end_date = null;
                        }

                        $assignment_stmt->execute([
                            $worker_id,
                            $assignment['location_id'] ?? null,
                            $assignment['shift_pattern_id'] ?? null,
                            $assignment['days_of_week'] ?? 'Mon,Tue,Wed,Thu,Fri',
                            $index === 0 ? 1 : 0, // First assignment is primary
                            $start_date,
                            $end_date,
                            $assignment['notes'] ?? null
                        ]);
                    }

                    // Update primary location_id in worker table for backward compatibility
                    if (!empty($location_assignments[0]['location_id'])) {
                        $pdo->prepare("UPDATE " . DB_PREFIX . "sponsored_workers SET location_id = ? WHERE id = ?")
                            ->execute([$location_assignments[0]['location_id'], $worker_id]);
                    }
                }
            }

            // Handle worker addresses from the modal
            if (isset($_POST['addresses_data'])) {
                $addresses_data = json_decode($_POST['addresses_data'], true);

                if (!empty($addresses_data)) {
                    // Insert addresses
                    $address_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_addresses
                        (worker_id, address_line1, address_line2, city, postcode, country, move_in_date, move_out_date)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)");

                    foreach ($addresses_data as $addr) {
                        $address_stmt->execute([
                            $worker_id,
                            $addr['addressLine1'] ?? $addr['address_line1'] ?? null,
                            $addr['addressLine2'] ?? $addr['address_line2'] ?? null,
                            $addr['city'] ?? null,
                            $addr['postcode'] ?? null,
                            $addr['country'] ?? 'United Kingdom',
                            $addr['moveInDate'] ?? $addr['move_in_date'] ?? null,
                            $addr['moveOutDate'] ?? $addr['move_out_date'] ?? null
                        ]);
                    }
                }
            }

            // Redirect based on how the page was accessed
            if (defined('MODULE_EMBEDDED')) {
                header("Location: ../../m/workers/list?success=1");
            } else {
                header("Location: list.php?success=1");
            }
            exit;

        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<style>
/* Professional Form Layout */
.dashboard-container {
    max-width: 1400px;
    margin: 0 auto;
}

.form-section {
    background: #ffffff;
    border: 1px solid #d2d4d7;
    border-radius: 6px;
    padding: 1.25rem 1.5rem;
    margin-bottom: 1.25rem;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
    transition: border-color 0.2s ease;
}

.form-section:hover {
    border-color: #a8acb0;
}

.form-section-title {
    font-size: 0.95rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 1.1rem;
    padding-bottom: 0.65rem;
    border-bottom: 2px solid #e9ecef;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    text-transform: uppercase;
    letter-spacing: 0.3px;
}

.form-section-title i {
    font-size: 1.1rem;
    color: #6366f1;
}

/* Compact Form Labels */
.form-label {
    font-size: 0.875rem;
    font-weight: 500;
    color: #495057;
    margin-bottom: 0.35rem;
    display: block;
}

.form-label strong {
    font-weight: 600;
    color: #212529;
}

.form-label .text-danger {
    color: #dc3545 !important;
    font-weight: 700;
}

/* Improved Form Controls */
.form-control, .form-select {
    font-size: 0.875rem;
    padding: 0.5rem 0.75rem;
    border: 1px solid #d2d4d7;
    border-radius: 4px;
    transition: all 0.15s ease;
    background-color: #ffffff;
}

.form-control:focus, .form-select:focus {
    border-color: #6366f1;
    box-shadow: 0 0 0 0.2rem rgba(99, 102, 241, 0.15);
    outline: none;
}

.form-control[readonly] {
    background-color: #f8f9fa;
    border-color: #e9ecef;
    cursor: not-allowed;
}

/* Small Helper Text */
small.text-muted {
    font-size: 0.75rem;
    color: #6c757d;
    margin-top: 0.25rem;
    display: block;
    line-height: 1.3;
}

/* Row Spacing */
.row {
    margin-bottom: 0.5rem;
}

.row:last-child {
    margin-bottom: 0;
}

.mb-3 {
    margin-bottom: 0.85rem !important;
}

/* Welcome Section Enhancement */
.welcome-section {
    background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
    padding: 1.5rem 2rem;
    border-radius: 6px;
    margin-bottom: 1.5rem;
    box-shadow: 0 2px 8px rgba(99, 102, 241, 0.2);
}

.welcome-title {
    font-size: 1.5rem;
    font-weight: 600;
    color: #ffffff;
    margin-bottom: 0.25rem;
}

.welcome-subtitle {
    font-size: 0.875rem;
    color: rgba(255, 255, 255, 0.9);
    margin-bottom: 0;
}

.welcome-actions .btn {
    font-size: 0.875rem;
    padding: 0.5rem 1rem;
}

.conditional-field {
    display: none;
}

.conditional-field.show {
    display: block;
    animation: fadeIn 0.3s ease;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(-10px); }
    to { opacity: 1; transform: translateY(0); }
}

.info-badge {
    background: #eff6ff;
    color: #1e40af;
    padding: 0.65rem 1rem;
    border-radius: 4px;
    font-size: 0.8rem;
    margin-bottom: 1rem;
    border-left: 3px solid #3b82f6;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.info-badge i {
    color: #3b82f6;
}

/* Address Box Styling - Professional Compact Design */
.address-add-box {
    width: 220px;
    height: 180px;
    border: 2px dashed #cbd5e0;
    border-radius: 6px;
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    gap: 8px;
    transition: all 0.2s ease;
    background: #fafbfc;
    cursor: pointer;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
}

.address-add-box:hover {
    border-color: #6366f1;
    background: #ffffff;
    transform: translateY(-2px);
    box-shadow: 0 3px 8px rgba(99, 102, 241, 0.15);
}

.address-add-box i {
    font-size: 2.5rem;
    color: #6366f1;
}

.address-add-box span {
    font-size: 0.875rem;
    font-weight: 500;
    color: #495057;
}

.address-add-box:hover span {
    color: #6366f1;
}

.address-card-box {
    width: 260px;
    min-height: 180px;
    border: 1px solid #d2d4d7;
    border-radius: 6px;
    padding: 1rem 1.25rem;
    background: #ffffff;
    box-shadow: 0 1px 3px rgba(0,0,0,0.08);
    position: relative;
    transition: all 0.2s ease;
}

.address-card-box:hover {
    transform: translateY(-2px);
    box-shadow: 0 3px 8px rgba(0,0,0,0.12);
    border-color: #a8acb0;
}

.address-card-box .remove-btn {
    position: absolute;
    top: 8px;
    right: 8px;
    padding: 3px 8px;
    font-size: 0.7rem;
    background-color: #dc3545;
    color: white;
    border: none;
    border-radius: 3px;
    opacity: 0.8;
    transition: opacity 0.2s;
}

.address-card-box .remove-btn:hover {
    opacity: 1;
}

.address-card-box small {
    display: block;
    margin-bottom: 4px;
    color: #6c757d;
    font-size: 0.75rem;
    text-transform: uppercase;
    letter-spacing: 0.3px;
    font-weight: 500;
}

.address-card-box strong {
    display: block;
    margin-bottom: 8px;
    color: #212529;
    font-size: 0.9rem;
    font-weight: 600;
    line-height: 1.4;
}

.address-card-box .date-range {
    display: block;
    margin-top: 10px;
    padding-top: 10px;
    border-top: 1px solid #e9ecef;
    font-size: 0.75rem;
    color: #6c757d;
}

/* Subtle placeholders */
.form-control::placeholder,
.form-control::-webkit-input-placeholder {
    color: #adb5bd;
    opacity: 0.6;
    font-weight: 400;
    font-size: 0.85rem;
}

.form-control::-moz-placeholder {
    color: #adb5bd;
    opacity: 0.6;
    font-weight: 400;
}

.form-control:-ms-input-placeholder {
    color: #adb5bd;
    opacity: 0.6;
    font-weight: 400;
}

/* Focus state - remove placeholder when user starts typing */
.form-control:focus::placeholder {
    opacity: 0.3;
    transition: opacity 0.2s ease;
}

/* Professional Button Styling */
.btn {
    font-size: 0.875rem;
    font-weight: 500;
    padding: 0.5rem 1.25rem;
    border-radius: 4px;
    transition: all 0.2s ease;
    border: 1px solid transparent;
}

.btn-primary {
    background: #6366f1;
    border-color: #6366f1;
    color: #ffffff;
}

.btn-primary:hover {
    background: #4f46e5;
    border-color: #4f46e5;
    transform: translateY(-1px);
    box-shadow: 0 2px 6px rgba(99, 102, 241, 0.3);
}

.btn-secondary {
    background: #64748b;
    border-color: #64748b;
    color: #ffffff;
}

.btn-secondary:hover {
    background: #475569;
    border-color: #475569;
}

.btn-success {
    background: #10b981;
    border-color: #10b981;
    color: #ffffff;
}

.btn-success:hover {
    background: #059669;
    border-color: #059669;
    transform: translateY(-1px);
    box-shadow: 0 2px 6px rgba(16, 185, 129, 0.3);
}

.btn-sm {
    font-size: 0.8rem;
    padding: 0.4rem 0.9rem;
}

/* Alert Styling */
.alert {
    font-size: 0.875rem;
    border-radius: 4px;
    border: 1px solid transparent;
    padding: 0.85rem 1.25rem;
    margin-bottom: 1.25rem;
}

.alert-danger {
    background-color: #fee2e2;
    border-color: #fecaca;
    color: #991b1b;
}

.alert-danger ul {
    margin-bottom: 0;
    padding-left: 1.25rem;
}

.alert-danger li {
    margin-bottom: 0.25rem;
    line-height: 1.5;
}

.alert-danger li:last-child {
    margin-bottom: 0;
}

/* Location Assignment Cards */
.location-assignment-card {
    border: 1px solid #d2d4d7;
    border-radius: 6px;
    padding: 1rem;
    background: #ffffff;
    margin-bottom: 0.75rem;
    transition: all 0.2s ease;
}

.location-assignment-card:hover {
    border-color: #a8acb0;
    box-shadow: 0 2px 6px rgba(0,0,0,0.08);
}

.location-assignment-card .remove-location-btn {
    font-size: 0.75rem;
    padding: 0.3rem 0.7rem;
}

/* Modal Improvements */
.modal-content {
    border: 1px solid #d2d4d7;
    border-radius: 6px;
    box-shadow: 0 4px 16px rgba(0,0,0,0.15);
}

.modal-header {
    border-bottom: 1px solid #e9ecef;
    padding: 1rem 1.25rem;
    background: #f8f9fa;
}

.modal-title {
    font-size: 1rem;
    font-weight: 600;
    color: #2c3e50;
    text-transform: capitalize;
}

.modal-body {
    padding: 1.25rem;
}

.modal-footer {
    border-top: 1px solid #e9ecef;
    padding: 0.85rem 1.25rem;
    background: #f8f9fa;
}

/* Card Container */
.modern-card {
    background: #ffffff;
    border: 1px solid #d2d4d7;
    border-radius: 6px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.05);
    overflow: hidden;
}

.card-body-modern {
    padding: 1.5rem;
}

/* Responsive Adjustments */
@media (max-width: 768px) {
    .form-section {
        padding: 1rem;
    }

    .welcome-section {
        padding: 1.25rem 1.5rem;
    }

    .address-card-box, .address-add-box {
        width: 100%;
    }
}
</style>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title"><i class="bi bi-person-plus-fill"></i> Add New Worker</h1>
            <p class="welcome-subtitle">Complete HR and immigration information</p>
        </div>
        <div class="welcome-actions">
            <a href="list.php" class="btn btn-secondary"><i class="bi bi-arrow-left"></i> Back</a>
        </div>
    </div>

    <?php if (!empty($errors)): ?>
    <div class="alert alert-danger alert-dismissible fade show">
        <ul class="mb-0">
            <?php foreach ($errors as $e): ?>
                <li><?php echo htmlspecialchars($e); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST" id="workerForm">
                <!-- Employee Number -->
                <div class="form-section">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Employee Number</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="employee_number"
                                   value="<?php echo htmlspecialchars($_POST['employee_number'] ?? $generated_employee_number); ?>"
                                   <?php echo ($employee_number_format !== 'manual' && empty($_POST['employee_number'])) ? 'readonly' : ''; ?>
                                   required>
                            <small class="text-muted">
                                <?php if ($employee_number_format !== 'manual'): ?>
                                    Auto-generated. Next: <strong><?php echo $generated_employee_number; ?></strong>
                                <?php else: ?>
                                    Unique identifier (e.g., EMP001, SW001)
                                <?php endif; ?>
                            </small>
                        </div>
                    </div>
                </div>

                <!-- Personal Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-person"></i> Personal Information
                    </h6>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>First Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="first_name"
                                   value="<?php echo htmlspecialchars($_POST['first_name'] ?? ''); ?>" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Middle Name</label>
                            <input type="text" class="form-control" name="middle_name"
                                   value="<?php echo htmlspecialchars($_POST['middle_name'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label"><strong>Last Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="last_name"
                                   value="<?php echo htmlspecialchars($_POST['last_name'] ?? ''); ?>" required>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Date of Birth</label>
                            <input type="date" class="form-control" name="date_of_birth"
                                   value="<?php echo htmlspecialchars($_POST['date_of_birth'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Gender</label>
                            <select class="form-select" name="gender">
                                <?php $selected_gender = $_POST['gender'] ?? ''; ?>
                                <option value="">Select...</option>
                                <option value="Male" <?php echo $selected_gender === 'Male' ? 'selected' : ''; ?>>Male</option>
                                <option value="Female" <?php echo $selected_gender === 'Female' ? 'selected' : ''; ?>>Female</option>
                                <option value="Other" <?php echo $selected_gender === 'Other' ? 'selected' : ''; ?>>Other</option>
                                <option value="Prefer not to say" <?php echo $selected_gender === 'Prefer not to say' ? 'selected' : ''; ?>>Prefer not to say</option>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Place of Birth</label>
                            <input type="text" class="form-control" name="place_of_birth"
                                   placeholder="City, Country"
                                   value="<?php echo htmlspecialchars($_POST['place_of_birth'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Nationality</strong> <span class="text-danger">*</span></label>
                            <select class="form-select" name="nationality" id="nationality" required>
                                <option value="">Select nationality...</option>
                                <?php foreach ($all_nationalities as $nat): ?>
                                    <option value="<?php echo htmlspecialchars($nat); ?>"
                                        <?php echo (isset($_POST['nationality']) && $_POST['nationality'] === $nat) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($nat); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Required for all workers - determines immigration requirements</small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">National Insurance Number</label>
                            <input type="text" class="form-control" name="ni_number"
                                   placeholder="AB123456C" pattern="[A-Z]{2}[0-9]{6}[A-D]"
                                   value="<?php echo htmlspecialchars($_POST['ni_number'] ?? ''); ?>">
                            <small class="text-muted">Format: 2 letters, 6 numbers, 1 letter</small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Driving License Number</label>
                            <input type="text" class="form-control" name="driving_license"
                                   value="<?php echo htmlspecialchars($_POST['driving_license'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Passport Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-passport"></i> Passport Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Number</label>
                            <input type="text" class="form-control" name="passport_number"
                                   value="<?php echo htmlspecialchars($_POST['passport_number'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Country of Issue</label>
                            <select class="form-select" name="passport_country" id="passport_country">
                                <option value="">Select country...</option>
                                <?php foreach ($all_countries as $country): ?>
                                    <option value="<?php echo htmlspecialchars($country); ?>"
                                        <?php echo (isset($_POST['passport_country']) && $_POST['passport_country'] === $country) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($country); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Usually matches nationality</small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Issue Date</label>
                            <input type="date" class="form-control" name="passport_issue_date"
                                   value="<?php echo htmlspecialchars($_POST['passport_issue_date'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Passport Expiry Date</label>
                            <input type="date" class="form-control" name="passport_expiry"
                                   value="<?php echo htmlspecialchars($_POST['passport_expiry'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Employment Status Section -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-briefcase-fill"></i> Employment Status
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Employment Status</strong> <span class="text-danger">*</span></label>
                            <select class="form-select" name="employment_status" id="employment_status" required>
                                <option value="">Select status...</option>
                                <option value="new_hire" <?php echo (isset($_POST['employment_status']) && $_POST['employment_status'] === 'new_hire') ? 'selected' : 'selected'; ?>>New Hire</option>
                                <option value="pre_employment" <?php echo (isset($_POST['employment_status']) && $_POST['employment_status'] === 'pre_employment') ? 'selected' : ''; ?>>Pre-Employment (CoS Issued, Awaiting Visa)</option>
                            </select>
                            <small class="text-muted">
                                <strong>New Hire:</strong> Standard new employee |
                                <strong>Pre-Employment:</strong> For sponsored workers awaiting visa approval
                            </small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Status Start Date</label>
                            <input type="date" class="form-control" name="employment_status_date"
                                   value="<?php echo htmlspecialchars($_POST['employment_status_date'] ?? date('Y-m-d')); ?>">
                        </div>
                    </div>

                    <!-- Pre-Employment Specific Fields -->
                    <div id="preEmploymentFields" class="conditional-field">
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> <strong>Pre-Employment Mode:</strong> Use this for workers who have been offered a job and CoS issued, but are awaiting visa approval.
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Location</label>
                                <select class="form-select" name="cos_location" id="cos_location">
                                    <option value="">Select...</option>
                                    <option value="in_country">In-Country Application</option>
                                    <option value="out_of_country">Out of Country Application</option>
                                </select>
                                <small class="text-muted">Determines which verification checks can be performed</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <div class="form-check mt-4">
                                    <input class="form-check-input" type="checkbox" name="awaiting_visa" id="awaiting_visa" value="1">
                                    <label class="form-check-label" for="awaiting_visa">
                                        <strong>Awaiting Visa Approval</strong>
                                    </label>
                                </div>
                            </div>
                        </div>

                        <div class="alert alert-warning" id="inCountryNotice" style="display:none;">
                            <i class="bi bi-exclamation-triangle"></i> <strong>In-Country Application:</strong> Right to Work checks and NI number verification should be completed.
                        </div>

                        <div class="alert alert-warning" id="outCountryNotice" style="display:none;">
                            <i class="bi bi-exclamation-triangle"></i> <strong>Out of Country Application:</strong> Right to Work checks and some verifications cannot be completed until worker arrives in UK.
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Employment Status Notes</label>
                            <textarea class="form-control" name="employment_status_notes" rows="2"><?php echo htmlspecialchars($_POST['employment_status_notes'] ?? ''); ?></textarea>
                        </div>
                    </div>
                </div>

                <!-- Immigration Status (Non-British/Irish) -->
                <div class="form-section immigration-section" id="immigrationSection">
                    <h6 class="form-section-title">
                        <i class="bi bi-globe-europe-africa"></i> Immigration Status
                    </h6>

                    <div class="info-badge mb-3" id="britishNotice" style="display: none;">
                        <i class="bi bi-info-circle"></i> British and Irish nationals have automatic right to work in the UK. Immigration details are optional.
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label"><strong>Visa/Immigration Status</strong> <span class="text-danger" id="visaRequired">*</span></label>
                            <select class="form-select" name="visa_type" id="visa_type">
                                <option value="">Select immigration status...</option>
                                <optgroup label="No Visa Required">
                                    <option value="British Citizen">British Citizen</option>
                                    <option value="Irish Citizen">Irish Citizen</option>
                                    <option value="Indefinite Leave to Remain">Indefinite Leave to Remain (ILR)</option>
                                </optgroup>
                                <optgroup label="EU Settlement Scheme">
                                    <option value="EU Settled Status">EU Settled Status</option>
                                    <option value="EU Pre-Settled Status">EU Pre-Settled Status</option>
                                </optgroup>
                                <optgroup label="Skilled Worker Route">
                                    <option value="Skilled Worker">Skilled Worker Visa</option>
                                    <option value="Health and Care Worker">Health and Care Worker Visa</option>
                                    <option value="Senior or Specialist Worker">Senior or Specialist Worker Visa</option>
                                </optgroup>
                                <optgroup label="Student & Graduate">
                                    <option value="Graduate">Graduate Visa</option>
                                    <option value="Student">Student Visa</option>
                                </optgroup>
                                <optgroup label="Family & Dependant">
                                    <option value="Spouse Visa">Spouse/Partner Visa</option>
                                    <option value="Dependant">Dependant Visa</option>
                                    <option value="Family">Family Visa</option>
                                </optgroup>
                                <optgroup label="Other Routes">
                                    <option value="Global Talent">Global Talent Visa</option>
                                    <option value="Innovator Founder">Innovator Founder Visa</option>
                                    <option value="Start-up">Start-up Visa</option>
                                    <option value="Minister of Religion">Minister of Religion Visa</option>
                                    <option value="Sportsperson">Sportsperson Visa</option>
                                    <option value="Youth Mobility">Youth Mobility Scheme</option>
                                    <option value="Other">Other</option>
                                </optgroup>
                            </select>
                        </div>
                    </div>

                    <!-- Settlement Status Checkboxes -->
                    <div class="row mb-3" id="settlementFields" style="display: none;">
                        <div class="col-md-12">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="settled_status" id="settled_status">
                                <label class="form-check-label" for="settled_status">
                                    EU Settled Status
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="pre_settled_status" id="pre_settled_status">
                                <label class="form-check-label" for="pre_settled_status">
                                    EU Pre-Settled Status
                                </label>
                            </div>
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" name="indefinite_leave" id="indefinite_leave">
                                <label class="form-check-label" for="indefinite_leave">
                                    Indefinite Leave to Remain (ILR)
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- BRP Information -->
                    <div class="conditional-field" id="brpFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Number</label>
                                <input type="text" class="form-control" name="brp_number"
                                       placeholder="12-digit number"
                                       value="<?php echo htmlspecialchars($_POST['brp_number'] ?? ''); ?>">
                                <small class="text-muted">Biometric Residence Permit</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">BRP Expiry Date</label>
                                <input type="date" class="form-control" name="brp_expiry"
                                       value="<?php echo htmlspecialchars($_POST['brp_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Share Code -->
                    <div class="conditional-field" id="shareCodeFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code</label>
                                <input type="text" class="form-control" name="share_code"
                                       placeholder="9-character code"
                                       value="<?php echo htmlspecialchars($_POST['share_code'] ?? ''); ?>">
                                <small class="text-muted">From gov.uk Right to Work check</small>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code Expiry</label>
                                <input type="date" class="form-control" name="share_code_expiry"
                                       value="<?php echo htmlspecialchars($_POST['share_code_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Visa Details (for visa holders) -->
                    <div class="conditional-field" id="visaFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Number</label>
                                <input type="text" class="form-control" name="visa_number"
                                       value="<?php echo htmlspecialchars($_POST['visa_number'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Status</label>
                                <select class="form-select" name="visa_status">
                                    <option value="">Select...</option>
                                    <option value="Valid">Valid</option>
                                    <option value="Pending">Pending</option>
                                    <option value="Expired">Expired</option>
                                    <option value="Cancelled">Cancelled</option>
                                </select>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Issue Date</label>
                                <input type="date" class="form-control" name="visa_issue_date"
                                       value="<?php echo htmlspecialchars($_POST['visa_issue_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Visa Expiry Date</label>
                                <input type="date" class="form-control" name="visa_expiry"
                                       value="<?php echo htmlspecialchars($_POST['visa_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Certificate of Sponsorship (for sponsored workers) -->
                    <div class="conditional-field" id="cosFields">
                        <h6 class="mt-3 mb-3"><i class="bi bi-file-earmark-text"></i> Certificate of Sponsorship</h6>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Number</label>
                                <input type="text" class="form-control" name="cos_number"
                                       placeholder="e.g., ABC123456789"
                                       value="<?php echo htmlspecialchars($_POST['cos_number'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Issue Date</label>
                                <input type="date" class="form-control" name="cos_issue_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_issue_date'] ?? ''); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS Start Date</label>
                                <input type="date" class="form-control" name="cos_start_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_start_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">CoS End Date</label>
                                <input type="date" class="form-control" name="cos_end_date"
                                       value="<?php echo htmlspecialchars($_POST['cos_end_date'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Right to Work Check -->
                    <div class="row mt-3">
                        <div class="col-md-12">
                            <div class="form-check mb-3">
                                <input class="form-check-input" type="checkbox" name="right_to_work_checked" id="right_to_work_checked">
                                <label class="form-check-label" for="right_to_work_checked">
                                    <strong>Right to Work Check Completed</strong>
                                </label>
                            </div>
                        </div>
                    </div>

                    <div class="conditional-field" id="rtwFields">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check Date</label>
                                <input type="date" class="form-control" name="right_to_work_check_date"
                                       value="<?php echo htmlspecialchars($_POST['right_to_work_check_date'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Next Check Due</label>
                                <input type="date" class="form-control" name="right_to_work_expiry"
                                       value="<?php echo htmlspecialchars($_POST['right_to_work_expiry'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Contact Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-envelope"></i> Contact Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Email</label>
                            <input type="email" class="form-control" name="email"
                                   value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Personal Email</label>
                            <input type="email" class="form-control" name="personal_email"
                                   value="<?php echo htmlspecialchars($_POST['personal_email'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Work Phone</label>
                            <input type="tel" class="form-control" name="phone"
                                   value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Mobile Phone</label>
                            <input type="tel" class="form-control" name="mobile_phone"
                                   value="<?php echo htmlspecialchars($_POST['mobile_phone'] ?? ''); ?>">
                        </div>
                    </div>

                    <h6 class="form-section-title">
                        <i class="bi bi-house-door"></i> Address History
                    </h6>

                    <div class="row">
                        <div class="col-md-12">
                            <p class="text-muted mb-3">Add current address and previous addresses for the last 3-5 years</p>
                            <div class="d-flex flex-wrap gap-3 align-items-start" id="addressHistoryBoxes">
                                <!-- Plus icon box to add address -->
                                <div class="address-add-box" onclick="openAddressModal()">
                                    <i class="bi bi-plus-circle"></i>
                                    <span>Add Address</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Employment Details -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-briefcase"></i> Employment Details
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Job Title</label>
                            <select class="form-select" name="job_title">
                                <option value="">Select job role...</option>
                                <?php foreach ($job_roles as $role): ?>
                                    <option value="<?php echo htmlspecialchars($role['title']); ?>"
                                        <?php echo (isset($_POST['job_title']) && $_POST['job_title'] === $role['title']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($role['title']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Or <a href="../../settings.php?tab=departments" target="_blank">add new role</a></small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Department</label>
                            <select class="form-select" name="department">
                                <option value="">Select department...</option>
                                <?php foreach ($departments as $dept): ?>
                                    <option value="<?php echo htmlspecialchars($dept['name']); ?>"
                                        <?php echo (isset($_POST['department']) && $_POST['department'] === $dept['name']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($dept['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Or <a href="../../settings.php?tab=departments" target="_blank">add new department</a></small>
                        </div>
                    </div>

                    <?php if ($locations_module_active): ?>
                    <!-- Multi-Location & Shift Assignments -->
                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">
                                <i class="bi bi-geo-alt-fill text-primary"></i> Location & Shift Assignments
                                <span class="text-danger">*</span>
                            </label>
                            <small class="text-muted d-block mb-2">
                                Assign worker to one or more locations with specific shift patterns and days.
                                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/list' : '../locations/list.php'; ?>" target="_blank">Manage locations</a>
                            </small>
                            <div id="location-assignments-container"></div>
                            <button type="button" class="btn btn-sm btn-outline-primary mt-2" onclick="addLocationAssignment()">
                                <i class="bi bi-plus-circle"></i> Add Location Assignment
                            </button>
                            <input type="hidden" name="location_assignments" id="location_assignments_input" value="">
                            <input type="hidden" name="location_id" id="primary_location_id" value="">
                            <input type="hidden" name="addresses_data" id="addresses_data_input" value="">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Line Manager</label>
                            <select class="form-select" name="line_manager" id="line_manager">
                                <option value="">Select line manager...</option>
                                <?php foreach ($existing_staff as $staff): ?>
                                    <option value="<?php echo htmlspecialchars($staff['full_name']); ?>"
                                        <?php echo (isset($_POST['line_manager']) && $_POST['line_manager'] === $staff['full_name']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($staff['full_name']); ?>
                                        <?php if (!empty($staff['job_title'])): ?>
                                            - <?php echo htmlspecialchars($staff['job_title']); ?>
                                        <?php endif; ?>
                                        <?php if (!empty($staff['employee_number'])): ?>
                                            (<?php echo htmlspecialchars($staff['employee_number']); ?>)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Required for company hierarchy reports</small>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contract Type</label>
                            <select class="form-select" name="contract_type">
                                <option value="">Select...</option>
                                <option value="Full-time">Full-time</option>
                                <option value="Part-time">Part-time</option>
                                <option value="Fixed Term">Fixed Term</option>
                                <option value="Temporary">Temporary</option>
                                <option value="Zero Hours">Zero Hours</option>
                            </select>
                        </div>
                    </div>
                    <?php else: ?>
                    <!-- If locations module not active, keep Line Manager and Contract Type in one row -->
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Line Manager</label>
                            <select class="form-select" name="line_manager" id="line_manager">
                                <option value="">Select line manager...</option>
                                <?php foreach ($existing_staff as $staff): ?>
                                    <option value="<?php echo htmlspecialchars($staff['full_name']); ?>"
                                        <?php echo (isset($_POST['line_manager']) && $_POST['line_manager'] === $staff['full_name']) ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($staff['full_name']); ?>
                                        <?php if (!empty($staff['job_title'])): ?>
                                            - <?php echo htmlspecialchars($staff['job_title']); ?>
                                        <?php endif; ?>
                                        <?php if (!empty($staff['employee_number'])): ?>
                                            (<?php echo htmlspecialchars($staff['employee_number']); ?>)
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <small class="text-muted">Required for company hierarchy reports</small>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Contract Type</label>
                            <select class="form-select" name="contract_type">
                                <option value="">Select...</option>
                                <option value="Full-time">Full-time</option>
                                <option value="Part-time">Part-time</option>
                                <option value="Fixed Term">Fixed Term</option>
                                <option value="Temporary">Temporary</option>
                                <option value="Zero Hours">Zero Hours</option>
                            </select>
                        </div>
                    </div>
                    <?php endif; ?>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Start Date</label>
                            <input type="date" class="form-control" name="start_date"
                                   value="<?php echo htmlspecialchars($_POST['start_date'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Probation End Date</label>
                            <input type="date" class="form-control" name="probation_end_date"
                                   value="<?php echo htmlspecialchars($_POST['probation_end_date'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Annual Salary (£)</label>
                            <input type="number" class="form-control" name="annual_salary" id="annual_salary" step="0.01"
                                   value="<?php echo htmlspecialchars($_POST['annual_salary'] ?? ''); ?>">
                            <small class="text-muted">Auto-calculates from hourly rate + hours</small>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hourly Rate (£)</label>
                            <input type="number" class="form-control" name="hourly_rate" id="hourly_rate" step="0.01"
                                   value="<?php echo htmlspecialchars($_POST['hourly_rate'] ?? ''); ?>">
                            <small class="text-muted">Auto-calculates from salary + hours</small>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Hours per Week</label>
                            <input type="number" class="form-control" name="work_hours_per_week" id="work_hours_per_week" step="0.5"
                                   value="<?php echo htmlspecialchars($_POST['work_hours_per_week'] ?? ''); ?>">
                            <small class="text-muted">Used for salary calculations</small>
                        </div>
                    </div>
                </div>

                <!-- Banking Information -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-bank"></i> Banking Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Bank Name</label>
                            <input type="text" class="form-control" name="bank_name"
                                   value="<?php echo htmlspecialchars($_POST['bank_name'] ?? ''); ?>">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Account Number</label>
                            <input type="text" class="form-control" name="bank_account_number"
                                   placeholder="12345678" pattern="[0-9]{8}"
                                   value="<?php echo htmlspecialchars($_POST['bank_account_number'] ?? ''); ?>">
                            <small class="text-muted">8 digits</small>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Sort Code</label>
                            <input type="text" class="form-control" name="bank_sort_code"
                                   placeholder="12-34-56" pattern="[0-9]{2}-[0-9]{2}-[0-9]{2}"
                                   value="<?php echo htmlspecialchars($_POST['bank_sort_code'] ?? ''); ?>">
                            <small class="text-muted">Format: 12-34-56</small>
                        </div>
                    </div>
                </div>

                <!-- Emergency Contact -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-telephone-plus"></i> Emergency Contact
                    </h6>

                    <h6 class="mb-3">Primary Contact</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="emergency_contact"
                                   value="<?php echo htmlspecialchars($_POST['emergency_contact'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="emergency_phone"
                                   value="<?php echo htmlspecialchars($_POST['emergency_phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="emergency_contact_relationship"
                                   placeholder="e.g., Spouse, Parent, Sibling"
                                   value="<?php echo htmlspecialchars($_POST['emergency_contact_relationship'] ?? ''); ?>">
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Contact Address</label>
                            <textarea class="form-control" name="emergency_contact_address" rows="2"><?php echo htmlspecialchars($_POST['emergency_contact_address'] ?? ''); ?></textarea>
                        </div>
                    </div>

                    <h6 class="mb-3 mt-4">Secondary Contact (Optional)</h6>
                    <div class="row">
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Name</label>
                            <input type="text" class="form-control" name="secondary_emergency_contact"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_contact'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Contact Phone</label>
                            <input type="tel" class="form-control" name="secondary_emergency_phone"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_phone'] ?? ''); ?>">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label class="form-label">Relationship</label>
                            <input type="text" class="form-control" name="secondary_emergency_relationship"
                                   value="<?php echo htmlspecialchars($_POST['secondary_emergency_relationship'] ?? ''); ?>">
                        </div>
                    </div>
                </div>

                <!-- Additional Notes -->
                <div class="form-section">
                    <h6 class="form-section-title">
                        <i class="bi bi-journal-text"></i> Additional Notes
                    </h6>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <label class="form-label">Notes</label>
                            <textarea class="form-control" name="notes" rows="4"><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                            <small class="text-muted">Any additional information about the worker</small>
                        </div>
                    </div>
                </div>

                <div class="text-end mt-4">
                    <a href="list.php" class="btn btn-secondary me-2"><i class="bi bi-x-circle"></i> Cancel</a>
                    <button type="submit" class="btn btn-primary"><i class="bi bi-check-circle"></i> Add Worker</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Address History Modal -->
<div class="modal fade" id="addressModal" tabindex="-1" data-bs-backdrop="static">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="bi bi-house-add"></i> Add Address History</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" onclick="resetAddressModal()"></button>
            </div>
            <div class="modal-body">

                <!-- Step 1: Select address history type -->
                <div id="addressTypeSelection">
                    <h6 class="mb-3">What address information would you like to provide?</h6>
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="radio" name="addressHistoryType" id="currentOnly" value="current">
                        <label class="form-check-label" for="currentOnly">
                            <strong>Current address only</strong>
                            <br><small class="text-muted">Just your current residential address</small>
                        </label>
                    </div>
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="radio" name="addressHistoryType" id="threeYears" value="3">
                        <label class="form-check-label" for="threeYears">
                            <strong>Last 3 years address history</strong>
                            <br><small class="text-muted">Current and previous addresses covering 3 years</small>
                        </label>
                    </div>
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="radio" name="addressHistoryType" id="fiveYears" value="5">
                        <label class="form-check-label" for="fiveYears">
                            <strong>Last 5 years address history</strong>
                            <br><small class="text-muted">Current and previous addresses covering 5 years (recommended for visa applications)</small>
                        </label>
                    </div>
                    <div class="mt-4">
                        <button type="button" class="btn btn-primary" onclick="startAddressEntry()">
                            <i class="bi bi-arrow-right"></i> Continue
                        </button>
                    </div>
                </div>

                <!-- Step 2: Address entry form -->
                <div id="addressEntryForm" style="display: none;">
                    <!-- Progress indicator -->
                    <div id="progressIndicator" class="alert alert-info mb-3" style="display: none;">
                        <strong>Progress:</strong> <span id="progressText"></span>
                    </div>

                    <form id="addressForm">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Address Line 1 <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="modal_address_line1" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Address Line 2</label>
                                <input type="text" class="form-control" id="modal_address_line2">
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label class="form-label">City</label>
                                <input type="text" class="form-control" id="modal_city">
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Postcode</label>
                                <input type="text" class="form-control" id="modal_postcode">
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Country</label>
                                <select class="form-select" id="modal_country">
                                    <?php foreach ($all_countries as $country): ?>
                                        <option value="<?php echo htmlspecialchars($country); ?>"
                                            <?php echo ($country === 'United Kingdom') ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($country); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Move In Date <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="modal_move_in_date" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label" id="moveOutLabel">Move Out Date <span class="text-danger">*</span></label>
                                <input type="date" class="form-control" id="modal_move_out_date" required>
                                <div class="form-check mt-2">
                                    <input class="form-check-input" type="checkbox" id="currentAddressCheck" onchange="toggleMoveOutDate()">
                                    <label class="form-check-label" for="currentAddressCheck">
                                        <strong>This is my current address</strong>
                                    </label>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>

            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" onclick="resetAddressModal()">
                    <i class="bi bi-x-circle"></i> Cancel
                </button>
                <button type="button" class="btn btn-outline-primary" id="backAddressBtn" onclick="goBackInModal()" style="display: none;">
                    <i class="bi bi-arrow-left"></i> Back
                </button>
                <button type="button" class="btn btn-primary" id="saveAddressBtn" onclick="saveAddress()" style="display: none;">
                    <i class="bi bi-check-circle"></i> <span id="saveButtonText">Add Address</span>
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Define nationality categories
const euEeaCountries = <?php echo json_encode($eu_eea_countries); ?>;
const noVisaCountries = <?php echo json_encode($no_visa_countries); ?>;

// Nationality to Country mapping
const nationalityToCountry = {
    'British': 'United Kingdom',
    'Irish': 'Ireland',
    'Austrian': 'Austria',
    'Belgian': 'Belgium',
    'Bulgarian': 'Bulgaria',
    'Croatian': 'Croatia',
    'Cypriot': 'Cyprus',
    'Czech': 'Czech Republic',
    'Danish': 'Denmark',
    'Dutch': 'Netherlands',
    'Estonian': 'Estonia',
    'Finnish': 'Finland',
    'French': 'France',
    'German': 'Germany',
    'Greek': 'Greece',
    'Hungarian': 'Hungary',
    'Italian': 'Italy',
    'Latvian': 'Latvia',
    'Lithuanian': 'Lithuania',
    'Luxembourgish': 'Luxembourg',
    'Maltese': 'Malta',
    'Polish': 'Poland',
    'Portuguese': 'Portugal',
    'Romanian': 'Romania',
    'Slovak': 'Slovakia',
    'Slovenian': 'Slovenia',
    'Spanish': 'Spain',
    'Swedish': 'Sweden',
    'Icelandic': 'Iceland',
    'Liechtenstein': 'Liechtenstein',
    'Norwegian': 'Norway',
    'Swiss': 'Switzerland',
    'Albanian': 'Albania',
    'Andorran': 'Andorra',
    'Armenian': 'Armenia',
    'Azerbaijani': 'Azerbaijan',
    'Belarusian': 'Belarus',
    'Bosnian': 'Bosnia and Herzegovina',
    'Georgian': 'Georgia',
    'Kosovar': 'Kosovo',
    'Macedonian': 'North Macedonia',
    'Moldovan': 'Moldova',
    'Montenegrin': 'Montenegro',
    'Russian': 'Russia',
    'San Marinese': 'San Marino',
    'Serbian': 'Serbia',
    'Turkish': 'Turkey',
    'Ukrainian': 'Ukraine',
    'Vatican': 'Vatican City',
    'Afghan': 'Afghanistan',
    'Bangladeshi': 'Bangladesh',
    'Bhutanese': 'Bhutan',
    'Bruneian': 'Brunei',
    'Burmese': 'Myanmar',
    'Cambodian': 'Cambodia',
    'Chinese': 'China',
    'Filipino': 'Philippines',
    'Indian': 'India',
    'Indonesian': 'Indonesia',
    'Japanese': 'Japan',
    'Kazakh': 'Kazakhstan',
    'Kyrgyz': 'Kyrgyzstan',
    'Laotian': 'Laos',
    'Malaysian': 'Malaysia',
    'Maldivian': 'Maldives',
    'Mongolian': 'Mongolia',
    'Nepalese': 'Nepal',
    'North Korean': 'North Korea',
    'Pakistani': 'Pakistan',
    'Singaporean': 'Singapore',
    'South Korean': 'South Korea',
    'Sri Lankan': 'Sri Lanka',
    'Taiwanese': 'Taiwan',
    'Tajik': 'Tajikistan',
    'Thai': 'Thailand',
    'Timorese': 'Timor-Leste',
    'Turkmen': 'Turkmenistan',
    'Uzbek': 'Uzbekistan',
    'Vietnamese': 'Vietnam',
    'Bahraini': 'Bahrain',
    'Emirati': 'United Arab Emirates',
    'Iranian': 'Iran',
    'Iraqi': 'Iraq',
    'Israeli': 'Israel',
    'Jordanian': 'Jordan',
    'Kuwaiti': 'Kuwait',
    'Lebanese': 'Lebanon',
    'Omani': 'Oman',
    'Palestinian': 'Palestine',
    'Qatari': 'Qatar',
    'Saudi Arabian': 'Saudi Arabia',
    'Syrian': 'Syria',
    'Yemeni': 'Yemen',
    'Algerian': 'Algeria',
    'Angolan': 'Angola',
    'Beninese': 'Benin',
    'Botswanan': 'Botswana',
    'Burkinabe': 'Burkina Faso',
    'Burundian': 'Burundi',
    'Cameroonian': 'Cameroon',
    'Cape Verdean': 'Cape Verde',
    'Central African': 'Central African Republic',
    'Chadian': 'Chad',
    'Comorian': 'Comoros',
    'Congolese': 'Congo',
    'Djiboutian': 'Djibouti',
    'Egyptian': 'Egypt',
    'Equatorial Guinean': 'Equatorial Guinea',
    'Eritrean': 'Eritrea',
    'Eswatini': 'Eswatini',
    'Ethiopian': 'Ethiopia',
    'Gabonese': 'Gabon',
    'Gambian': 'Gambia',
    'Ghanaian': 'Ghana',
    'Guinean': 'Guinea',
    'Guinean-Bissauan': 'Guinea-Bissau',
    'Ivorian': 'Costa Rica',
    'Kenyan': 'Kenya',
    'Lesotho': 'Lesotho',
    'Liberian': 'Liberia',
    'Libyan': 'Libya',
    'Malagasy': 'Madagascar',
    'Malawian': 'Malawi',
    'Malian': 'Mali',
    'Mauritanian': 'Mauritania',
    'Mauritian': 'Mauritius',
    'Moroccan': 'Morocco',
    'Mozambican': 'Mozambique',
    'Namibian': 'Namibia',
    'Nigerian': 'Nigeria',
    'Nigerien': 'Niger',
    'Rwandan': 'Rwanda',
    'Sao Tomean': 'Sao Tome and Principe',
    'Senegalese': 'Senegal',
    'Seychellois': 'Seychelles',
    'Sierra Leonean': 'Sierra Leone',
    'Somali': 'Somalia',
    'South African': 'South Africa',
    'South Sudanese': 'South Sudan',
    'Sudanese': 'Sudan',
    'Tanzanian': 'Tanzania',
    'Togolese': 'Togo',
    'Tunisian': 'Tunisia',
    'Ugandan': 'Uganda',
    'Zambian': 'Zambia',
    'Zimbabwean': 'Zimbabwe',
    'American': 'United States',
    'Antiguan': 'Antigua and Barbuda',
    'Argentine': 'Argentina',
    'Bahamian': 'Bahamas',
    'Barbadian': 'Barbados',
    'Belizean': 'Belize',
    'Bolivian': 'Bolivia',
    'Brazilian': 'Brazil',
    'Canadian': 'Canada',
    'Chilean': 'Chile',
    'Colombian': 'Colombia',
    'Costa Rican': 'Costa Rica',
    'Cuban': 'Cuba',
    'Dominican': 'Dominican Republic',
    'Ecuadorian': 'Ecuador',
    'Salvadoran': 'El Salvador',
    'Grenadian': 'Grenada',
    'Guatemalan': 'Guatemala',
    'Guyanese': 'Guyana',
    'Haitian': 'Haiti',
    'Honduran': 'Honduras',
    'Jamaican': 'Jamaica',
    'Mexican': 'Mexico',
    'Nicaraguan': 'Nicaragua',
    'Panamanian': 'Panama',
    'Paraguayan': 'Paraguay',
    'Peruvian': 'Peru',
    'Saint Lucian': 'Saint Lucia',
    'Surinamese': 'Suriname',
    'Trinidadian': 'Trinidad and Tobago',
    'Uruguayan': 'Uruguay',
    'Venezuelan': 'Venezuela',
    'Australian': 'Australia',
    'Fijian': 'Fiji',
    'Kiribati': 'Kiribati',
    'Marshallese': 'Marshall Islands',
    'Micronesian': 'Micronesia',
    'Nauruan': 'Nauru',
    'New Zealand': 'New Zealand',
    'Palauan': 'Palau',
    'Papua New Guinean': 'Papua New Guinea',
    'Samoan': 'Samoa',
    'Solomon Islander': 'Solomon Islands',
    'Tongan': 'Tonga',
    'Tuvaluan': 'Tuvalu',
    'Vanuatuan': 'Vanuatu'
};

// Nationality change handler with progressive logic
document.getElementById('nationality').addEventListener('change', function() {
    const nationality = this.value.trim();
    const britishNotice = document.getElementById('britishNotice');
    const visaRequired = document.getElementById('visaRequired');
    const immigrationSection = document.querySelector('.immigration-section');
    const passportCountry = document.getElementById('passport_country');

    // Auto-populate passport country using nationality to country mapping
    if (nationality && passportCountry) {
        const countryName = nationalityToCountry[nationality];
        if (countryName) {
            const option = Array.from(passportCountry.options).find(opt => opt.value === countryName);
            if (option) {
                passportCountry.value = countryName;
            }
        }
    }

    // British or Irish - hide immigration section entirely
    if (noVisaCountries.includes(nationality)) {
        if (britishNotice) britishNotice.style.display = 'block';
        if (visaRequired) visaRequired.style.display = 'none';
        if (immigrationSection) immigrationSection.style.display = 'none';
        document.getElementById('visa_type').required = false;
    }
    // EU/EEA nationals - show EU-specific options
    else if (euEeaCountries.includes(nationality)) {
        if (britishNotice) britishNotice.style.display = 'none';
        if (visaRequired) visaRequired.style.display = 'inline';
        if (immigrationSection) immigrationSection.style.display = 'block';
        document.getElementById('visa_type').required = true;

        // Pre-filter visa options to show EU Settlement Scheme first
        highlightEuOptions();
    }
    // All other nationals - show full visa options
    else if (nationality) {
        if (britishNotice) britishNotice.style.display = 'none';
        if (visaRequired) visaRequired.style.display = 'inline';
        if (immigrationSection) immigrationSection.style.display = 'block';
        document.getElementById('visa_type').required = true;
    }
});

function highlightEuOptions() {
    // This function can be enhanced to reorder or highlight EU options
    const visaSelect = document.getElementById('visa_type');
    const euGroup = visaSelect.querySelector('optgroup[label="EU Settlement Scheme"]');
    if (euGroup) {
        // Scroll to EU Settlement Scheme options
        visaSelect.focus();
    }
}

// Employment Status change handler
document.getElementById('employment_status').addEventListener('change', function() {
    const status = this.value;
    const preEmpFields = document.getElementById('preEmploymentFields');
    const rtwSection = document.querySelector('.rtw-section');

    if (status === 'pre_employment') {
        // Pre-Employment: Show CoS-related fields
        preEmpFields.classList.add('show');
        const rtwCheckType = document.getElementById('rtw_check_type');
        if (rtwCheckType) rtwCheckType.value = 'pending';
    } else if (status === 'new_hire') {
        // New Hire: Hide pre-employment fields
        preEmpFields.classList.remove('show');
        const rtwCheckType = document.getElementById('rtw_check_type');
        if (rtwCheckType && rtwCheckType.value === 'pending') {
            rtwCheckType.value = 'initial';
        }
    }
});

// CoS Location handler (for pre-employment)
document.getElementById('cos_location').addEventListener('change', function() {
    const location = this.value;
    const inCountryNotice = document.getElementById('inCountryNotice');
    const outCountryNotice = document.getElementById('outCountryNotice');
    const niField = document.querySelector('input[name="ni_number"]');
    const rtwChecked = document.getElementById('right_to_work_checked');

    if (location === 'in_country') {
        inCountryNotice.style.display = 'block';
        outCountryNotice.style.display = 'none';
        // NI and RTW should be required for in-country
        if (niField) niField.required = true;
    } else if (location === 'out_of_country') {
        inCountryNotice.style.display = 'none';
        outCountryNotice.style.display = 'block';
        // NI and RTW not required for out-of-country
        if (niField) niField.required = false;
        if (rtwChecked) rtwChecked.checked = false;
    } else {
        inCountryNotice.style.display = 'none';
        outCountryNotice.style.display = 'none';
    }
});

// Visa type change handler
document.getElementById('visa_type').addEventListener('change', function() {
    const visaType = this.value;
    const brpFields = document.getElementById('brpFields');
    const shareCodeFields = document.getElementById('shareCodeFields');
    const visaFields = document.getElementById('visaFields');
    const cosFields = document.getElementById('cosFields');

    // Hide all conditional fields
    brpFields.classList.remove('show');
    shareCodeFields.classList.remove('show');
    visaFields.classList.remove('show');
    cosFields.classList.remove('show');

    // Visa types that don't need additional immigration fields
    const noVisaNeeded = ['British Citizen', 'Irish Citizen', 'Indefinite Leave to Remain'];

    if (noVisaNeeded.includes(visaType)) {
        // Show only BRP for ILR
        if (visaType === 'Indefinite Leave to Remain') {
            brpFields.classList.add('show');
        }
        return;
    }

    // EU Settlement Scheme
    if (visaType === 'EU Settled Status' || visaType === 'EU Pre-Settled Status') {
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        return;
    }

    // Sponsored worker visas
    const sponsoredVisas = ['Skilled Worker', 'Health and Care Worker', 'Senior or Specialist Worker'];
    if (sponsoredVisas.includes(visaType)) {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
        cosFields.classList.add('show');
        return;
    }

    // All other visas show standard fields
    if (visaType && visaType !== '') {
        brpFields.classList.add('show');
        shareCodeFields.classList.add('show');
        visaFields.classList.add('show');
    }
});

// Right to Work check handler
document.getElementById('right_to_work_checked').addEventListener('change', function() {
    const rtwFields = document.getElementById('rtwFields');
    if (this.checked) {
        rtwFields.classList.add('show');
    } else {
        rtwFields.classList.remove('show');
    }
});

// Restore state if form was submitted with errors
<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['visa_type'])): ?>
document.getElementById('visa_type').dispatchEvent(new Event('change'));
<?php endif; ?>

<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['nationality'])): ?>
document.getElementById('nationality').dispatchEvent(new Event('input'));
<?php endif; ?>

<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['right_to_work_checked'])): ?>
document.getElementById('right_to_work_checked').checked = true;
document.getElementById('right_to_work_checked').dispatchEvent(new Event('change'));
<?php endif; ?>

// ========== ADDRESS MODAL FUNCTIONS ==========
let addressHistoryCount = 0;
let addressesArray = [];
let addressHistoryType = '';
let requiredYears = 0;
let yearsCovered = 0;

// Restore addresses from POST if validation failed
<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($errors) && isset($_POST['addresses_data'])): ?>
try {
    const restoredAddresses = <?php echo $_POST['addresses_data']; ?>;
    if (restoredAddresses && Array.isArray(restoredAddresses)) {
        addressesArray = restoredAddresses;
        let addressHistoryCount = 1;
        addressesArray.forEach(addr => {
            addr.id = addressHistoryCount++;
            addAddressCard(addr);
        });
    }
} catch(e) {
    console.error('Error restoring addresses:', e);
}
<?php endif; ?>

function openAddressModal() {
    const modal = new bootstrap.Modal(document.getElementById('addressModal'));
    modal.show();
    resetAddressModal();
}

function resetAddressModal() {
    document.getElementById('addressTypeSelection').style.display = 'block';
    document.getElementById('addressEntryForm').style.display = 'none';
    document.getElementById('saveAddressBtn').style.display = 'none';
    document.getElementById('progressIndicator').style.display = 'none';

    const radios = document.querySelectorAll('input[name="addressHistoryType"]');
    radios.forEach(radio => radio.checked = false);

    document.getElementById('addressForm').reset();
    document.getElementById('currentAddressCheck').checked = false;
    document.getElementById('modal_move_out_date').disabled = false;
    document.getElementById('modal_move_out_date').required = true;

    addressHistoryType = '';
    requiredYears = 0;
    yearsCovered = 0;
}

function startAddressEntry() {
    const selectedRadio = document.querySelector('input[name="addressHistoryType"]:checked');
    if (!selectedRadio) {
        alert('Please select an address history option');
        return;
    }

    addressHistoryType = selectedRadio.value;

    if (addressHistoryType === 'current') {
        requiredYears = 0;
    } else if (addressHistoryType === '3') {
        requiredYears = 3;
    } else if (addressHistoryType === '5') {
        requiredYears = 5;
    }

    if (addressHistoryType === 'current') {
        document.getElementById('currentAddressCheck').checked = true;
        toggleMoveOutDate();
    }

    document.getElementById('addressTypeSelection').style.display = 'none';
    document.getElementById('addressEntryForm').style.display = 'block';
    document.getElementById('saveAddressBtn').style.display = 'block';

    // Show back button when entering addresses (after type selection)
    document.getElementById('backAddressBtn').style.display = 'inline-block';

    if (requiredYears > 0) {
        document.getElementById('progressIndicator').style.display = 'block';
        updateProgressIndicator();
    }
}

// Back button - return to address type selection and keep existing addresses
function goBackInModal() {
    document.getElementById('addressEntryForm').style.display = 'none';
    document.getElementById('addressTypeSelection').style.display = 'block';
    document.getElementById('saveAddressBtn').style.display = 'none';
    document.getElementById('backAddressBtn').style.display = 'none';

    // Reset the current form but keep addressesArray intact
    document.getElementById('addressForm').reset();
}

function toggleMoveOutDate() {
    const checkbox = document.getElementById('currentAddressCheck');
    const moveOutField = document.getElementById('modal_move_out_date');
    const moveOutLabel = document.getElementById('moveOutLabel');

    if (checkbox.checked) {
        moveOutField.value = new Date().toISOString().split('T')[0];
        moveOutField.disabled = true;
        moveOutField.required = false;
        moveOutLabel.innerHTML = 'Move Out Date';
    } else {
        moveOutField.value = '';
        moveOutField.disabled = false;
        moveOutField.required = true;
        moveOutLabel.innerHTML = 'Move Out Date <span class="text-danger">*</span>';
    }
}

function saveAddress() {
    const addressLine1 = document.getElementById('modal_address_line1').value.trim();
    const addressLine2 = document.getElementById('modal_address_line2').value.trim();
    const city = document.getElementById('modal_city').value.trim();
    const postcode = document.getElementById('modal_postcode').value.trim();
    const country = document.getElementById('modal_country').value.trim();
    const moveInDate = document.getElementById('modal_move_in_date').value;
    const moveOutDate = document.getElementById('modal_move_out_date').value;
    const isCurrent = document.getElementById('currentAddressCheck').checked;

    if (!addressLine1 || !moveInDate) {
        alert('Please fill in all required fields');
        return;
    }

    if (!isCurrent && !moveOutDate) {
        alert('Please provide Move Out Date or check "This is my current address"');
        return;
    }

    const moveIn = new Date(moveInDate);
    const moveOut = new Date(moveOutDate);

    if (moveOut < moveIn) {
        alert('Move Out Date cannot be before Move In Date');
        return;
    }

    const overlap = checkForOverlap(moveInDate, moveOutDate);
    if (overlap) {
        alert('This address period overlaps with an existing address. Please adjust the dates.\\n\\n' + overlap);
        return;
    }

    addressHistoryCount++;
    const address = {
        id: addressHistoryCount,
        addressLine1,
        addressLine2,
        city,
        postcode,
        country,
        moveInDate,
        moveOutDate,
        isCurrent
    };

    addressesArray.push(address);
    yearsCovered = calculateTotalYearsCovered();
    addAddressCard(address);

    if (addressHistoryType === 'current') {
        closeModal();
    } else if (yearsCovered >= requiredYears) {
        alert(`Address history complete! You've provided ${yearsCovered.toFixed(1)} years of address history.`);
        closeModal();
    } else {
        clearFormForNextAddress();
        updateProgressIndicator();
        document.getElementById('saveButtonText').textContent = 'Add Next Address';
    }
}

function checkForOverlap(newMoveIn, newMoveOut) {
    const newStart = new Date(newMoveIn);
    const newEnd = new Date(newMoveOut);

    for (const addr of addressesArray) {
        const existingStart = new Date(addr.moveInDate);
        const existingEnd = new Date(addr.moveOutDate);

        if (newStart <= existingEnd && newEnd >= existingStart) {
            return `Overlaps with: ${addr.addressLine1} (${formatDate(addr.moveInDate)} - ${formatDate(addr.moveOutDate)})`;
        }
    }

    return null;
}

function calculateTotalYearsCovered() {
    if (addressesArray.length === 0) return 0;

    const sorted = [...addressesArray].sort((a, b) =>
        new Date(a.moveInDate) - new Date(b.moveInDate)
    );

    const earliest = new Date(sorted[0].moveInDate);
    const latest = new Date(sorted[sorted.length - 1].moveOutDate);

    const diffTime = Math.abs(latest - earliest);
    const diffYears = diffTime / (1000 * 60 * 60 * 24 * 365.25);

    return diffYears;
}

function updateProgressIndicator() {
    const remaining = Math.max(0, requiredYears - yearsCovered);
    const progressText = document.getElementById('progressText');

    if (yearsCovered === 0) {
        progressText.innerHTML = `You need to provide <strong>${requiredYears} years</strong> of address history.`;
    } else if (remaining > 0) {
        progressText.innerHTML = `<strong>${yearsCovered.toFixed(1)} years</strong> covered. <strong>${remaining.toFixed(1)} years</strong> remaining.`;
    } else {
        progressText.innerHTML = `<strong>Complete!</strong> ${yearsCovered.toFixed(1)} years covered.`;
    }
}

function clearFormForNextAddress() {
    document.getElementById('modal_address_line1').value = '';
    document.getElementById('modal_address_line2').value = '';
    document.getElementById('modal_city').value = '';
    document.getElementById('modal_postcode').value = '';
    document.getElementById('modal_country').value = 'United Kingdom';
    document.getElementById('modal_move_in_date').value = '';
    document.getElementById('modal_move_out_date').value = '';

    document.getElementById('currentAddressCheck').checked = false;
    toggleMoveOutDate();
}

function closeModal() {
    const modal = bootstrap.Modal.getInstance(document.getElementById('addressModal'));
    modal.hide();
    resetAddressModal();
}

function addAddressCard(address) {
    const container = document.getElementById('addressHistoryBoxes');

    const today = new Date().toISOString().split('T')[0];
    const isCurrentAddr = address.isCurrent || address.moveOutDate === today;
    const dateRangeText = isCurrentAddr ?
        `${formatDate(address.moveInDate)} - Present` :
        `${formatDate(address.moveInDate)} - ${formatDate(address.moveOutDate)}`;

    const cardHTML = `
        <div class="address-card-box" id="address-card-${address.id}" data-move-out="${address.moveOutDate}">
            <button type="button" class="btn btn-danger btn-sm remove-btn" onclick="removeAddress(${address.id})">
                <i class="bi bi-x"></i>
            </button>

            ${isCurrentAddr ? '<span class="badge bg-success mb-2">Current Address</span>' : ''}

            <strong>${address.addressLine1}</strong>
            ${address.addressLine2 ? `<small>${address.addressLine2}</small>` : ''}
            <small>${address.city ? address.city + ', ' : ''}${address.postcode}</small>
            <small>${address.country}</small>

            <span class="date-range">
                ${dateRangeText}
            </span>

            <input type="hidden" name="address_history[${address.id}][address_line1]" value="${escapeHtml(address.addressLine1)}">
            <input type="hidden" name="address_history[${address.id}][address_line2]" value="${escapeHtml(address.addressLine2)}">
            <input type="hidden" name="address_history[${address.id}][city]" value="${escapeHtml(address.city)}">
            <input type="hidden" name="address_history[${address.id}][postcode]" value="${escapeHtml(address.postcode)}">
            <input type="hidden" name="address_history[${address.id}][country]" value="${escapeHtml(address.country)}">
            <input type="hidden" name="address_history[${address.id}][move_in_date]" value="${address.moveInDate}">
            <input type="hidden" name="address_history[${address.id}][move_out_date]" value="${address.moveOutDate}">
        </div>
    `;

    const plusButton = container.querySelector('.address-add-box');
    const existingCards = Array.from(container.querySelectorAll('.address-card-box'));

    existingCards.push({ html: cardHTML, moveOutDate: new Date(address.moveOutDate) });
    existingCards.sort((a, b) => b.moveOutDate - a.moveOutDate);

    container.querySelectorAll('.address-card-box').forEach(card => card.remove());

    existingCards.forEach(card => {
        if (card.html) {
            plusButton.insertAdjacentHTML('beforebegin', card.html);
        } else {
            plusButton.insertAdjacentElement('beforebegin', card);
        }
    });
}

function removeAddress(id) {
    addressesArray = addressesArray.filter(addr => addr.id !== id);

    const card = document.getElementById(`address-card-${id}`);
    if (card) {
        card.remove();
    }

    yearsCovered = calculateTotalYearsCovered();
}

function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
    return `${months[date.getMonth()]} ${date.getFullYear()}`;
}

function escapeHtml(text) {
    if (!text) return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

// ========== SALARY CALCULATIONS ==========
const annualSalaryField = document.getElementById('annual_salary');
const hourlyRateField = document.getElementById('hourly_rate');
const hoursPerWeekField = document.getElementById('work_hours_per_week');

// Calculate hourly rate from annual salary and hours
annualSalaryField.addEventListener('input', function() {
    const annual = parseFloat(this.value);
    const hours = parseFloat(hoursPerWeekField.value);

    if (annual && hours) {
        // Annual to hourly: annual / (hours * 52 weeks)
        const hourly = annual / (hours * 52);
        hourlyRateField.value = hourly.toFixed(2);
    }
});

// Calculate hourly rate from annual salary when hours change
hoursPerWeekField.addEventListener('input', function() {
    const annual = parseFloat(annualSalaryField.value);
    const hours = parseFloat(this.value);

    if (annual && hours) {
        const hourly = annual / (hours * 52);
        hourlyRateField.value = hourly.toFixed(2);
    } else if (!annual && hours) {
        // If no annual but has hourly, calculate annual
        const hourly = parseFloat(hourlyRateField.value);
        if (hourly) {
            const annual = hourly * hours * 52;
            annualSalaryField.value = annual.toFixed(2);
        }
    }
});

// Calculate annual salary from hourly rate and hours
hourlyRateField.addEventListener('input', function() {
    const hourly = parseFloat(this.value);
    const hours = parseFloat(hoursPerWeekField.value);

    if (hourly && hours) {
        // Hourly to annual: hourly * hours * 52 weeks
        const annual = hourly * hours * 52;
        annualSalaryField.value = annual.toFixed(2);
    }
});

// ========== HIGHLIGHT ERROR FIELDS FROM SERVER VALIDATION ==========
<?php if (!empty($error_fields)): ?>
document.addEventListener('DOMContentLoaded', function() {
    const errorFields = <?php echo json_encode($error_fields); ?>;
    errorFields.forEach(fieldName => {
        const field = document.querySelector('[name="' + fieldName + '"]');
        if (field) {
            field.classList.add('is-invalid');
            field.style.borderColor = '#dc3545';
            field.style.boxShadow = '0 0 0 0.2rem rgba(220, 53, 69, 0.25)';
        }
    });

    // Scroll to first error field
    if (errorFields.length > 0) {
        const firstField = document.querySelector('[name="' + errorFields[0] + '"]');
        if (firstField) {
            firstField.scrollIntoView({ behavior: 'smooth', block: 'center' });
            firstField.focus();
        }
    }
});
<?php endif; ?>

// ========== FORM VALIDATION WITH ERROR HIGHLIGHTING ==========
document.getElementById('workerForm').addEventListener('submit', function(e) {
    // Update addresses hidden input with current addresses array
    document.getElementById('addresses_data_input').value = JSON.stringify(addressesArray);

    // Update location assignments hidden input if locations module is active
    <?php if ($locations_module_active): ?>
    if (typeof locationAssignments !== 'undefined' && typeof updateHiddenInput === 'function') {
        updateHiddenInput();
    }
    <?php endif; ?>

    const requiredFields = this.querySelectorAll('[required]');
    let firstErrorField = null;

    // Remove previous error highlights
    document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));

    // Check all required fields
    requiredFields.forEach(field => {
        if (!field.value.trim() && field.offsetParent !== null) { // offsetParent check if visible
            field.classList.add('is-invalid');
            if (!firstErrorField) {
                firstErrorField = field;
            }
        }
    });

    // If there are errors, scroll to first error and prevent submission
    if (firstErrorField) {
        e.preventDefault();

        // Scroll to the first error field
        firstErrorField.scrollIntoView({
            behavior: 'smooth',
            block: 'center'
        });

        // Focus on the field after scroll
        setTimeout(() => {
            firstErrorField.focus();
        }, 500);

        // Show alert
        alert('Please fill in all required fields. The first missing field has been highlighted.');
    }
});

// Add CSS for invalid field highlighting
const style = document.createElement('style');
style.textContent = `
    .is-invalid {
        border-color: #dc3545 !important;
        box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.25) !important;
    }
    .is-invalid:focus {
        border-color: #dc3545 !important;
        box-shadow: 0 0 0 0.2rem rgba(220, 53, 69, 0.5) !important;
    }
`;
document.head.appendChild(style);
</script>

<?php if ($locations_module_active): ?>
<script>
// Location Assignment Management
let locationAssignments = [];
let assignmentCounter = 0;

// Restore location assignments from POST if validation failed
<?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($errors) && isset($_POST['location_assignments'])): ?>
try {
    const restoredAssignments = <?php echo $_POST['location_assignments']; ?>;
    if (restoredAssignments && Array.isArray(restoredAssignments)) {
        restoredAssignments.forEach(assignment => {
            addLocationAssignment(assignment);
        });
    }
} catch(e) {
    console.error('Error restoring location assignments:', e);
}
<?php endif; ?>

// All locations and shift patterns data from PHP
const allLocations = <?php echo json_encode($work_locations ?? []); ?>;
const shiftPatternsByLocation = <?php echo json_encode($shift_patterns_by_location ?? []); ?>;

// Add one default assignment on page load
document.addEventListener('DOMContentLoaded', function() {
    addLocationAssignment();
});

function addLocationAssignment(data = {}) {
    const id = assignmentCounter++;
    const assignment = {
        id: id,
        location_id: data.location_id || '',
        shift_pattern_id: data.shift_pattern_id || '',
        days_of_week: data.days_of_week || 'Mon,Tue,Wed,Thu,Fri',
        start_date: data.start_date || '',
        end_date: data.end_date || '',
        notes: data.notes || ''
    };

    locationAssignments.push(assignment);
    renderLocationAssignments();
}

function removeLocationAssignment(id) {
    locationAssignments = locationAssignments.filter(a => a.id !== id);
    renderLocationAssignments();
}

function updateLocationAssignment(id, field, value) {
    const assignment = locationAssignments.find(a => a.id === id);
    if (assignment) {
        assignment[field] = value;
        if (field === 'location_id') {
            assignment.shift_pattern_id = '';
            renderLocationAssignments();
        }
        updateHiddenInput();
    }
}

function renderLocationAssignments() {
    const container = document.getElementById('location-assignments-container');
    container.innerHTML = '';

    locationAssignments.forEach((assignment, index) => {
        const card = document.createElement('div');
        card.className = 'card mb-3';
        const shiftOptions = assignment.location_id && shiftPatternsByLocation[assignment.location_id] ?
            shiftPatternsByLocation[assignment.location_id].map(shift =>
                '<option value="' + shift.id + '"' + (assignment.shift_pattern_id == shift.id ? ' selected' : '') + '>' +
                escapeHtml(shift.shift_name) + ' (' + (shift.start_time ? shift.start_time.substring(0,5) : '') + ' - ' + (shift.end_time ? shift.end_time.substring(0,5) : '') + ')' +
                '</option>'
            ).join('') :
            '<option value="">No shifts available</option>';

        card.innerHTML = '<div class="card-body">' +
            '<div class="d-flex justify-content-between align-items-center mb-3">' +
            '<h6 class="mb-0"><i class="bi bi-geo-alt-fill text-primary"></i> Location Assignment ' + (index + 1) +
            (index === 0 ? ' <span class="badge bg-success ms-2">Primary</span>' : '') + '</h6>' +
            '<button type="button" class="btn btn-sm btn-outline-danger" onclick="removeLocationAssignment(' + assignment.id + ')">' +
            '<i class="bi bi-trash"></i> Remove</button></div>' +
            '<div class="row"><div class="col-md-6 mb-3">' +
            '<label class="form-label">Location <span class="text-danger">*</span></label>' +
            '<select class="form-select" onchange="updateLocationAssignment(' + assignment.id + ', \'location_id\', this.value)" required>' +
            '<option value="">Select location...</option>' +
            allLocations.map(loc =>
                '<option value="' + loc.id + '"' + (assignment.location_id == loc.id ? ' selected' : '') + '>' +
                escapeHtml(loc.name) + (loc.type ? ' (' + loc.type + ')' : '') + '</option>'
            ).join('') +
            '</select></div><div class="col-md-6 mb-3">' +
            '<label class="form-label">Shift Pattern <span class="text-danger">*</span></label>' +
            '<select class="form-select" onchange="updateLocationAssignment(' + assignment.id + ', \'shift_pattern_id\', this.value)"' +
            (!assignment.location_id ? ' disabled' : ' required') + '>' +
            '<option value="">Select shift...</option>' + shiftOptions + '</select></div></div>' +
            '<div class="row"><div class="col-md-12 mb-3"><label class="form-label">Days of Week <span class="text-danger">*</span></label>' +
            '<div class="btn-group w-100" role="group">' +
            ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'].map(day =>
                '<input type="checkbox" class="btn-check" id="day_' + assignment.id + '_' + day + '"' +
                (assignment.days_of_week.includes(day) ? ' checked' : '') +
                ' onchange="updateDaysOfWeek(' + assignment.id + ')">' +
                '<label class="btn btn-outline-primary btn-sm" for="day_' + assignment.id + '_' + day + '">' + day + '</label>'
            ).join('') + '</div></div></div>' +
            '<div class="row"><div class="col-md-6 mb-3"><label class="form-label">Start Date</label>' +
            '<input type="date" class="form-control" value="' + (assignment.start_date || '') + '" onchange="updateLocationAssignment(' + assignment.id + ', \'start_date\', this.value)">' +
            '</div><div class="col-md-6 mb-3"><label class="form-label">End Date</label>' +
            '<input type="date" class="form-control" value="' + (assignment.end_date || '') + '" onchange="updateLocationAssignment(' + assignment.id + ', \'end_date\', this.value)">' +
            '<small class="text-muted">Leave empty for ongoing</small></div></div></div>';
        container.appendChild(card);
    });
    updateHiddenInput();
}

function updateDaysOfWeek(assignmentId) {
    const days = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    const selected = days.filter(day => document.getElementById('day_' + assignmentId + '_' + day).checked);
    updateLocationAssignment(assignmentId, 'days_of_week', selected.join(','));
}

function updateHiddenInput() {
    const data = locationAssignments.map(({id, ...rest}) => {
        // Clean up invalid dates - convert "0000-00-00" or empty to null
        const cleanData = {...rest};
        if (!cleanData.start_date || cleanData.start_date === '0000-00-00') {
            cleanData.start_date = null;
        }
        if (!cleanData.end_date || cleanData.end_date === '0000-00-00') {
            cleanData.end_date = null;
        }
        return cleanData;
    });
    document.getElementById('location_assignments_input').value = JSON.stringify(data);
    if (locationAssignments.length > 0 && locationAssignments[0].location_id) {
        document.getElementById('primary_location_id').value = locationAssignments[0].location_id;
    }
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Helper function to get shift details by ID
function getShiftDetails(shiftPatternId) {
    for (let locId in shiftPatternsByLocation) {
        const shift = shiftPatternsByLocation[locId].find(s => s.id == shiftPatternId);
        if (shift) return shift;
    }
    return null;
}

// Helper function to convert time to minutes for comparison
function timeToMinutes(timeStr) {
    if (!timeStr) return 0;
    const [hours, minutes] = timeStr.split(':').map(Number);
    return hours * 60 + minutes;
}

// Helper function to check if time ranges overlap
function timeRangesOverlap(start1, end1, start2, end2) {
    const s1 = timeToMinutes(start1);
    let e1 = timeToMinutes(end1);
    const s2 = timeToMinutes(start2);
    let e2 = timeToMinutes(end2);

    // Handle overnight shifts (end time < start time)
    if (e1 < s1) e1 += 24 * 60;
    if (e2 < s2) e2 += 24 * 60;

    return (s1 < e2 && e1 > s2);
}

// Check if shift is night shift (ends after midnight or labeled as night)
function isNightShift(shift) {
    if (!shift) return false;

    // Check shift type
    if (shift.shift_type && shift.shift_type.toLowerCase().includes('night')) {
        return true;
    }

    // Check if shift ends after midnight (end time < start time)
    const startMinutes = timeToMinutes(shift.start_time);
    const endMinutes = timeToMinutes(shift.end_time);
    if (endMinutes < startMinutes) {
        return true;
    }

    // Check if shift ends late (after 11 PM) or starts before 6 AM
    if (endMinutes >= 23 * 60 || startMinutes < 6 * 60) {
        return true;
    }

    return false;
}

// Check if shift is morning shift
function isMorningShift(shift) {
    if (!shift) return false;

    // Check shift type
    if (shift.shift_type && shift.shift_type.toLowerCase().includes('morning')) {
        return true;
    }

    // Check if shift starts before 10 AM
    const startMinutes = timeToMinutes(shift.start_time);
    if (startMinutes < 10 * 60) {
        return true;
    }

    return false;
}

// Validate shift assignments
function validateShiftAssignments() {
    const errors = [];

    // Build a map of day -> shift assignments
    const dayAssignments = {}; // { 'Mon': [{assignment, shift}, ...], ... }

    for (let assignment of locationAssignments) {
        if (!assignment.location_id || !assignment.shift_pattern_id || !assignment.days_of_week) {
            continue; // Will be caught by basic validation
        }

        const shift = getShiftDetails(assignment.shift_pattern_id);
        if (!shift) continue;

        const days = assignment.days_of_week.split(',').map(d => d.trim()).filter(d => d);

        for (let day of days) {
            if (!dayAssignments[day]) {
                dayAssignments[day] = [];
            }
            dayAssignments[day].push({
                assignment: assignment,
                shift: shift,
                day: day
            });
        }
    }

    // Check for overlaps on same day
    for (let day in dayAssignments) {
        const assignments = dayAssignments[day];

        for (let i = 0; i < assignments.length; i++) {
            for (let j = i + 1; j < assignments.length; j++) {
                const a1 = assignments[i];
                const a2 = assignments[j];

                if (timeRangesOverlap(a1.shift.start_time, a1.shift.end_time,
                                     a2.shift.start_time, a2.shift.end_time)) {
                    const loc1 = allLocations.find(l => l.id == a1.assignment.location_id);
                    const loc2 = allLocations.find(l => l.id == a2.assignment.location_id);
                    errors.push(
                        day + ': Shift overlap detected!\n' +
                        '• ' + (loc1 ? loc1.name : 'Location') + ' - ' + a1.shift.shift_name +
                        ' (' + a1.shift.start_time.substring(0,5) + '-' + a1.shift.end_time.substring(0,5) + ')\n' +
                        '• ' + (loc2 ? loc2.name : 'Location') + ' - ' + a2.shift.shift_name +
                        ' (' + a2.shift.start_time.substring(0,5) + '-' + a2.shift.end_time.substring(0,5) + ')'
                    );
                }
            }
        }
    }

    // Check for night shift → morning shift rule
    const allDays = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    for (let i = 0; i < allDays.length; i++) {
        const currentDay = allDays[i];
        const nextDay = allDays[(i + 1) % 7];

        const currentDayAssignments = dayAssignments[currentDay] || [];
        const nextDayAssignments = dayAssignments[nextDay] || [];

        for (let currentAssignment of currentDayAssignments) {
            if (isNightShift(currentAssignment.shift)) {
                for (let nextAssignment of nextDayAssignments) {
                    if (isMorningShift(nextAssignment.shift)) {
                        const loc1 = allLocations.find(l => l.id == currentAssignment.assignment.location_id);
                        const loc2 = allLocations.find(l => l.id == nextAssignment.assignment.location_id);
                        errors.push(
                            currentDay + ' → ' + nextDay + ': Invalid consecutive shifts!\n' +
                            'Worker cannot do morning shift after night shift.\n' +
                            '• ' + currentDay + ': ' + (loc1 ? loc1.name : 'Location') + ' - ' +
                            currentAssignment.shift.shift_name + ' (Night)\n' +
                            '• ' + nextDay + ': ' + (loc2 ? loc2.name : 'Location') + ' - ' +
                            nextAssignment.shift.shift_name + ' (Morning)\n' +
                            'Suggestion: Assign another night shift or afternoon/evening shift for ' + nextDay
                        );
                    }
                }
            }
        }
    }

    return errors;
}

// Validate before form submission
const mainForm = document.querySelector('form');
mainForm.addEventListener('submit', function(e) {
    if (locationAssignments.length > 0) {
        // Basic validation
        for (let assignment of locationAssignments) {
            if (!assignment.location_id || !assignment.shift_pattern_id || !assignment.days_of_week) {
                e.preventDefault();
                alert('Please complete all required fields for location assignments');
                return false;
            }
        }

        // Advanced validation: overlaps and consecutive shifts
        const errors = validateShiftAssignments();
        if (errors.length > 0) {
            e.preventDefault();
            alert('Shift Assignment Errors:\n\n' + errors.join('\n\n---\n\n'));
            return false;
        }
    }
    updateHiddenInput();
});
</script>
<?php endif; ?>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
