<?php
/**
 * Worker Activation Modal
 * Allows activation of pre_employment and pending_rtw workers
 */

require_once '../../config/database.php';
require_once '../../includes/database.php';

$pdo = get_database_connection();

// Get worker ID
$worker_id = $_GET['id'] ?? null;

if (!$worker_id) {
    die('Worker ID required');
}

// Fetch worker data
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
$stmt->execute([$worker_id]);
$worker = $stmt->fetch();

if (!$worker) {
    die('Worker not found');
}

// Check if worker is already active
if ($worker['status'] === 'active') {
    die('Worker is already active');
}

// Check if RTW module is active and fetch latest RTW check
$rtw_active = is_module_active('rtw');
$latest_rtw = null;

if ($rtw_active) {
    $rtw_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "worker_rtw_checks
                               WHERE worker_id = ?
                               ORDER BY check_date DESC LIMIT 1");
    $rtw_stmt->execute([$worker_id]);
    $latest_rtw = $rtw_stmt->fetch();
}

$is_british = (strtolower($worker['nationality']) === 'british' || strtolower($worker['nationality']) === 'irish');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Activate Worker - <?= htmlspecialchars($worker['full_name']) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
</head>
<body>
    <div class="container-fluid p-4">
        <h4 class="mb-4">
            <i class="bi bi-person-check-fill text-success"></i>
            Activate Worker: <?= htmlspecialchars($worker['full_name']) ?>
        </h4>

        <?php if ($worker['status'] === 'pending_rtw' && !$is_british): ?>
            <?php if ($latest_rtw && $latest_rtw['check_status'] === 'passed'): ?>
                <div class="alert alert-success">
                    <i class="bi bi-check-circle-fill"></i>
                    <strong>RTW Check Completed:</strong> Right to Work check passed on <?= date('d/m/Y', strtotime($latest_rtw['check_date'])) ?>
                </div>
            <?php else: ?>
                <div class="alert alert-danger">
                    <i class="bi bi-exclamation-triangle-fill"></i>
                    <strong>Warning:</strong> No valid Right to Work check found for this worker. Please complete RTW check before activation.
                    <?php if ($rtw_active): ?>
                        <br><a href="../rtw/add_check.php?worker_id=<?= $worker_id ?>" class="btn btn-sm btn-danger mt-2" target="_blank">
                            <i class="bi bi-plus-circle"></i> Perform RTW Check
                        </a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>

        <?php if ($worker['status'] === 'pre_employment'): ?>
            <div class="alert alert-info">
                <i class="bi bi-info-circle-fill"></i>
                <strong>Pre-Employment Status:</strong> This worker is currently awaiting visa. Ensure visa has been granted before activation.
            </div>
        <?php endif; ?>

        <form id="activationForm" method="POST" action="process_activation.php">
            <input type="hidden" name="worker_id" value="<?= $worker_id ?>">
            <input type="hidden" name="rtw_check_id" value="<?= $latest_rtw['check_id'] ?? '' ?>">

            <!-- Worker Information Summary -->
            <div class="card mb-3">
                <div class="card-header bg-light">
                    <h6 class="mb-0"><i class="bi bi-person-badge"></i> Worker Information</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Employee Number:</strong> <?= htmlspecialchars($worker['employee_number']) ?></p>
                            <p><strong>Full Name:</strong> <?= htmlspecialchars($worker['full_name']) ?></p>
                            <p><strong>Date of Birth:</strong> <?= $worker['date_of_birth'] ? date('d/m/Y', strtotime($worker['date_of_birth'])) : 'Not set' ?></p>
                            <p><strong>Nationality:</strong> <?= htmlspecialchars($worker['nationality']) ?></p>
                        </div>
                        <div class="col-md-6">
                            <p><strong>Job Title:</strong> <?= htmlspecialchars($worker['job_title']) ?></p>
                            <p><strong>Department:</strong> <?= htmlspecialchars($worker['department']) ?></p>
                            <p><strong>Contract Type:</strong> <?= htmlspecialchars($worker['contract_type']) ?></p>
                            <p><strong>Current Status:</strong>
                                <span class="badge <?= $worker['status'] === 'pre_employment' ? 'bg-info' : 'bg-warning text-dark' ?>">
                                    <?= ucfirst(str_replace('_', ' ', $worker['status'])) ?>
                                </span>
                            </p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Activation Details -->
            <div class="card mb-3">
                <div class="card-header bg-primary text-white">
                    <h6 class="mb-0"><i class="bi bi-calendar-check"></i> Activation Details</h6>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label for="start_date" class="form-label">
                                Employment Start Date <span class="text-danger">*</span>
                            </label>
                            <input type="date"
                                   class="form-control"
                                   id="start_date"
                                   name="start_date"
                                   value="<?= $latest_rtw ? $latest_rtw['check_date'] : date('Y-m-d') ?>"
                                   required>
                            <small class="text-muted">
                                <?php if ($latest_rtw): ?>
                                    Defaulted to RTW check date (<?= date('d/m/Y', strtotime($latest_rtw['check_date'])) ?>)
                                <?php else: ?>
                                    Set the date when employment will commence
                                <?php endif; ?>
                            </small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label for="probation_end_date" class="form-label">Probation End Date</label>
                            <input type="date"
                                   class="form-control"
                                   id="probation_end_date"
                                   name="probation_end_date"
                                   value="<?= $worker['probation_end_date'] ?>">
                            <small class="text-muted">Optional: Set probation period end date</small>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label for="activation_notes" class="form-label">Activation Notes</label>
                        <textarea class="form-control"
                                  id="activation_notes"
                                  name="activation_notes"
                                  rows="3"
                                  placeholder="Add any notes about this activation..."></textarea>
                    </div>
                </div>
            </div>

            <!-- Confirmation Checklist -->
            <div class="card mb-3">
                <div class="card-header bg-warning text-dark">
                    <h6 class="mb-0"><i class="bi bi-check2-square"></i> Activation Checklist</h6>
                </div>
                <div class="card-body">
                    <div class="form-check mb-2">
                        <input class="form-check-input" type="checkbox" id="confirm_details" required>
                        <label class="form-check-label" for="confirm_details">
                            I confirm that all worker details are accurate and up to date
                        </label>
                    </div>

                    <?php if (!$is_british): ?>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="checkbox" id="confirm_rtw" required>
                            <label class="form-check-label" for="confirm_rtw">
                                I confirm that Right to Work check has been completed and verified
                            </label>
                        </div>
                    <?php endif; ?>

                    <?php if ($worker['status'] === 'pre_employment'): ?>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="checkbox" id="confirm_visa" required>
                            <label class="form-check-label" for="confirm_visa">
                                I confirm that visa has been granted and worker is eligible to work
                            </label>
                        </div>
                    <?php endif; ?>

                    <div class="form-check">
                        <input class="form-check-input" type="checkbox" id="confirm_start" required>
                        <label class="form-check-label" for="confirm_start">
                            I confirm the start date is correct and employment will commence on this date
                        </label>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="d-flex justify-content-between">
                <button type="button" class="btn btn-secondary" onclick="window.close()">
                    <i class="bi bi-x-circle"></i> Cancel
                </button>
                <button type="submit" class="btn btn-success btn-lg">
                    <i class="bi bi-check-circle-fill"></i> Activate Worker
                </button>
            </div>
        </form>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Set minimum start date to today or RTW check date
        const startDateInput = document.getElementById('start_date');
        const probationInput = document.getElementById('probation_end_date');

        // When start date changes, auto-calculate probation end (3 months later)
        startDateInput.addEventListener('change', function() {
            if (this.value && !probationInput.value) {
                const startDate = new Date(this.value);
                startDate.setMonth(startDate.getMonth() + 3);
                probationInput.value = startDate.toISOString().split('T')[0];
            }
        });

        // Form submission confirmation
        document.getElementById('activationForm').addEventListener('submit', function(e) {
            if (!confirm('Are you sure you want to activate this worker? This will change their status to Active and set their employment start date.')) {
                e.preventDefault();
            }
        });
    </script>
</body>
</html>
