<?php
/**
 * View RTW Check Details
 * Display complete details of a specific RTW check
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(30);

$page_title = 'RTW Check Details';
$current_page = 'rtw';

$check_id = $_GET['id'] ?? '';

if (!$check_id) {
    header('Location: list.php');
    exit;
}

// Get check details
$stmt = $pdo->prepare("
    SELECT c.*,
           w.full_name as worker_name,
           w.employee_number,
           w.department,
           w.location,
           w.email as worker_email,
           w.phone as worker_phone,
           u.first_name as checker_fname,
           u.last_name as checker_lname,
           u.email as checker_email
    FROM " . DB_PREFIX . "worker_rtw_checks c
    JOIN " . DB_PREFIX . "sponsored_workers w ON c.worker_id = w.id
    LEFT JOIN " . DB_PREFIX . "users u ON c.checked_by = u.id
    WHERE c.id = ?
");
$stmt->execute([$check_id]);
$check = $stmt->fetch();

if (!$check) {
    $_SESSION['error'] = 'RTW check not found';
    header('Location: list.php');
    exit;
}

// Get related documents
$stmt = $pdo->prepare("
    SELECT d.*,
           u.first_name as uploader_fname,
           u.last_name as uploader_lname
    FROM " . DB_PREFIX . "rtw_documents d
    LEFT JOIN " . DB_PREFIX . "users u ON d.uploaded_by = u.id
    WHERE d.check_id = ?
    ORDER BY d.uploaded_at DESC
");
$stmt->execute([$check_id]);
$documents = $stmt->fetchAll();

// Get share code if used
$share_code = null;
if ($check['share_code_used']) {
    $stmt = $pdo->prepare("
        SELECT * FROM " . DB_PREFIX . "rtw_share_codes
        WHERE share_code = ? AND worker_id = ?
    ");
    $stmt->execute([$check['share_code_used'], $check['worker_id']]);
    $share_code = $stmt->fetch();
}

// Get statutory excuse if exists
$stmt = $pdo->prepare("
    SELECT * FROM " . DB_PREFIX . "rtw_statutory_excuse
    WHERE check_id = ?
");
$stmt->execute([$check_id]);
$statutory_excuse = $stmt->fetch();

// Get audit trail
$stmt = $pdo->prepare("
    SELECT a.*,
           u.first_name,
           u.last_name
    FROM " . DB_PREFIX . "rtw_audit a
    LEFT JOIN " . DB_PREFIX . "users u ON a.performed_by = u.id
    WHERE a.check_id = ?
    ORDER BY a.created_at DESC
");
$stmt->execute([$check_id]);
$audit_trail = $stmt->fetchAll();

// Get previous checks for this worker
$stmt = $pdo->prepare("
    SELECT id, check_date, check_type, outcome, next_check_due
    FROM " . DB_PREFIX . "worker_rtw_checks
    WHERE worker_id = ? AND id != ?
    ORDER BY check_date DESC
    LIMIT 5
");
$stmt->execute([$check['worker_id'], $check_id]);
$previous_checks = $stmt->fetchAll();

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <!-- Header -->
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-file-earmark-check"></i> RTW Check Details
            </h1>
            <p class="welcome-subtitle">
                <?php echo htmlspecialchars($check['worker_name']); ?> -
                <?php echo date('d/m/Y', strtotime($check['check_date'])); ?>
            </p>
        </div>
        <div class="welcome-actions">
            <a href="document_upload.php?check_id=<?php echo $check_id; ?>" class="btn btn-primary">
                <i class="bi bi-upload"></i> Upload Documents
            </a>
            <a href="list.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
    </div>

    <div class="row">
        <!-- Main Details -->
        <div class="col-lg-8">
            <!-- Check Information -->
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-info-circle"></i> Check Information
                    </h5>
                </div>
                <div class="card-body-modern">
                    <div class="row">
                        <div class="col-md-6">
                            <table class="table table-sm">
                                <tr>
                                    <th width="40%">Check ID:</th>
                                    <td><code>#<?php echo str_pad($check['id'], 6, '0', STR_PAD_LEFT); ?></code></td>
                                </tr>
                                <tr>
                                    <th>Check Date:</th>
                                    <td><?php echo date('d/m/Y', strtotime($check['check_date'])); ?></td>
                                </tr>
                                <tr>
                                    <th>Check Type:</th>
                                    <td>
                                        <span class="badge bg-secondary">
                                            <?php echo ucfirst(str_replace('_', ' ', $check['check_type'])); ?>
                                        </span>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Check Method:</th>
                                    <td>
                                        <?php if ($check['online_check']): ?>
                                            <span class="badge bg-info">Online</span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary">Manual</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            </table>
                        </div>
                        <div class="col-md-6">
                            <table class="table table-sm">
                                <tr>
                                    <th width="40%">Outcome:</th>
                                    <td>
                                        <?php
                                        $outcome_colors = ['pass' => 'success', 'fail' => 'danger', 'refer' => 'warning', 'pending' => 'secondary'];
                                        $color = $outcome_colors[$check['outcome']] ?? 'secondary';
                                        ?>
                                        <span class="badge bg-<?php echo $color; ?>">
                                            <?php echo ucfirst($check['outcome']); ?>
                                        </span>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Next Check Due:</th>
                                    <td>
                                        <?php if ($check['next_check_due']): ?>
                                            <?php echo date('d/m/Y', strtotime($check['next_check_due'])); ?>
                                            <?php
                                            $days_until = floor((strtotime($check['next_check_due']) - time()) / 86400);
                                            if ($days_until < 0): ?>
                                                <span class="badge bg-danger">Overdue</span>
                                            <?php elseif ($days_until <= 30): ?>
                                                <span class="badge bg-warning"><?php echo $days_until; ?> days</span>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="text-muted">Not required</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <tr>
                                    <th>Checked By:</th>
                                    <td>
                                        <?php echo htmlspecialchars($check['checker_fname'] . ' ' . $check['checker_lname']); ?><br>
                                        <small class="text-muted"><?php echo htmlspecialchars($check['checker_email']); ?></small>
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>

                    <!-- Documents Seen -->
                    <div class="mt-3">
                        <h6><i class="bi bi-file-text"></i> Documents Verified</h6>
                        <div class="alert alert-light">
                            <?php echo nl2br(htmlspecialchars($check['documents_seen'])); ?>
                            <?php if ($check['document_type']): ?>
                                <hr>
                                <strong>Primary Document:</strong> <?php echo htmlspecialchars($check['document_type']); ?>
                                <?php if ($check['document_number']): ?>
                                    (<?php echo htmlspecialchars($check['document_number']); ?>)
                                <?php endif; ?>
                                <?php if ($check['document_expiry']): ?>
                                    <br><strong>Expires:</strong> <?php echo date('d/m/Y', strtotime($check['document_expiry'])); ?>
                                <?php endif; ?>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Share Code Info -->
                    <?php if ($share_code): ?>
                    <div class="mt-3">
                        <h6><i class="bi bi-globe"></i> Online Verification</h6>
                        <div class="alert alert-info">
                            <strong>Share Code:</strong> <code><?php echo htmlspecialchars($share_code['share_code']); ?></code><br>
                            <strong>Provided:</strong> <?php echo date('d/m/Y', strtotime($share_code['date_provided'])); ?><br>
                            <?php if ($share_code['profile_url']): ?>
                                <strong>Profile URL:</strong>
                                <a href="<?php echo htmlspecialchars($share_code['profile_url']); ?>" target="_blank">
                                    View on GOV.UK <i class="bi bi-box-arrow-up-right"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Notes -->
                    <?php if ($check['notes']): ?>
                    <div class="mt-3">
                        <h6><i class="bi bi-journal-text"></i> Notes</h6>
                        <div class="alert alert-secondary">
                            <?php echo nl2br(htmlspecialchars($check['notes'])); ?>
                        </div>
                    </div>
                    <?php endif; ?>

                    <!-- Statutory Excuse -->
                    <?php if ($statutory_excuse): ?>
                    <div class="mt-3">
                        <h6><i class="bi bi-shield-check"></i> Statutory Excuse</h6>
                        <div class="alert alert-success">
                            <strong>Established:</strong> <?php echo date('d/m/Y', strtotime($statutory_excuse['established_date'])); ?><br>
                            <strong>Valid From:</strong> <?php echo date('d/m/Y', strtotime($statutory_excuse['valid_from'])); ?><br>
                            <strong>Valid Until:</strong>
                            <?php if ($statutory_excuse['valid_until']): ?>
                                <?php echo date('d/m/Y', strtotime($statutory_excuse['valid_until'])); ?>
                            <?php else: ?>
                                Indefinite
                            <?php endif; ?><br>
                            <strong>Type:</strong> <?php echo ucfirst(str_replace('_', ' ', $statutory_excuse['excuse_type'])); ?>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Documents -->
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-files"></i> Supporting Documents
                        <a href="document_upload.php?check_id=<?php echo $check_id; ?>" class="btn btn-sm btn-primary float-end">
                            <i class="bi bi-plus"></i> Add
                        </a>
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php if (empty($documents)): ?>
                        <div class="text-center py-4">
                            <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                            <p class="text-muted mt-2">No documents uploaded</p>
                            <a href="document_upload.php?check_id=<?php echo $check_id; ?>" class="btn btn-primary">
                                <i class="bi bi-upload"></i> Upload Documents
                            </a>
                        </div>
                    <?php else: ?>
                        <div class="row g-3">
                            <?php foreach ($documents as $doc): ?>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-body">
                                        <h6 class="card-title">
                                            <?php
                                            $icon = 'bi-file-earmark';
                                            if (strpos($doc['mime_type'], 'pdf') !== false) {
                                                $icon = 'bi-file-earmark-pdf text-danger';
                                            } elseif (strpos($doc['mime_type'], 'image') !== false) {
                                                $icon = 'bi-file-earmark-image text-primary';
                                            }
                                            ?>
                                            <i class="bi <?php echo $icon; ?>"></i>
                                            <?php echo htmlspecialchars($doc['document_name']); ?>
                                        </h6>
                                        <p class="card-text">
                                            <small class="text-muted">
                                                Uploaded: <?php echo date('d/m/Y H:i', strtotime($doc['uploaded_at'])); ?><br>
                                                By: <?php echo htmlspecialchars($doc['uploader_fname'] . ' ' . $doc['uploader_lname']); ?><br>
                                                Size: <?php echo number_format($doc['file_size'] / 1024, 1); ?> KB
                                            </small>
                                        </p>
                                        <a href="../../<?php echo $doc['file_path']; ?>" target="_blank" class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-eye"></i> View
                                        </a>
                                        <?php if ($doc['verified']): ?>
                                            <span class="badge bg-success float-end">
                                                <i class="bi bi-check-circle"></i> Verified
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Sidebar -->
        <div class="col-lg-4">
            <!-- Worker Information -->
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-person"></i> Worker Information
                    </h5>
                </div>
                <div class="card-body-modern">
                    <table class="table table-sm">
                        <tr>
                            <th>Name:</th>
                            <td><?php echo htmlspecialchars($check['worker_name']); ?></td>
                        </tr>
                        <tr>
                            <th>Employee #:</th>
                            <td><?php echo htmlspecialchars($check['employee_number']); ?></td>
                        </tr>
                        <tr>
                            <th>Department:</th>
                            <td><?php echo htmlspecialchars($check['department'] ?: '-'); ?></td>
                        </tr>
                        <tr>
                            <th>Location:</th>
                            <td><?php echo htmlspecialchars($check['location'] ?: '-'); ?></td>
                        </tr>
                        <tr>
                            <th>Email:</th>
                            <td>
                                <?php if ($check['worker_email']): ?>
                                    <a href="mailto:<?php echo $check['worker_email']; ?>">
                                        <?php echo htmlspecialchars($check['worker_email']); ?>
                                    </a>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Phone:</th>
                            <td><?php echo htmlspecialchars($check['worker_phone'] ?: '-'); ?></td>
                        </tr>
                    </table>
                    <hr>
                    <a href="../workers/view.php?id=<?php echo $check['worker_id']; ?>" class="btn btn-sm btn-outline-primary w-100">
                        <i class="bi bi-eye"></i> View Full Profile
                    </a>
                </div>
            </div>

            <!-- Previous Checks -->
            <?php if (!empty($previous_checks)): ?>
            <div class="modern-card mb-4">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-clock-history"></i> Check History
                    </h5>
                </div>
                <div class="card-body-modern">
                    <div class="list-group list-group-flush">
                        <?php foreach ($previous_checks as $prev): ?>
                        <a href="view_check.php?id=<?php echo $prev['id']; ?>" class="list-group-item list-group-item-action">
                            <div class="d-flex justify-content-between">
                                <div>
                                    <strong><?php echo date('d/m/Y', strtotime($prev['check_date'])); ?></strong><br>
                                    <small><?php echo ucfirst($prev['check_type']); ?></small>
                                </div>
                                <div class="text-end">
                                    <?php
                                    $color = $outcome_colors[$prev['outcome']] ?? 'secondary';
                                    ?>
                                    <span class="badge bg-<?php echo $color; ?>">
                                        <?php echo ucfirst($prev['outcome']); ?>
                                    </span>
                                </div>
                            </div>
                        </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Actions -->
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-gear"></i> Actions
                    </h5>
                </div>
                <div class="card-body-modern">
                    <div class="d-grid gap-2">
                        <a href="add_check.php?worker_id=<?php echo $check['worker_id']; ?>" class="btn btn-primary">
                            <i class="bi bi-plus-circle"></i> New Check
                        </a>
                        <a href="document_upload.php?check_id=<?php echo $check_id; ?>" class="btn btn-secondary">
                            <i class="bi bi-upload"></i> Upload Documents
                        </a>
                        <button onclick="window.print()" class="btn btn-outline-secondary">
                            <i class="bi bi-printer"></i> Print Details
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Audit Trail -->
    <?php if (!empty($audit_trail)): ?>
    <div class="modern-card mt-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-shield-lock"></i> Audit Trail
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-sm">
                    <thead>
                        <tr>
                            <th>Date/Time</th>
                            <th>Action</th>
                            <th>User</th>
                            <th>Details</th>
                            <th>IP Address</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($audit_trail as $audit): ?>
                        <tr>
                            <td><?php echo date('d/m/Y H:i', strtotime($audit['created_at'])); ?></td>
                            <td><?php echo htmlspecialchars($audit['action']); ?></td>
                            <td><?php echo htmlspecialchars($audit['first_name'] . ' ' . $audit['last_name']); ?></td>
                            <td><?php echo htmlspecialchars($audit['details'] ?: '-'); ?></td>
                            <td><code><?php echo htmlspecialchars($audit['ip_address']); ?></code></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<style>
@media print {
    .welcome-actions, .btn, .btn-group {
        display: none !important;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>