<?php
/**
 * Share Code Verification Dashboard for HR
 * Allows HR to review and verify submitted share codes
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_once 'includes/alert_helper.php';
require_authentication();

// Initialize alert helper
$alertHelper = new RTWAlertHelper($pdo);

$page_title = 'Verify Share Codes';
$current_page = 'rtw';
$message = '';
$message_type = 'success';

// Handle verification actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $submission_id = $_POST['submission_id'] ?? 0;

    if ($action === 'verify' && $submission_id) {
        try {
            // Update submission status
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "rtw_share_code_submissions
                                  SET submission_status = 'verified',
                                      verified_by = ?,
                                      verified_at = NOW(),
                                      verification_notes = ?,
                                      expires_at = ?
                                  WHERE id = ?");

            $expires_at = $_POST['expires_at'] ?? date('Y-m-d', strtotime('+1 year'));
            $notes = $_POST['verification_notes'] ?? '';

            $stmt->execute([$_SESSION['user_id'], $notes, $expires_at, $submission_id]);

            // Get worker details
            $stmt = $pdo->prepare("SELECT s.*, w.full_name, w.email
                                  FROM " . DB_PREFIX . "rtw_share_code_submissions s
                                  LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON s.worker_id = w.id
                                  WHERE s.id = ?");
            $stmt->execute([$submission_id]);
            $submission = $stmt->fetch(PDO::FETCH_ASSOC);

            // Create RTW check record
            if ($_POST['create_rtw_check'] ?? false) {
                $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_rtw_checks
                                      (worker_id, check_date, check_type, outcome,
                                       document_type, document_number, document_expiry,
                                       next_check_due, documents_seen, notes, checked_by, created_at)
                                      VALUES (?, NOW(), 'online_check', 'pass', 'share_code', ?,
                                              ?, ?, 'Share Code: ' || ?, ?, ?, NOW())");

                $stmt->execute([
                    $submission['worker_id'],
                    $submission['share_code'],
                    $expires_at,
                    $expires_at,
                    $submission['share_code'],
                    'Share code verified: ' . $notes,
                    $_SESSION['user_id']
                ]);
            }

            // Log activity and resolve alerts
            $alertHelper->logActivity('share_code_verified',
                "Share code verified for {$submission['full_name']} ({$submission['share_code']})",
                $submission['worker_id'], $submission_id);

            // Resolve any pending alerts for this submission
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "rtw_alerts
                                  SET status = 'resolved', resolved_at = NOW(), resolved_by = ?
                                  WHERE submission_id = ? AND status = 'active'");
            $stmt->execute([$_SESSION['user_id'], $submission_id]);

            $message = "Share code verified successfully for " . $submission['full_name'];
            $message_type = 'success';

        } catch (PDOException $e) {
            $message = "Error verifying share code: " . $e->getMessage();
            $message_type = 'danger';
        }
    }

    elseif ($action === 'reject' && $submission_id) {
        try {
            $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "rtw_share_code_submissions
                                  SET submission_status = 'rejected',
                                      verified_by = ?,
                                      verified_at = NOW(),
                                      verification_notes = ?
                                  WHERE id = ?");

            $notes = $_POST['rejection_reason'] ?? 'Invalid or expired share code';
            $stmt->execute([$_SESSION['user_id'], $notes, $submission_id]);

            $message = "Share code rejected";
            $message_type = 'warning';

        } catch (PDOException $e) {
            $message = "Error rejecting share code: " . $e->getMessage();
            $message_type = 'danger';
        }
    }
}

// Get filter parameters
$filter = $_GET['filter'] ?? 'pending';

// Build query based on filter
$where_clause = "1=1";
$params = [];

switch ($filter) {
    case 'pending':
        $where_clause = "s.submission_status = 'pending'";
        break;
    case 'verified':
        $where_clause = "s.submission_status = 'verified'";
        break;
    case 'rejected':
        $where_clause = "s.submission_status = 'rejected'";
        break;
    case 'expired':
        $where_clause = "s.submission_status = 'expired' OR (s.expires_at < CURDATE() AND s.submission_status = 'verified')";
        break;
}

// Get submissions
try {
    $sql = "SELECT s.*, w.full_name, w.employee_number, w.department, w.email,
                   CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as submitted_by_name,
                   CONCAT(v.first_name, ' ', COALESCE(v.last_name, '')) as verified_by_name,
                   DATEDIFF(s.expires_at, CURDATE()) as days_until_expiry
            FROM " . DB_PREFIX . "rtw_share_code_submissions s
            LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON s.worker_id = w.id
            LEFT JOIN " . DB_PREFIX . "users u ON s.submitted_by = u.id
            LEFT JOIN " . DB_PREFIX . "users v ON s.verified_by = v.id
            WHERE $where_clause
            ORDER BY s.submitted_at DESC";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $submissions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get counts for badges
    $stmt = $pdo->query("SELECT
                        COUNT(CASE WHEN submission_status = 'pending' THEN 1 END) as pending_count,
                        COUNT(CASE WHEN submission_status = 'verified' THEN 1 END) as verified_count,
                        COUNT(CASE WHEN submission_status = 'rejected' THEN 1 END) as rejected_count,
                        COUNT(CASE WHEN submission_status = 'expired' OR (expires_at < CURDATE() AND submission_status = 'verified') THEN 1 END) as expired_count
                        FROM " . DB_PREFIX . "rtw_share_code_submissions");
    $counts = $stmt->fetch(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    // Table might not exist
    $submissions = [];
    $counts = ['pending_count' => 0, 'verified_count' => 0, 'rejected_count' => 0, 'expired_count' => 0];
}

include '../../includes/header.php';
?>

<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-patch-check"></i> Verify Share Codes</h4>
            <p class="text-muted mb-0">Review and verify worker share code submissions</p>
        </div>
        <div>
            <a href="settings.php" class="btn btn-outline-secondary me-2">
                <i class="bi bi-gear"></i> Settings
            </a>
            <a href="list.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to RTW Checks
            </a>
        </div>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-<?= $message_type ?> alert-dismissible fade show">
        <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?>"></i>
        <?= htmlspecialchars($message) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Filter Tabs -->
    <ul class="nav nav-pills mb-4">
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'pending' ? 'active' : '' ?>" href="?filter=pending">
                <i class="bi bi-clock"></i> Pending
                <?php if ($counts['pending_count'] > 0): ?>
                    <span class="badge bg-warning text-dark ms-1"><?= $counts['pending_count'] ?></span>
                <?php endif; ?>
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'verified' ? 'active' : '' ?>" href="?filter=verified">
                <i class="bi bi-check-circle"></i> Verified
                <span class="badge bg-success ms-1"><?= $counts['verified_count'] ?></span>
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'rejected' ? 'active' : '' ?>" href="?filter=rejected">
                <i class="bi bi-x-circle"></i> Rejected
                <span class="badge bg-danger ms-1"><?= $counts['rejected_count'] ?></span>
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'expired' ? 'active' : '' ?>" href="?filter=expired">
                <i class="bi bi-exclamation-triangle"></i> Expired
                <?php if ($counts['expired_count'] > 0): ?>
                    <span class="badge bg-secondary ms-1"><?= $counts['expired_count'] ?></span>
                <?php endif; ?>
            </a>
        </li>
    </ul>

    <!-- Submissions List -->
    <div class="modern-card">
        <div class="card-body-modern">
            <?php if (empty($submissions)): ?>
                <div class="text-center py-5">
                    <i class="bi bi-inbox" style="font-size: 3rem; color: var(--text-muted);"></i>
                    <p class="text-muted mt-3">No share code submissions found</p>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Submitted</th>
                                <th>Worker</th>
                                <th>Share Code</th>
                                <th>Type</th>
                                <th>Status</th>
                                <th>Expires</th>
                                <th class="text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($submissions as $submission): ?>
                            <tr>
                                <td>
                                    <?= date('d/m/Y H:i', strtotime($submission['submitted_at'])) ?><br>
                                    <small class="text-muted">by <?= htmlspecialchars($submission['submitted_by_name'] ?? 'Worker') ?></small>
                                </td>
                                <td>
                                    <strong><?= htmlspecialchars($submission['full_name'] ?? 'Unknown') ?></strong><br>
                                    <small class="text-muted">
                                        <?= htmlspecialchars($submission['employee_number'] ?? 'No ID') ?>
                                        <?php if ($submission['department']): ?>
                                            | <?= htmlspecialchars($submission['department']) ?>
                                        <?php endif; ?>
                                    </small>
                                </td>
                                <td>
                                    <code style="font-size: 1.1em;"><?= htmlspecialchars($submission['share_code']) ?></code>
                                    <?php if ($submission['document_path']): ?>
                                        <br><a href="../../<?= htmlspecialchars($submission['document_path']) ?>" target="_blank" class="small">
                                            <i class="bi bi-file-earmark-pdf"></i> View Document
                                        </a>
                                    <?php endif; ?>
                                </td>
                                <td><?= htmlspecialchars(ucwords(str_replace('_', ' ', $submission['document_type']))) ?></td>
                                <td>
                                    <?php
                                    $status_class = [
                                        'pending' => 'warning',
                                        'verified' => 'success',
                                        'rejected' => 'danger',
                                        'expired' => 'secondary'
                                    ][$submission['submission_status']] ?? 'secondary';
                                    ?>
                                    <span class="badge bg-<?= $status_class ?>">
                                        <?= ucfirst($submission['submission_status']) ?>
                                    </span>
                                    <?php if ($submission['verified_by_name']): ?>
                                        <br><small class="text-muted">by <?= htmlspecialchars($submission['verified_by_name']) ?></small>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($submission['expires_at']): ?>
                                        <?= date('d/m/Y', strtotime($submission['expires_at'])) ?>
                                        <?php if ($submission['days_until_expiry'] !== null): ?>
                                            <br>
                                            <?php if ($submission['days_until_expiry'] < 0): ?>
                                                <span class="badge bg-danger">Expired</span>
                                            <?php elseif ($submission['days_until_expiry'] <= 30): ?>
                                                <span class="badge bg-warning"><?= $submission['days_until_expiry'] ?> days</span>
                                            <?php else: ?>
                                                <small class="text-muted"><?= $submission['days_until_expiry'] ?> days</small>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td class="text-end">
                                    <?php if ($submission['submission_status'] === 'pending'): ?>
                                        <button class="btn btn-sm btn-success" onclick="verifyShareCode(<?= $submission['id'] ?>)"
                                                data-bs-toggle="modal" data-bs-target="#verifyModal">
                                            <i class="bi bi-check"></i> Verify
                                        </button>
                                        <button class="btn btn-sm btn-danger" onclick="rejectShareCode(<?= $submission['id'] ?>)"
                                                data-bs-toggle="modal" data-bs-target="#rejectModal">
                                            <i class="bi bi-x"></i> Reject
                                        </button>
                                    <?php elseif ($submission['submission_status'] === 'verified'): ?>
                                        <a href="https://www.gov.uk/view-right-to-work" target="_blank"
                                           class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-box-arrow-up-right"></i> Check Online
                                        </a>
                                    <?php endif; ?>
                                    <button class="btn btn-sm btn-outline-secondary" onclick="viewDetails(<?= htmlspecialchars(json_encode($submission)) ?>)"
                                            data-bs-toggle="modal" data-bs-target="#detailsModal">
                                        <i class="bi bi-eye"></i>
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Verify Modal -->
<div class="modal fade" id="verifyModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Verify Share Code</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="verify">
                <input type="hidden" name="submission_id" id="verify_submission_id">
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i>
                        Before verifying, please check the share code on the
                        <a href="https://www.gov.uk/view-right-to-work" target="_blank">UK Government website</a>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Expiry Date</label>
                        <input type="date" class="form-control" name="expires_at"
                               value="<?= date('Y-m-d', strtotime('+1 year')) ?>" required>
                        <small class="text-muted">When does this share code/document expire?</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Verification Notes</label>
                        <textarea class="form-control" name="verification_notes" rows="3"
                                  placeholder="Any notes about the verification..."></textarea>
                    </div>

                    <div class="form-check">
                        <input type="checkbox" class="form-check-input" name="create_rtw_check" id="create_rtw_check" value="1" checked>
                        <label class="form-check-label" for="create_rtw_check">
                            Create RTW check record
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">
                        <i class="bi bi-check-circle"></i> Verify Share Code
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Reject Modal -->
<div class="modal fade" id="rejectModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Reject Share Code</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST">
                <input type="hidden" name="action" value="reject">
                <input type="hidden" name="submission_id" id="reject_submission_id">
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Rejection Reason</label>
                        <textarea class="form-control" name="rejection_reason" rows="3" required
                                  placeholder="Please provide a reason for rejection..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">
                        <i class="bi bi-x-circle"></i> Reject Share Code
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Details Modal -->
<div class="modal fade" id="detailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Submission Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="details_content">
                <!-- Details will be populated by JavaScript -->
            </div>
        </div>
    </div>
</div>

<script>
function verifyShareCode(id) {
    document.getElementById('verify_submission_id').value = id;
}

function rejectShareCode(id) {
    document.getElementById('reject_submission_id').value = id;
}

function viewDetails(submission) {
    const content = `
        <table class="table">
            <tr>
                <th width="200">Worker:</th>
                <td>${submission.full_name || 'Unknown'}</td>
            </tr>
            <tr>
                <th>Employee Number:</th>
                <td>${submission.employee_number || 'N/A'}</td>
            </tr>
            <tr>
                <th>Email:</th>
                <td>${submission.email || 'N/A'}</td>
            </tr>
            <tr>
                <th>Share Code:</th>
                <td><code style="font-size: 1.2em;">${submission.share_code}</code></td>
            </tr>
            <tr>
                <th>Document Type:</th>
                <td>${submission.document_type}</td>
            </tr>
            <tr>
                <th>Submitted:</th>
                <td>${new Date(submission.submitted_at).toLocaleString()}</td>
            </tr>
            <tr>
                <th>Status:</th>
                <td>${submission.submission_status}</td>
            </tr>
            ${submission.verified_at ? `
            <tr>
                <th>Verified:</th>
                <td>${new Date(submission.verified_at).toLocaleString()} by ${submission.verified_by_name}</td>
            </tr>
            ` : ''}
            ${submission.expires_at ? `
            <tr>
                <th>Expires:</th>
                <td>${new Date(submission.expires_at).toLocaleDateString()}</td>
            </tr>
            ` : ''}
            ${submission.additional_info ? `
            <tr>
                <th>Additional Info:</th>
                <td>${submission.additional_info}</td>
            </tr>
            ` : ''}
            ${submission.verification_notes ? `
            <tr>
                <th>Verification Notes:</th>
                <td>${submission.verification_notes}</td>
            </tr>
            ` : ''}
        </table>
        ${submission.document_path ? `
        <div class="text-center">
            <a href="../../${submission.document_path}" target="_blank" class="btn btn-outline-primary">
                <i class="bi bi-file-earmark-pdf"></i> View Uploaded Document
            </a>
        </div>
        ` : ''}
    `;
    document.getElementById('details_content').innerHTML = content;
}
</script>

<?php include '../../includes/footer.php'; ?>