<?php
/**
 * Share Code Submission Portal
 * Allows workers to submit their RTW share codes and documents
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_once 'includes/alert_helper.php';
require_authentication();

// Initialize alert helper
$alertHelper = new RTWAlertHelper($pdo);

$page_title = 'Submit Share Code';
$current_page = 'rtw';
$message = '';
$message_type = 'success';

// Get current user
$user_id = $_SESSION['user_id'] ?? 0;
$user_email = $_SESSION['user_email'] ?? '';

// Try to find worker record based on user email
$worker = null;
try {
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE email = ?");
    $stmt->execute([$user_email]);
    $worker = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Silent fail
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $errors = [];

    // Validation
    if (empty($_POST['share_code'])) $errors[] = "Share code is required";
    if (empty($_POST['first_name'])) $errors[] = "First name is required";
    if (empty($_POST['last_name'])) $errors[] = "Last name is required";
    if (empty($_POST['email'])) $errors[] = "Email is required";

    if (empty($errors)) {
        try {
            // Find or create worker record
            if (!$worker) {
                // Check if worker exists by email
                $stmt = $pdo->prepare("SELECT id FROM " . DB_PREFIX . "sponsored_workers WHERE email = ?");
                $stmt->execute([$_POST['email']]);
                $existing = $stmt->fetch();

                if ($existing) {
                    $worker_id = $existing['id'];
                } else {
                    // Create minimal worker record
                    $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "sponsored_workers
                                          (full_name, first_name, last_name, email, created_at)
                                          VALUES (?, ?, ?, ?, NOW())");
                    $full_name = $_POST['first_name'] . ' ' . $_POST['last_name'];
                    $stmt->execute([$full_name, $_POST['first_name'], $_POST['last_name'], $_POST['email']]);
                    $worker_id = $pdo->lastInsertId();
                }
            } else {
                $worker_id = $worker['id'];
            }

            // Store share code submission
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "rtw_share_code_submissions
                                  (worker_id, share_code, document_type, additional_info,
                                   submission_status, submitted_by, submitted_at)
                                  VALUES (?, ?, ?, ?, 'pending', ?, NOW())");

            $stmt->execute([
                $worker_id,
                $_POST['share_code'],
                $_POST['document_type'] ?? 'share_code',
                $_POST['additional_info'] ?? '',
                $user_id
            ]);

            // Handle file upload if provided
            if (isset($_FILES['document_upload']) && $_FILES['document_upload']['error'] === UPLOAD_ERR_OK) {
                $upload_dir = '../../uploads/rtw/';
                if (!is_dir($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }

                $file_extension = strtolower(pathinfo($_FILES['document_upload']['name'], PATHINFO_EXTENSION));
                $allowed_extensions = ['pdf', 'jpg', 'jpeg', 'png'];

                if (in_array($file_extension, $allowed_extensions)) {
                    $new_filename = 'share_code_' . $worker_id . '_' . time() . '.' . $file_extension;
                    $upload_path = $upload_dir . $new_filename;

                    if (move_uploaded_file($_FILES['document_upload']['tmp_name'], $upload_path)) {
                        // Update submission with document path
                        $submission_id = $pdo->lastInsertId();
                        $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "rtw_share_code_submissions
                                              SET document_path = ? WHERE id = ?");
                        $stmt->execute(['uploads/rtw/' . $new_filename, $submission_id]);
                    }
                }
            }

            $submission_id = $pdo->lastInsertId();
            $full_name = $_POST['first_name'] . ' ' . $_POST['last_name'];

            // Create alert for new submission
            $alertHelper->createShareCodeAlert($submission_id, $full_name, $_POST['share_code']);

            // Log activity
            $alertHelper->logActivity('share_code_submitted',
                "Share code {$_POST['share_code']} submitted by {$full_name}",
                $worker_id, $submission_id);

            // Send notification to HR if configured
            $stmt = $pdo->query("SELECT setting_value FROM " . DB_PREFIX . "system_settings
                                WHERE setting_key = 'rtw_hr_notification_email'");
            $hr_email = $stmt->fetchColumn();

            if ($hr_email) {
                // TODO: Send email notification to HR about new submission
            }

            $message = "Share code submitted successfully! HR will review and verify your submission.";
            $message_type = 'success';

            // Clear form
            $_POST = [];

        } catch (PDOException $e) {
            $message = "Error submitting share code: " . $e->getMessage();
            $message_type = 'danger';
        }
    } else {
        $message = implode('<br>', $errors);
        $message_type = 'danger';
    }
}

// Check for existing pending submissions
$pending_submissions = [];
if ($worker) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "rtw_share_code_submissions
                               WHERE worker_id = ? AND submission_status = 'pending'
                               ORDER BY submitted_at DESC");
        $stmt->execute([$worker['id']]);
        $pending_submissions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Table might not exist yet
    }
}

include '../../includes/header.php';
?>

<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-shield-check"></i> Submit Right to Work Share Code</h4>
            <p class="text-muted mb-0">Submit your share code or upload RTW documents for verification</p>
        </div>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-<?= $message_type ?> alert-dismissible fade show">
        <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?>"></i>
        <?= $message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Instructions -->
    <div class="alert alert-info mb-4">
        <h5><i class="bi bi-info-circle"></i> How to Get Your Share Code</h5>
        <ol class="mb-0">
            <li>Visit the UK Government website: <a href="https://www.gov.uk/view-right-to-work" target="_blank">View a job applicant's right to work details</a></li>
            <li>Enter your details to generate a share code</li>
            <li>Copy the 9-character share code (format: XXX-XXX-XXX)</li>
            <li>Submit the share code using the form below</li>
        </ol>
    </div>

    <?php if (!empty($pending_submissions)): ?>
    <!-- Pending Submissions -->
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-clock-history"></i> Pending Submissions
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Submitted Date</th>
                            <th>Share Code</th>
                            <th>Type</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pending_submissions as $submission): ?>
                        <tr>
                            <td><?= date('d/m/Y H:i', strtotime($submission['submitted_at'])) ?></td>
                            <td><code><?= htmlspecialchars($submission['share_code']) ?></code></td>
                            <td><?= htmlspecialchars($submission['document_type']) ?></td>
                            <td><span class="badge bg-warning">Pending Review</span></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Submission Form -->
    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST" enctype="multipart/form-data">

                <!-- Personal Information -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-person"></i> Personal Information
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>First Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="first_name"
                                   value="<?= htmlspecialchars($_POST['first_name'] ?? $worker['first_name'] ?? '') ?>" required>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Last Name</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="last_name"
                                   value="<?= htmlspecialchars($_POST['last_name'] ?? $worker['last_name'] ?? '') ?>" required>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label"><strong>Email Address</strong> <span class="text-danger">*</span></label>
                        <input type="email" class="form-control" name="email"
                               value="<?= htmlspecialchars($_POST['email'] ?? $worker['email'] ?? $user_email) ?>" required>
                        <small class="text-muted">We'll use this to send you confirmation</small>
                    </div>
                </div>

                <!-- Share Code Information -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-key"></i> Share Code Details
                    </h6>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Share Code</strong> <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="share_code"
                                   value="<?= htmlspecialchars($_POST['share_code'] ?? '') ?>"
                                   pattern="[A-Z0-9]{3}-[A-Z0-9]{3}-[A-Z0-9]{3}"
                                   placeholder="XXX-XXX-XXX"
                                   style="text-transform: uppercase;" required>
                            <small class="text-muted">9-character code from UK Government website</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Document Type</strong></label>
                            <select class="form-select" name="document_type">
                                <option value="share_code">Online Share Code</option>
                                <option value="passport">Passport</option>
                                <option value="brp">Biometric Residence Permit</option>
                                <option value="visa">Visa</option>
                                <option value="other">Other Document</option>
                            </select>
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Upload Document (Optional)</label>
                        <input type="file" class="form-control" name="document_upload"
                               accept=".pdf,.jpg,.jpeg,.png">
                        <small class="text-muted">Upload a PDF or image of your RTW document (max 5MB)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Additional Information</label>
                        <textarea class="form-control" name="additional_info" rows="3"
                                  placeholder="Any additional information about your right to work status..."><?= htmlspecialchars($_POST['additional_info'] ?? '') ?></textarea>
                    </div>
                </div>

                <!-- Consent -->
                <div class="form-section mb-4">
                    <div class="form-check">
                        <input type="checkbox" class="form-check-input" id="consent" required>
                        <label class="form-check-label" for="consent">
                            I confirm that the information provided is accurate and I consent to the verification of my right to work status.
                        </label>
                    </div>
                </div>

                <div class="text-end">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="bi bi-send"></i> Submit Share Code
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 1.5rem;
}

.form-section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}
</style>

<script>
// Auto-format share code input
document.querySelector('input[name="share_code"]').addEventListener('input', function(e) {
    let value = e.target.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
    if (value.length > 3 && value.length <= 6) {
        value = value.slice(0, 3) + '-' + value.slice(3);
    } else if (value.length > 6) {
        value = value.slice(0, 3) + '-' + value.slice(3, 6) + '-' + value.slice(6, 9);
    }
    e.target.value = value;
});
</script>

<?php include '../../includes/footer.php'; ?>