<?php
/**
 * RTW Module Settings
 * Configure share code reminders and other module-specific settings
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();

$page_title = 'RTW Module Settings';
$current_page = 'rtw';
$message = '';
$message_type = 'success';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($_POST['action'] === 'save_settings') {
        try {
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_settings (setting_key, setting_value)
                                  VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)");

            $settings_to_save = [
                // Core RTW Settings (matching screenshot)
                ['rtw_require_photo_evidence', isset($_POST['require_photo_evidence']) ? '1' : '0'],
                ['rtw_list_b_followup_days', $_POST['list_b_followup_days'] ?? '90'],
                ['rtw_annual_check_reminder_days', $_POST['annual_check_reminder_days'] ?? '30'],

                // Additional settings
                ['rtw_share_code_reminder_enabled', isset($_POST['share_code_reminder_enabled']) ? '1' : '0'],
                ['rtw_share_code_reminder_frequency', $_POST['reminder_frequency'] ?? '30'],
                ['rtw_share_code_advance_notice', $_POST['advance_notice'] ?? '14'],
                ['rtw_auto_email_enabled', isset($_POST['auto_email_enabled']) ? '1' : '0'],
                ['rtw_hr_notification_email', $_POST['hr_notification_email'] ?? ''],
                ['rtw_expiry_alert_days', $_POST['expiry_alert_days'] ?? '30,14,7,1'],
                ['rtw_allow_worker_upload', isset($_POST['allow_worker_upload']) ? '1' : '0'],
                ['rtw_require_approval', isset($_POST['require_approval']) ? '1' : '0'],
                ['rtw_default_check_interval', $_POST['default_check_interval'] ?? '365']
            ];

            foreach ($settings_to_save as $setting) {
                $stmt->execute($setting);
            }

            $message = "RTW module settings saved successfully.";
            $message_type = 'success';
        } catch (PDOException $e) {
            $message = "Error saving settings: " . $e->getMessage();
            $message_type = 'danger';
        }
    }
}

// Get current settings
$settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key LIKE 'rtw_%'");
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

include '../../includes/header.php';
?>

<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-gear"></i> RTW Module Settings</h4>
            <p class="text-muted mb-0">Configure Right to Work checks and reminders</p>
        </div>
        <a href="list.php" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Back to RTW Checks
        </a>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-<?= $message_type ?> alert-dismissible fade show">
        <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : 'exclamation-triangle' ?>"></i>
        <?= htmlspecialchars($message) ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <div class="modern-card">
        <div class="card-body-modern">
            <form method="POST">
                <input type="hidden" name="action" value="save_settings">

                <!-- Core RTW Settings (Primary Settings from Screenshot) -->
                <div class="form-section mb-4" style="background: linear-gradient(135deg, rgba(11, 163, 96, 0.05) 0%, rgba(60, 186, 146, 0.05) 100%);">
                    <h6 class="form-section-title">
                        <i class="bi bi-shield-check"></i> Core RTW Settings
                    </h6>
                    <p class="text-muted small mb-4">Primary configuration options for Right to Work checks</p>

                    <div class="row">
                        <div class="col-md-12 mb-3">
                            <div class="form-check form-switch">
                                <input type="checkbox" class="form-check-input" id="require_photo_evidence"
                                       name="require_photo_evidence" value="1"
                                       <?= (!empty($settings['rtw_require_photo_evidence']) && $settings['rtw_require_photo_evidence'] == '1') ? 'checked' : '' ?>>
                                <label class="form-check-label" for="require_photo_evidence">
                                    <strong>Require Photo Evidence</strong>
                                </label>
                                <small class="d-block text-muted">Workers must upload photographic evidence of their RTW documents</small>
                            </div>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>List B Follow-up Days</strong></label>
                            <input type="number" class="form-control" name="list_b_followup_days"
                                   value="<?= htmlspecialchars($settings['rtw_list_b_followup_days'] ?? '90') ?>"
                                   min="1" max="365">
                            <small class="text-muted">Number of days to wait before follow-up for List B documents</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label"><strong>Annual Check Reminder (Days)</strong></label>
                            <input type="number" class="form-control" name="annual_check_reminder_days"
                                   value="<?= htmlspecialchars($settings['rtw_annual_check_reminder_days'] ?? '30') ?>"
                                   min="1" max="90">
                            <small class="text-muted">Days before annual check is due to send reminder notification</small>
                        </div>
                    </div>
                </div>

                <!-- Share Code Reminder Settings -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-clock"></i> Share Code Reminders
                    </h6>

                    <div class="form-check form-switch mb-3">
                        <input type="checkbox" class="form-check-input" id="share_code_reminder_enabled"
                               name="share_code_reminder_enabled"
                               <?= (!empty($settings['rtw_share_code_reminder_enabled']) && $settings['rtw_share_code_reminder_enabled'] == '1') ? 'checked' : '' ?>>
                        <label class="form-check-label" for="share_code_reminder_enabled">
                            <strong>Enable Share Code Reminders</strong>
                        </label>
                        <small class="d-block text-muted">Automatically remind workers to submit share codes</small>
                    </div>

                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Reminder Frequency (days)</label>
                            <input type="number" class="form-control" name="reminder_frequency"
                                   value="<?= htmlspecialchars($settings['rtw_share_code_reminder_frequency'] ?? '30') ?>"
                                   min="1" max="365">
                            <small class="text-muted">How often to remind workers with expiring share codes</small>
                        </div>

                        <div class="col-md-6 mb-3">
                            <label class="form-label">Advance Notice (days)</label>
                            <input type="number" class="form-control" name="advance_notice"
                                   value="<?= htmlspecialchars($settings['rtw_share_code_advance_notice'] ?? '14') ?>"
                                   min="1" max="90">
                            <small class="text-muted">Start reminding X days before expiry</small>
                        </div>
                    </div>
                </div>

                <!-- Email Notification Settings -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-envelope"></i> Email Notifications
                    </h6>

                    <div class="form-check form-switch mb-3">
                        <input type="checkbox" class="form-check-input" id="auto_email_enabled"
                               name="auto_email_enabled"
                               <?= (!empty($settings['rtw_auto_email_enabled']) && $settings['rtw_auto_email_enabled'] == '1') ? 'checked' : '' ?>>
                        <label class="form-check-label" for="auto_email_enabled">
                            <strong>Enable Automatic Email Notifications</strong>
                        </label>
                        <small class="d-block text-muted">Send automatic expiry alerts and reminders</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">HR Notification Email</label>
                        <input type="email" class="form-control" name="hr_notification_email"
                               value="<?= htmlspecialchars($settings['rtw_hr_notification_email'] ?? '') ?>"
                               placeholder="hr@company.com">
                        <small class="text-muted">Email address for HR notifications (leave empty to use default)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Expiry Alert Days</label>
                        <input type="text" class="form-control" name="expiry_alert_days"
                               value="<?= htmlspecialchars($settings['rtw_expiry_alert_days'] ?? '30,14,7,1') ?>">
                        <small class="text-muted">Comma-separated days before expiry to send alerts (e.g., 30,14,7,1)</small>
                    </div>
                </div>

                <!-- Worker Submission Settings -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-person-check"></i> Worker Submissions
                    </h6>

                    <div class="form-check mb-3">
                        <input type="checkbox" class="form-check-input" id="allow_worker_upload"
                               name="allow_worker_upload"
                               <?= (!empty($settings['rtw_allow_worker_upload']) && $settings['rtw_allow_worker_upload'] == '1') ? 'checked' : '' ?>>
                        <label class="form-check-label" for="allow_worker_upload">
                            Allow workers to upload documents/share codes
                        </label>
                    </div>

                    <div class="form-check mb-3">
                        <input type="checkbox" class="form-check-input" id="require_approval"
                               name="require_approval"
                               <?= (!empty($settings['rtw_require_approval']) && $settings['rtw_require_approval'] == '1') ? 'checked' : '' ?>>
                        <label class="form-check-label" for="require_approval">
                            Require HR approval for worker submissions
                        </label>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Default Check Interval (days)</label>
                        <input type="number" class="form-control" name="default_check_interval"
                               value="<?= htmlspecialchars($settings['rtw_default_check_interval'] ?? '365') ?>"
                               min="30" max="730">
                        <small class="text-muted">Default days between RTW checks</small>
                    </div>
                </div>

                <!-- Share Code Submission Link -->
                <div class="form-section mb-4">
                    <h6 class="form-section-title">
                        <i class="bi bi-link-45deg"></i> Worker Submission Portal
                    </h6>

                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i>
                        Workers can submit their share codes at:
                        <strong><?= $_SERVER['HTTP_HOST'] ?>/modules/rtw/submit_share_code.php</strong>
                        <button type="button" class="btn btn-sm btn-outline-info ms-2"
                                onclick="copyToClipboard('<?= $_SERVER['HTTP_HOST'] ?>/modules/rtw/submit_share_code.php')">
                            <i class="bi bi-clipboard"></i> Copy Link
                        </button>
                    </div>
                </div>

                <div class="text-end">
                    <a href="list.php" class="btn btn-secondary me-2">
                        <i class="bi bi-x-circle"></i> Cancel
                    </a>
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-save"></i> Save Settings
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText('https://' + text).then(function() {
        alert('Link copied to clipboard!');
    }, function(err) {
        console.error('Could not copy text: ', err);
    });
}
</script>

<style>
.form-section {
    background: var(--card-bg);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    padding: 1.5rem;
}

.form-section-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--primary-color);
    margin-bottom: 1rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}
</style>

<?php include '../../includes/footer.php'; ?>