<?php
/**
 * Quick RTW Check with Share Code
 * Allows HR to manually enter and verify share codes for immediate RTW checks
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();

$page_title = 'Quick RTW Check';
$current_page = 'rtw';
$message = '';
$message_type = 'success';
$verification_result = null;

// Get worker if specified
$worker_id = $_GET['worker_id'] ?? $_POST['worker_id'] ?? '';
$worker = null;

if ($worker_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
        $stmt->execute([$worker_id]);
        $worker = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Silent fail
    }
}

// Get all workers for dropdown
$workers = [];
try {
    $stmt = $pdo->query("SELECT id, full_name, employee_number, department FROM " . DB_PREFIX . "sponsored_workers ORDER BY full_name");
    $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Silent fail
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $_POST['action'] === 'verify_code') {
    $errors = [];

    // Validation
    if (empty($_POST['worker_id'])) $errors[] = "Please select a worker";
    if (empty($_POST['share_code'])) $errors[] = "Share code is required";
    if (empty($_POST['check_date'])) $errors[] = "Check date is required";

    if (empty($errors)) {
        // Format share code
        $share_code = strtoupper(trim($_POST['share_code']));
        $share_code = preg_replace('/[^A-Z0-9]/', '', $share_code);
        if (strlen($share_code) == 9) {
            $share_code = substr($share_code, 0, 3) . '-' . substr($share_code, 3, 3) . '-' . substr($share_code, 6, 3);
        }

        // Show verification result
        $verification_result = [
            'share_code' => $share_code,
            'worker_id' => $_POST['worker_id'],
            'check_date' => $_POST['check_date'],
            'verified' => true // In real implementation, this would check with UK Gov API
        ];

        $message = "Share code format is valid. Please verify on UK Government website before confirming.";
        $message_type = 'info';
    } else {
        $message = implode('<br>', $errors);
        $message_type = 'danger';
    }
}

// Handle confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $_POST['action'] === 'confirm_check') {
    try {
        // Create RTW check record
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_rtw_checks
                              (worker_id, check_date, check_type, outcome, document_type,
                               document_number, document_expiry, next_check_due,
                               documents_seen, notes, checked_by, created_at)
                              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())");

        $outcome = $_POST['outcome'] ?? 'pass';
        $expires_date = $_POST['expires_date'] ?? date('Y-m-d', strtotime('+1 year'));
        $next_check = $_POST['next_check_due'] ?? $expires_date;

        $stmt->execute([
            $_POST['worker_id'],
            $_POST['check_date'],
            'online_check',
            $outcome,
            'share_code',
            $_POST['share_code'],
            $expires_date,
            $next_check,
            'Share Code: ' . $_POST['share_code'],
            $_POST['notes'] ?? 'Quick check via share code verification',
            $_SESSION['user_id']
        ]);

        // Also store in share code submissions for tracking
        $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "rtw_share_code_submissions
                              (worker_id, share_code, document_type, submission_status,
                               verified_by, verified_at, verification_notes, submitted_by, submitted_at, expires_at)
                              VALUES (?, ?, 'share_code', 'verified', ?, NOW(), ?, ?, NOW(), ?)");

        $stmt->execute([
            $_POST['worker_id'],
            $_POST['share_code'],
            $_SESSION['user_id'],
            'Manual verification by HR',
            $_SESSION['user_id'],
            $expires_date
        ]);

        $message = "RTW check completed successfully!";
        $message_type = 'success';

        // Clear form
        $verification_result = null;
        $_POST = [];

    } catch (PDOException $e) {
        $message = "Error creating RTW check: " . $e->getMessage();
        $message_type = 'danger';
    }
}

include '../../includes/header.php';
?>

<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-lightning"></i> Quick RTW Check</h4>
            <p class="text-muted mb-0">Manually enter and verify share codes for immediate RTW checks</p>
        </div>
        <a href="list.php" class="btn btn-secondary">
            <i class="bi bi-arrow-left"></i> Back to RTW Checks
        </a>
    </div>

    <?php if ($message): ?>
    <div class="alert alert-<?= $message_type ?> alert-dismissible fade show">
        <i class="bi bi-<?= $message_type === 'success' ? 'check-circle' : ($message_type === 'danger' ? 'exclamation-triangle' : 'info-circle') ?>"></i>
        <?= $message ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Step 1: Enter Share Code -->
    <?php if (!$verification_result): ?>
    <div class="modern-card">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <span class="badge bg-primary me-2">Step 1</span> Enter Share Code Details
            </h5>
        </div>
        <div class="card-body-modern">
            <form method="POST" id="verifyForm">
                <input type="hidden" name="action" value="verify_code">

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label"><strong>Worker</strong> <span class="text-danger">*</span></label>
                        <select class="form-select" name="worker_id" required>
                            <option value="">Select worker...</option>
                            <?php foreach ($workers as $w): ?>
                                <option value="<?= $w['id'] ?>"
                                        <?= ($worker_id == $w['id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($w['full_name']) ?>
                                    (<?= htmlspecialchars($w['employee_number']) ?>)
                                    <?php if ($w['department']): ?>
                                        - <?= htmlspecialchars($w['department']) ?>
                                    <?php endif; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-md-6 mb-3">
                        <label class="form-label"><strong>Check Date</strong> <span class="text-danger">*</span></label>
                        <input type="date" class="form-control" name="check_date"
                               value="<?= date('Y-m-d') ?>" required>
                    </div>
                </div>

                <div class="mb-3">
                    <label class="form-label"><strong>Share Code</strong> <span class="text-danger">*</span></label>
                    <input type="text" class="form-control form-control-lg" name="share_code"
                           id="share_code_input"
                           pattern="[A-Z0-9]{3}-?[A-Z0-9]{3}-?[A-Z0-9]{3}"
                           placeholder="XXX-XXX-XXX or XXXXXXXXX"
                           style="text-transform: uppercase; font-family: monospace; font-size: 1.5rem; text-align: center;"
                           required>
                    <small class="text-muted">Enter the 9-character share code provided by the worker</small>
                </div>

                <div class="alert alert-info">
                    <h6><i class="bi bi-info-circle"></i> How to Verify Share Codes</h6>
                    <ol class="mb-0">
                        <li>Enter the share code above</li>
                        <li>Visit <a href="https://www.gov.uk/view-right-to-work" target="_blank">UK Government RTW Check Service</a></li>
                        <li>Enter the share code and worker's date of birth</li>
                        <li>Review the displayed information</li>
                        <li>Confirm the check below if valid</li>
                    </ol>
                </div>

                <div class="text-end">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="bi bi-search"></i> Proceed to Verification
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Recent Share Code Submissions -->
    <?php
    try {
        $stmt = $pdo->query("SELECT s.*, w.full_name, w.employee_number
                            FROM " . DB_PREFIX . "rtw_share_code_submissions s
                            LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON s.worker_id = w.id
                            WHERE s.submission_status = 'pending'
                            ORDER BY s.submitted_at DESC
                            LIMIT 5");
        $recent_submissions = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!empty($recent_submissions)):
    ?>
    <div class="modern-card mt-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-clock-history"></i> Recent Pending Submissions
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-sm table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Share Code</th>
                            <th>Submitted</th>
                            <th class="text-end">Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_submissions as $submission): ?>
                        <tr>
                            <td><?= htmlspecialchars($submission['full_name']) ?></td>
                            <td><code><?= htmlspecialchars($submission['share_code']) ?></code></td>
                            <td><?= date('d/m/Y H:i', strtotime($submission['submitted_at'])) ?></td>
                            <td class="text-end">
                                <button type="button" class="btn btn-sm btn-outline-primary"
                                        onclick="useShareCode('<?= $submission['worker_id'] ?>', '<?= htmlspecialchars($submission['share_code']) ?>')">
                                    Use This Code
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <div class="text-end mt-2">
                <a href="verify_share_codes.php" class="btn btn-sm btn-outline-secondary">
                    View All Submissions <i class="bi bi-arrow-right"></i>
                </a>
            </div>
        </div>
    </div>
    <?php
        endif;
    } catch (PDOException $e) {
        // Table might not exist
    }
    ?>

    <?php else: ?>
    <!-- Step 2: Confirm Verification -->
    <div class="modern-card">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <span class="badge bg-success me-2">Step 2</span> Confirm RTW Check
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="alert alert-warning">
                <i class="bi bi-exclamation-triangle"></i>
                <strong>Important:</strong> Please verify this share code on the
                <a href="https://www.gov.uk/view-right-to-work" target="_blank" class="alert-link">UK Government website</a>
                before confirming.
            </div>

            <div class="row mb-4">
                <div class="col-md-6">
                    <div class="card bg-light">
                        <div class="card-body text-center">
                            <h6 class="text-muted">Share Code</h6>
                            <h2 class="font-monospace"><?= htmlspecialchars($verification_result['share_code']) ?></h2>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="card bg-light">
                        <div class="card-body text-center">
                            <h6 class="text-muted">Worker</h6>
                            <?php
                            $selected_worker = null;
                            foreach ($workers as $w) {
                                if ($w['id'] == $verification_result['worker_id']) {
                                    $selected_worker = $w;
                                    break;
                                }
                            }
                            ?>
                            <h5><?= htmlspecialchars($selected_worker['full_name'] ?? 'Unknown') ?></h5>
                            <p class="mb-0"><?= htmlspecialchars($selected_worker['employee_number'] ?? '') ?></p>
                        </div>
                    </div>
                </div>
            </div>

            <form method="POST">
                <input type="hidden" name="action" value="confirm_check">
                <input type="hidden" name="worker_id" value="<?= htmlspecialchars($verification_result['worker_id']) ?>">
                <input type="hidden" name="share_code" value="<?= htmlspecialchars($verification_result['share_code']) ?>">
                <input type="hidden" name="check_date" value="<?= htmlspecialchars($verification_result['check_date']) ?>">

                <div class="row">
                    <div class="col-md-4 mb-3">
                        <label class="form-label"><strong>Check Outcome</strong></label>
                        <select class="form-select" name="outcome" required>
                            <option value="pass" selected>Pass - Valid RTW</option>
                            <option value="fail">Fail - Invalid</option>
                            <option value="refer">Refer - Further Check Required</option>
                        </select>
                    </div>

                    <div class="col-md-4 mb-3">
                        <label class="form-label"><strong>Document Expiry</strong></label>
                        <input type="date" class="form-control" name="expires_date"
                               value="<?= date('Y-m-d', strtotime('+1 year')) ?>" required>
                    </div>

                    <div class="col-md-4 mb-3">
                        <label class="form-label"><strong>Next Check Due</strong></label>
                        <input type="date" class="form-control" name="next_check_due"
                               value="<?= date('Y-m-d', strtotime('+1 year')) ?>" required>
                    </div>
                </div>

                <div class="mb-3">
                    <label class="form-label"><strong>Notes</strong></label>
                    <textarea class="form-control" name="notes" rows="3"
                              placeholder="Any additional notes about this check...">Quick check completed via share code verification on UK Government website.</textarea>
                </div>

                <div class="d-flex justify-content-between">
                    <a href="quick_check.php" class="btn btn-secondary">
                        <i class="bi bi-arrow-left"></i> Back
                    </a>
                    <div>
                        <a href="https://www.gov.uk/view-right-to-work" target="_blank" class="btn btn-outline-primary me-2">
                            <i class="bi bi-box-arrow-up-right"></i> Open UK Gov Site
                        </a>
                        <button type="submit" class="btn btn-success btn-lg">
                            <i class="bi bi-check-circle"></i> Confirm RTW Check
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
    <?php endif; ?>
</div>

<script>
// Auto-format share code input
document.getElementById('share_code_input')?.addEventListener('input', function(e) {
    let value = e.target.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
    if (value.length > 3 && value.length <= 6) {
        value = value.slice(0, 3) + '-' + value.slice(3);
    } else if (value.length > 6) {
        value = value.slice(0, 3) + '-' + value.slice(3, 6) + '-' + value.slice(6, 9);
    }
    e.target.value = value;
});

function useShareCode(workerId, shareCode) {
    document.querySelector('select[name="worker_id"]').value = workerId;
    document.getElementById('share_code_input').value = shareCode;
    document.getElementById('verifyForm').submit();
}
</script>

<?php include '../../includes/footer.php'; ?>