<?php
/**
 * RTW Online Check - Share Code Verification
 * Handles online right to work checks using share codes
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(60);

$page_title = 'Online RTW Check';
$current_page = 'rtw';

$worker_id = $_GET['worker_id'] ?? '';
$errors = [];
$check_result = null;

// Get worker details
$worker = null;
if ($worker_id) {
    $stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
    $stmt->execute([$worker_id]);
    $worker = $stmt->fetch();
}

// Handle share code submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $share_code = trim(strtoupper($_POST['share_code'] ?? ''));
    $worker_id = $_POST['worker_id'] ?? '';
    $date_of_birth = $_POST['date_of_birth'] ?? '';

    // Validate inputs
    if (empty($worker_id)) {
        $errors[] = "Please select a worker";
    }
    if (empty($share_code)) {
        $errors[] = "Share code is required";
    } elseif (!preg_match('/^[A-Z0-9]{9}$/', $share_code)) {
        $errors[] = "Invalid share code format. Must be 9 characters (letters and numbers)";
    }
    if (empty($date_of_birth)) {
        $errors[] = "Date of birth is required for verification";
    }

    if (empty($errors)) {
        try {
            // Check if share code already exists
            $stmt = $pdo->prepare("
                SELECT * FROM " . DB_PREFIX . "rtw_share_codes
                WHERE worker_id = ? AND share_code = ?
            ");
            $stmt->execute([$worker_id, $share_code]);
            $existing = $stmt->fetch();

            if (!$existing) {
                // Save share code
                $stmt = $pdo->prepare("
                    INSERT INTO " . DB_PREFIX . "rtw_share_codes
                    (worker_id, share_code, date_provided, expiry_date, checked_by, profile_url, notes)
                    VALUES (?, ?, CURDATE(), DATE_ADD(CURDATE(), INTERVAL 90 DAY), ?, ?, ?)
                ");

                $profile_url = "https://www.gov.uk/view-right-to-work?code=" . $share_code;
                $notes = "DOB used: " . $date_of_birth;

                $stmt->execute([
                    $worker_id,
                    $share_code,
                    $_SESSION['user_id'],
                    $profile_url,
                    $notes
                ]);
                $share_code_id = $pdo->lastInsertId();
            } else {
                $share_code_id = $existing['id'];
            }

            // Create RTW check record
            $stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "worker_rtw_checks
                (worker_id, check_date, check_type, check_method, documents_seen,
                 share_code_used, online_check, outcome, next_check_due, checked_by, notes, created_at)
                VALUES (?, CURDATE(), ?, 'online', ?, ?, 1, ?, ?, ?, ?, NOW())
            ");

            $check_type = $_POST['check_type'] ?? 'initial';
            $documents_seen = "Online share code verification: " . $share_code;
            $outcome = $_POST['outcome'] ?? 'pending';
            $next_check_date = $_POST['next_check_due'] ?? null;
            $notes = $_POST['notes'] ?? '';

            $stmt->execute([
                $worker_id,
                $check_type,
                $documents_seen,
                $share_code,
                $outcome,
                $next_check_date,
                $_SESSION['user_id'],
                $notes
            ]);
            $check_id = $pdo->lastInsertId();

            // Update share code as checked
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "rtw_share_codes
                SET check_performed = 1,
                    check_date = NOW(),
                    check_result = ?
                WHERE id = ?
            ");
            $stmt->execute([$outcome, $share_code_id]);

            // Create statutory excuse if check passed
            if ($outcome === 'pass') {
                $stmt = $pdo->prepare("
                    INSERT INTO " . DB_PREFIX . "rtw_statutory_excuse
                    (worker_id, check_id, established_date, valid_from, valid_until, excuse_type, notes)
                    VALUES (?, ?, CURDATE(), CURDATE(), ?, 'online_check', ?)
                ");

                $valid_until = $next_check_date ?: date('Y-m-d', strtotime('+6 months'));
                $excuse_notes = "Online check via share code: " . $share_code;

                $stmt->execute([
                    $worker_id,
                    $check_id,
                    $valid_until,
                    $excuse_notes
                ]);
            }

            // Log audit
            $stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "rtw_audit
                (worker_id, check_id, action, details, performed_by, ip_address, user_agent)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $worker_id,
                $check_id,
                'ONLINE_CHECK',
                "Share code: $share_code, Outcome: $outcome",
                $_SESSION['user_id'],
                $_SERVER['REMOTE_ADDR'],
                $_SERVER['HTTP_USER_AGENT']
            ]);

            $_SESSION['success'] = 'Online RTW check recorded successfully';

            // Redirect to check details
            header('Location: view_check.php?id=' . $check_id);
            exit;

        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Get all workers for dropdown
$workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers
                WHERE status = 'active' ORDER BY full_name";
$workers = $pdo->query($workers_sql)->fetchAll();

// Get previous share codes for this worker
$previous_codes = [];
if ($worker_id) {
    $stmt = $pdo->prepare("
        SELECT * FROM " . DB_PREFIX . "rtw_share_codes
        WHERE worker_id = ?
        ORDER BY date_provided DESC
    ");
    $stmt->execute([$worker_id]);
    $previous_codes = $stmt->fetchAll();
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-globe"></i> Online Right to Work Check
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <!-- Information Alert -->
                    <div class="alert alert-info">
                        <h6><i class="bi bi-info-circle"></i> Online Check Process</h6>
                        <ol class="mb-0">
                            <li>Ask the worker to generate a share code at: <a href="https://www.gov.uk/prove-right-to-work" target="_blank">www.gov.uk/prove-right-to-work</a></li>
                            <li>Enter the share code and worker's date of birth below</li>
                            <li>Click the verification link to check on GOV.UK website</li>
                            <li>Record the outcome and any relevant details</li>
                            <li>Save a PDF/screenshot of the online check result</li>
                        </ol>
                    </div>

                    <form method="POST">
                        <!-- Worker Selection -->
                        <div class="mb-3">
                            <label class="form-label">Worker <span class="text-danger">*</span></label>
                            <select name="worker_id" id="worker_id" class="form-select" required>
                                <option value="">Select worker...</option>
                                <?php foreach ($workers as $w): ?>
                                    <option value="<?php echo $w['id']; ?>"
                                            <?php echo $worker_id == $w['id'] ? 'selected' : ''; ?>
                                            data-dob="<?php echo $w['date_of_birth'] ?? ''; ?>">
                                        <?php echo htmlspecialchars($w['full_name']); ?> (<?php echo $w['employee_number']; ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Share Code -->
                        <div class="mb-3">
                            <label class="form-label">Share Code <span class="text-danger">*</span></label>
                            <input type="text" name="share_code" class="form-control text-uppercase"
                                   placeholder="e.g., AB12C3D4E" maxlength="9" required
                                   pattern="[A-Za-z0-9]{9}"
                                   value="<?php echo htmlspecialchars($_POST['share_code'] ?? ''); ?>">
                            <small class="text-muted">9-character code provided by the worker</small>
                        </div>

                        <!-- Date of Birth -->
                        <div class="mb-3">
                            <label class="form-label">Worker's Date of Birth <span class="text-danger">*</span></label>
                            <input type="date" name="date_of_birth" id="date_of_birth" class="form-control" required
                                   value="<?php echo htmlspecialchars($_POST['date_of_birth'] ?? ''); ?>">
                            <small class="text-muted">Required to access the online check</small>
                        </div>

                        <!-- Check Type -->
                        <div class="mb-3">
                            <label class="form-label">Check Type <span class="text-danger">*</span></label>
                            <select name="check_type" class="form-select" required>
                                <option value="initial">Initial (Pre-employment)</option>
                                <option value="follow_up">Follow-up</option>
                                <option value="annual">Annual</option>
                            </select>
                        </div>

                        <!-- Verification Link -->
                        <div class="card bg-light mb-3">
                            <div class="card-body">
                                <h6><i class="bi bi-link-45deg"></i> Verification Link</h6>
                                <p>After entering the share code above, use this link to verify:</p>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="govLink" readonly
                                           value="https://www.gov.uk/view-right-to-work">
                                    <button type="button" class="btn btn-primary" onclick="openGovLink()">
                                        <i class="bi bi-box-arrow-up-right"></i> Open GOV.UK
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Check Outcome -->
                        <div class="mb-3">
                            <label class="form-label">Check Outcome <span class="text-danger">*</span></label>
                            <select name="outcome" class="form-select" required>
                                <option value="">Select outcome...</option>
                                <option value="pass">Pass - Right to work confirmed</option>
                                <option value="fail">Fail - No right to work</option>
                                <option value="refer">Refer - Further checks required</option>
                            </select>
                        </div>

                        <!-- Next Check Due -->
                        <div class="mb-3">
                            <label class="form-label">Next Check Due</label>
                            <input type="date" name="next_check_due" class="form-control"
                                   min="<?php echo date('Y-m-d'); ?>">
                            <small class="text-muted">Leave blank for indefinite right to work</small>
                        </div>

                        <!-- Notes -->
                        <div class="mb-3">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="3"
                                      placeholder="Any additional information from the online check..."></textarea>
                        </div>

                        <!-- Important Reminders -->
                        <div class="alert alert-warning">
                            <h6><i class="bi bi-exclamation-triangle"></i> Important</h6>
                            <ul class="mb-0">
                                <li>Save a PDF or screenshot of the online check result</li>
                                <li>Upload the evidence using the document management system</li>
                                <li>Online checks are valid for 6 months from the check date</li>
                                <li>Record any work restrictions shown in the online check</li>
                            </ul>
                        </div>

                        <hr>

                        <div class="d-flex justify-content-between">
                            <a href="list.php" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Record Online Check
                            </button>
                        </div>
                    </form>

                    <!-- Previous Share Codes -->
                    <?php if (!empty($previous_codes)): ?>
                    <hr class="my-4">
                    <h6><i class="bi bi-clock-history"></i> Previous Share Codes for This Worker</h6>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Share Code</th>
                                    <th>Date Provided</th>
                                    <th>Status</th>
                                    <th>Result</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($previous_codes as $code): ?>
                                <tr>
                                    <td><code><?php echo htmlspecialchars($code['share_code']); ?></code></td>
                                    <td><?php echo date('d/m/Y', strtotime($code['date_provided'])); ?></td>
                                    <td>
                                        <?php if ($code['check_performed']): ?>
                                            <span class="badge bg-success">Checked</span>
                                        <?php else: ?>
                                            <span class="badge bg-warning">Pending</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($code['check_result']): ?>
                                            <?php
                                            $result_colors = ['valid' => 'success', 'invalid' => 'danger', 'expired' => 'warning'];
                                            $color = $result_colors[$code['check_result']] ?? 'secondary';
                                            ?>
                                            <span class="badge bg-<?php echo $color; ?>">
                                                <?php echo ucfirst($code['check_result']); ?>
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Auto-populate date of birth when worker is selected
document.getElementById('worker_id').addEventListener('change', function() {
    const selected = this.options[this.selectedIndex];
    const dob = selected.getAttribute('data-dob');
    if (dob) {
        document.getElementById('date_of_birth').value = dob;
    }
});

// Open GOV.UK link in new window
function openGovLink() {
    const shareCode = document.querySelector('input[name="share_code"]').value;
    if (shareCode) {
        window.open('https://www.gov.uk/view-right-to-work', '_blank');
    } else {
        alert('Please enter a share code first');
    }
}

// Format share code input
document.querySelector('input[name="share_code"]').addEventListener('input', function(e) {
    this.value = this.value.toUpperCase().replace(/[^A-Z0-9]/g, '');
});
</script>

<?php include '../../includes/footer.php'; ?>