<?php
/**
 * RTW Checks List
 * Fixed version with proper integration
 */

$page_title = 'Right to Work Checks';
$current_page = 'rtw';

// Include header which handles authentication and database
require_once '../../includes/header.php';

// Initialize variables
$worker_id = $_GET['worker_id'] ?? '';
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';

// Initialize statistics with defaults
$stats = [
    'total_checks' => 0,
    'passed' => 0,
    'failed' => 0,
    'overdue' => 0
];

// Check if RTW tables exist
$tables_exist = false;
try {
    $check_table = $pdo->query("SHOW TABLES LIKE '" . DB_PREFIX . "worker_rtw_checks'");
    if ($check_table->rowCount() > 0) {
        $tables_exist = true;
    }
} catch (PDOException $e) {
    // Tables don't exist yet
}

if (!$tables_exist) {
    // Show setup message if tables don't exist
    ?>
    <div class="dashboard-container">
        <div class="welcome-section mb-4">
            <div class="welcome-content">
                <h1 class="welcome-title">
                    <i class="bi bi-file-earmark-check"></i> Right to Work Checks
                </h1>
                <p class="welcome-subtitle">Setup Required</p>
            </div>
        </div>

        <div class="modern-card">
            <div class="card-body-modern">
                <div class="alert alert-warning">
                    <h5><i class="bi bi-exclamation-triangle"></i> Database Tables Not Found</h5>
                    <p>The RTW module tables need to be created. Please run the installation SQL:</p>
                    <ol>
                        <li>Go to your database management tool (phpMyAdmin)</li>
                        <li>Run the SQL from: <code>/modules/rtw/install.sql</code></li>
                        <li>Refresh this page</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>
    <?php
    require_once '../../includes/footer.php';
    exit;
}

// Build query
$where = ["1=1"];
$params = [];

if ($worker_id) {
    $where[] = "r.worker_id = ?";
    $params[] = $worker_id;
}

if ($search) {
    $where[] = "(w.full_name LIKE ? OR r.document_type LIKE ? OR r.document_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

// Filter by check status
if ($status === 'due') {
    $where[] = "r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
} elseif ($status === 'overdue') {
    $where[] = "r.next_check_due < CURDATE()";
}

$where_clause = implode(' AND ', $where);

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 25;
$offset = ($page - 1) * $per_page;

// Initialize empty results
$checks = [];
$total = 0;

try {
    // Count total
    $count_sql = "SELECT COUNT(*) FROM " . DB_PREFIX . "worker_rtw_checks r
                  LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
                  WHERE $where_clause";
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($params);
    $total = $stmt->fetchColumn();

    // Fetch checks
    $sql = "SELECT r.*,
            w.full_name as worker_name,
            w.employee_number,
            CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name,
            DATEDIFF(r.next_check_due, CURDATE()) as days_to_next_check
            FROM " . DB_PREFIX . "worker_rtw_checks r
            LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
            LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
            WHERE $where_clause
            ORDER BY r.check_date DESC
            LIMIT $per_page OFFSET $offset";

    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get statistics
    $stats_sql = "SELECT
        COUNT(*) as total_checks,
        SUM(CASE WHEN outcome = 'pass' THEN 1 ELSE 0 END) as passed,
        SUM(CASE WHEN outcome = 'fail' THEN 1 ELSE 0 END) as failed,
        SUM(CASE WHEN next_check_due < CURDATE() THEN 1 ELSE 0 END) as overdue
        FROM " . DB_PREFIX . "worker_rtw_checks";
    $stats_result = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);
    if ($stats_result) {
        $stats = $stats_result;
    }
} catch (PDOException $e) {
    // Handle error silently, show empty results
}

// Get workers for filter dropdown
$workers = [];
try {
    $workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers
                    WHERE status = 'active' ORDER BY full_name";
    $workers = $pdo->query($workers_sql)->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // No workers table or empty
}

$total_pages = ceil($total / $per_page);
?>

<div class="dashboard-container">
    <!-- Header Section -->
    <div class="welcome-section mb-4">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-file-earmark-check"></i> Right to Work Checks
            </h1>
            <p class="welcome-subtitle">Track RTW verification and maintain statutory excuse</p>
        </div>
        <div class="welcome-actions">
            <a href="add_check.php" class="btn-modern-primary">
                <i class="bi bi-plus-circle"></i> Add RTW Check
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="modern-card">
        <div class="card-body-modern">
            <!-- Filters -->
            <form method="GET" class="mb-4">
                <div class="row g-3">
                    <div class="col-md-3">
                        <input type="text"
                               name="search"
                               class="form-control"
                               placeholder="Search worker, document..."
                               value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                    <div class="col-md-3">
                        <select name="worker_id" class="form-select">
                            <option value="">All Workers</option>
                            <?php foreach ($workers as $worker): ?>
                                <option value="<?php echo $worker['id']; ?>"
                                        <?php echo $worker_id == $worker['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($worker['full_name']); ?>
                                    (<?php echo $worker['employee_number']; ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <select name="status" class="form-select">
                            <option value="">All Checks</option>
                            <option value="due" <?php echo $status === 'due' ? 'selected' : ''; ?>>
                                Due Soon (30 days)
                            </option>
                            <option value="overdue" <?php echo $status === 'overdue' ? 'selected' : ''; ?>>
                                Overdue
                            </option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <button type="submit" class="btn btn-secondary w-100">
                            <i class="bi bi-filter"></i> Filter
                        </button>
                    </div>
                </div>
            </form>

            <!-- Results Table -->
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Check Date</th>
                            <th>Check Type</th>
                            <th>Document</th>
                            <th>Outcome</th>
                            <th>Next Check Due</th>
                            <th>Checked By</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($checks)): ?>
                        <tr>
                            <td colspan="8" class="text-center py-5">
                                <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                                <p class="text-muted mt-2">No RTW checks found</p>
                                <a href="add_check.php" class="btn btn-primary">
                                    <i class="bi bi-plus-circle"></i> Add First Check
                                </a>
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($checks as $check): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($check['worker_name']); ?></strong><br>
                                <small class="text-muted"><?php echo htmlspecialchars($check['employee_number']); ?></small>
                            </td>
                            <td><?php echo date('d/m/Y', strtotime($check['check_date'])); ?></td>
                            <td>
                                <span class="badge bg-secondary">
                                    <?php echo ucfirst(str_replace('_', ' ', $check['check_type'])); ?>
                                </span>
                            </td>
                            <td>
                                <?php echo htmlspecialchars($check['document_type'] ?: '-'); ?><br>
                                <?php if ($check['document_number']): ?>
                                    <small class="text-muted"><?php echo htmlspecialchars($check['document_number']); ?></small>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php
                                $outcome_colors = [
                                    'pass' => 'success',
                                    'fail' => 'danger',
                                    'refer' => 'warning',
                                    'pending' => 'secondary'
                                ];
                                $color = $outcome_colors[$check['outcome']] ?? 'secondary';
                                ?>
                                <span class="badge bg-<?php echo $color; ?>">
                                    <?php echo ucfirst($check['outcome']); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($check['next_check_due']): ?>
                                    <?php echo date('d/m/Y', strtotime($check['next_check_due'])); ?>
                                    <?php if ($check['days_to_next_check'] !== null): ?>
                                        <?php if ($check['days_to_next_check'] < 0): ?>
                                            <span class="badge bg-danger">Overdue</span>
                                        <?php elseif ($check['days_to_next_check'] <= 30): ?>
                                            <span class="badge bg-warning"><?php echo $check['days_to_next_check']; ?> days</span>
                                        <?php endif; ?>
                                    <?php endif; ?>
                                <?php else: ?>
                                    -
                                <?php endif; ?>
                            </td>
                            <td><?php echo htmlspecialchars($check['checked_by_name'] ?: '-'); ?></td>
                            <td>
                                <a href="view_check.php?id=<?php echo $check['id']; ?>"
                                   class="btn btn-sm btn-outline-primary"
                                   title="View Details">
                                    <i class="bi bi-eye"></i>
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <?php if ($total_pages > 1): ?>
            <nav>
                <ul class="pagination justify-content-center">
                    <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                        <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                            <a class="page-link" href="?page=<?php echo $i; ?>&worker_id=<?php echo $worker_id; ?>&status=<?php echo $status; ?>&search=<?php echo urlencode($search); ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                    <?php endfor; ?>
                </ul>
            </nav>
            <?php endif; ?>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row g-3 mt-4">
        <div class="col-md-3">
            <div class="card bg-primary text-white">
                <div class="card-body">
                    <h4><?php echo $stats['total_checks'] ?: 0; ?></h4>
                    <p class="mb-0">Total Checks</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-success text-white">
                <div class="card-body">
                    <h4><?php echo $stats['passed'] ?: 0; ?></h4>
                    <p class="mb-0">Passed</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-danger text-white">
                <div class="card-body">
                    <h4><?php echo $stats['failed'] ?: 0; ?></h4>
                    <p class="mb-0">Failed</p>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card bg-warning text-white">
                <div class="card-body">
                    <h4><?php echo $stats['overdue'] ?: 0; ?></h4>
                    <p class="mb-0">Overdue</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Links -->
    <div class="modern-card mt-4">
        <div class="card-header">
            <h5><i class="bi bi-lightning"></i> Quick Actions</h5>
        </div>
        <div class="card-body">
            <div class="row g-3">
                <div class="col-md-3">
                    <a href="online_check.php" class="btn btn-outline-primary w-100">
                        <i class="bi bi-globe"></i> Online Check
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="document_upload.php" class="btn btn-outline-secondary w-100">
                        <i class="bi bi-upload"></i> Upload Documents
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="compliance_report.php" class="btn btn-outline-info w-100">
                        <i class="bi bi-graph-up"></i> Compliance Report
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="expiry_alerts.php" class="btn btn-outline-warning w-100">
                        <i class="bi bi-bell"></i> Expiry Alerts
                    </a>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once '../../includes/footer.php'; ?>