<?php
/**
 * RTW Checks List - COMPREHENSIVE VERSION with all features
 */

$page_title = 'Right to Work Checks';
$current_page = 'rtw';
require_once '../../includes/header.php';

// Filters
$worker_id = $_GET['worker_id'] ?? '';
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$department = $_GET['department'] ?? '';

// Get statistics FIRST (before building query)
$stats_sql = "SELECT
    COUNT(*) as total_checks,
    SUM(CASE WHEN outcome = 'pass' THEN 1 ELSE 0 END) as passed,
    SUM(CASE WHEN outcome = 'fail' THEN 1 ELSE 0 END) as failed,
    SUM(CASE WHEN next_check_due < CURDATE() THEN 1 ELSE 0 END) as overdue,
    SUM(CASE WHEN next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as due_soon
    FROM " . DB_PREFIX . "worker_rtw_checks";
$stats = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);

// Build query for list
$where = ["1=1"];
$params = [];

if ($worker_id) {
    $where[] = "r.worker_id = ?";
    $params[] = $worker_id;
}

if ($search) {
    $where[] = "(w.full_name LIKE ? OR r.document_type LIKE ? OR r.document_number LIKE ? OR w.employee_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

if ($status === 'due') {
    $where[] = "r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
} elseif ($status === 'overdue') {
    $where[] = "r.next_check_due < CURDATE()";
} elseif ($status === 'passed') {
    $where[] = "r.outcome = 'pass'";
} elseif ($status === 'failed') {
    $where[] = "r.outcome = 'fail'";
}

if ($department) {
    $where[] = "w.department = ?";
    $params[] = $department;
}

$where_clause = implode(' AND ', $where);

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 25;
$offset = ($page - 1) * $per_page;

// Count total
$count_sql = "SELECT COUNT(*) FROM " . DB_PREFIX . "worker_rtw_checks r
              LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
              WHERE $where_clause";
$stmt = $pdo->prepare($count_sql);
$stmt->execute($params);
$total = $stmt->fetchColumn();

// Fetch checks
$sql = "SELECT r.*,
        w.full_name as worker_name,
        w.employee_number,
        w.department,
        w.visa_expiry,
        CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name,
        DATEDIFF(r.next_check_due, CURDATE()) as days_to_next_check
        FROM " . DB_PREFIX . "worker_rtw_checks r
        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
        LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
        WHERE $where_clause
        ORDER BY r.check_date DESC
        LIMIT $per_page OFFSET $offset";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get workers for filter
$workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers
                ORDER BY full_name";
$workers = $pdo->query($workers_sql)->fetchAll(PDO::FETCH_ASSOC);

// Get departments for filter
$dept_sql = "SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers
             WHERE department IS NOT NULL ORDER BY department";
$departments = $pdo->query($dept_sql)->fetchAll(PDO::FETCH_COLUMN);

// Get recent alerts
$alerts_sql = "SELECT COUNT(*) as count FROM " . DB_PREFIX . "worker_rtw_checks
               WHERE next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
$urgent_alerts = $pdo->query($alerts_sql)->fetch(PDO::FETCH_ASSOC)['count'];

$total_pages = ceil($total / $per_page);
?>

<!-- Main Container with system-wide stat-card styling like Workers module -->
<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-file-earmark-check"></i> Right to Work Checks</h4>
            <p class="text-muted mb-0">UK Home Office Compliance Management | Total: <?= number_format($total) ?> records</p>
        </div>
        <a href="add_check.php" class="btn btn-primary">
            <i class="bi bi-plus-circle"></i> Add RTW Check
        </a>
    </div>

    <!-- Alert Banner if urgent checks -->
    <?php if ($urgent_alerts > 0): ?>
    <div class="alert alert-warning alert-dismissible fade show mb-4" role="alert">
        <i class="bi bi-exclamation-triangle-fill me-2"></i>
        <strong>Attention!</strong> You have <?= $urgent_alerts ?> RTW checks due within the next 7 days.
        <a href="expiry_alerts.php" class="alert-link">View Details</a>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php endif; ?>

    <!-- Statistics Cards at Top (like Workers module) -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="stat-card gradient-blue">
                <div class="d-flex align-items-center">
                    <div class="stat-icon bg-primary bg-opacity-10 text-primary">
                        <i class="bi bi-clipboard-check fs-4"></i>
                    </div>
                    <div class="ms-3">
                        <h2 class="mb-0"><?= number_format($stats['total_checks'] ?? 0) ?></h2>
                        <p class="text-muted mb-0">Total Checks</p>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card gradient-green">
                <div class="d-flex align-items-center">
                    <div class="stat-icon bg-success bg-opacity-10 text-success">
                        <i class="bi bi-check-circle fs-4"></i>
                    </div>
                    <div class="ms-3">
                        <h2 class="mb-0"><?= number_format($stats['passed'] ?? 0) ?></h2>
                        <p class="text-muted mb-0">Passed</p>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card gradient-yellow">
                <div class="d-flex align-items-center">
                    <div class="stat-icon bg-warning bg-opacity-10 text-warning">
                        <i class="bi bi-clock-history fs-4"></i>
                    </div>
                    <div class="ms-3">
                        <h2 class="mb-0"><?= number_format($stats['due_soon'] ?? 0) ?></h2>
                        <p class="text-muted mb-0">Due Soon</p>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card gradient-red">
                <div class="d-flex align-items-center">
                    <div class="stat-icon bg-danger bg-opacity-10 text-danger">
                        <i class="bi bi-exclamation-triangle fs-4"></i>
                    </div>
                    <div class="ms-3">
                        <h2 class="mb-0"><?= number_format($stats['overdue'] ?? 0) ?></h2>
                        <p class="text-muted mb-0">Overdue</p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Actions Panel -->
    <div class="modern-card mb-4">
        <div class="card-body-modern">
            <h5 class="mb-3"><i class="bi bi-lightning-charge"></i> Quick Actions</h5>
            <div class="row g-2">
                <div class="col-md-3">
                    <a href="add_check.php" class="btn btn-outline-primary w-100">
                        <i class="bi bi-plus-lg"></i> New RTW Check
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="document_upload.php" class="btn btn-outline-info w-100">
                        <i class="bi bi-cloud-upload"></i> Upload Documents
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="online_check.php" class="btn btn-outline-success w-100">
                        <i class="bi bi-globe"></i> Online Verification
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="compliance_report.php" class="btn btn-outline-warning w-100">
                        <i class="bi bi-file-earmark-pdf"></i> Compliance Report
                    </a>
                </div>
            </div>
            <div class="row g-2 mt-1">
                <div class="col-md-3">
                    <a href="expiry_alerts.php" class="btn btn-outline-danger w-100">
                        <i class="bi bi-bell"></i> Expiry Alerts
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="?status=overdue" class="btn btn-outline-secondary w-100">
                        <i class="bi bi-exclamation-circle"></i> View Overdue
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="?status=due" class="btn btn-outline-info w-100">
                        <i class="bi bi-calendar-check"></i> Due Soon
                    </a>
                </div>
                <div class="col-md-3">
                    <button onclick="exportToCSV()" class="btn btn-outline-dark w-100">
                        <i class="bi bi-download"></i> Export CSV
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters (matching Workers module layout) -->
    <form method="GET" class="row g-3 mb-4">
        <div class="col-md-4">
            <input type="text" name="search" class="form-control"
                   placeholder="Search by name, document, employee # or document number..."
                   value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-3">
            <select name="status" class="form-select">
                <option value="">All Statuses</option>
                <option value="passed" <?= $status === 'passed' ? 'selected' : '' ?>>✓ Passed</option>
                <option value="failed" <?= $status === 'failed' ? 'selected' : '' ?>>✗ Failed</option>
                <option value="due" <?= $status === 'due' ? 'selected' : '' ?>>⏰ Due Soon (30 days)</option>
                <option value="overdue" <?= $status === 'overdue' ? 'selected' : '' ?>>⚠ Overdue</option>
            </select>
        </div>
        <div class="col-md-3">
            <select name="department" class="form-select">
                <option value="">All Departments</option>
                <?php foreach ($departments as $dept): ?>
                    <option value="<?= htmlspecialchars($dept) ?>" <?= $department === $dept ? 'selected' : '' ?>>
                        <?= htmlspecialchars($dept) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="submit" class="btn btn-secondary w-100">
                <i class="bi bi-search"></i> Search
            </button>
        </div>
    </form>

    <!-- RTW Checks Table -->
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Employee</th>
                    <th>Check Date</th>
                    <th>Type</th>
                    <th>Document</th>
                    <th>Status</th>
                    <th>Next Check</th>
                    <th>Checked By</th>
                    <th width="150">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($checks)): ?>
                <tr>
                    <td colspan="8" class="text-center py-4">
                        <i class="bi bi-inbox fs-1 text-muted"></i>
                        <p class="text-muted mt-2">No RTW checks found</p>
                        <a href="add_check.php" class="btn btn-primary mt-2">
                            <i class="bi bi-plus-circle"></i> Add First Check
                        </a>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($checks as $check): ?>
                <tr>
                    <td>
                        <strong><?= htmlspecialchars($check['worker_name'] ?: 'Unknown') ?></strong>
                        <br>
                        <small class="text-muted">
                            #<?= htmlspecialchars($check['employee_number'] ?: 'N/A') ?>
                            <?php if ($check['department']): ?>
                                | <?= htmlspecialchars($check['department']) ?>
                            <?php endif; ?>
                        </small>
                    </td>
                    <td>
                        <?= date('d/m/Y', strtotime($check['check_date'])) ?>
                        <br>
                        <small class="text-muted"><?= date('H:i', strtotime($check['created_at'] ?? $check['check_date'])) ?></small>
                    </td>
                    <td>
                        <?php
                        $type_badges = [
                            'initial' => 'primary',
                            'follow_up' => 'info',
                            'annual' => 'secondary',
                            'expiry' => 'warning'
                        ];
                        $badge_class = $type_badges[$check['check_type']] ?? 'secondary';
                        ?>
                        <span class="badge bg-<?= $badge_class ?>">
                            <?= ucfirst(str_replace('_', ' ', $check['check_type'])) ?>
                        </span>
                    </td>
                    <td>
                        <?= htmlspecialchars($check['document_type'] ?: 'Multiple') ?>
                        <?php if ($check['document_number']): ?>
                            <br><small class="text-muted">№ <?= htmlspecialchars(substr($check['document_number'], -4)) ?></small>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php
                        $outcome_badges = [
                            'pass' => 'success',
                            'fail' => 'danger',
                            'refer' => 'warning',
                            'pending' => 'secondary'
                        ];
                        $badge_class = $outcome_badges[$check['outcome']] ?? 'secondary';
                        $outcome_icons = [
                            'pass' => 'bi-check-circle-fill',
                            'fail' => 'bi-x-circle-fill',
                            'refer' => 'bi-question-circle-fill',
                            'pending' => 'bi-hourglass-split'
                        ];
                        $icon = $outcome_icons[$check['outcome']] ?? 'bi-circle';
                        ?>
                        <span class="badge bg-<?= $badge_class ?>">
                            <i class="bi <?= $icon ?> me-1"></i>
                            <?= ucfirst($check['outcome']) ?>
                        </span>
                    </td>
                    <td>
                        <?php if ($check['next_check_due']): ?>
                            <?= date('d/m/Y', strtotime($check['next_check_due'])) ?>
                            <?php if ($check['days_to_next_check'] !== null): ?>
                                <?php if ($check['days_to_next_check'] < 0): ?>
                                    <br><span class="badge bg-danger"><i class="bi bi-exclamation-triangle"></i> Overdue</span>
                                <?php elseif ($check['days_to_next_check'] <= 7): ?>
                                    <br><span class="badge bg-danger"><i class="bi bi-clock"></i> <?= $check['days_to_next_check'] ?> days</span>
                                <?php elseif ($check['days_to_next_check'] <= 30): ?>
                                    <br><span class="badge bg-warning"><i class="bi bi-clock"></i> <?= $check['days_to_next_check'] ?> days</span>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="text-muted">Not required</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?= htmlspecialchars($check['checked_by_name'] ?: 'System') ?>
                    </td>
                    <td>
                        <div class="btn-group" role="group">
                            <a href="view_check.php?id=<?= $check['id'] ?>"
                               class="btn btn-sm btn-outline-primary"
                               data-bs-toggle="tooltip" title="View & Download PDF">
                                <i class="bi bi-eye"></i>
                            </a>
                            <a href="document_upload.php?check_id=<?= $check['id'] ?>"
                               class="btn btn-sm btn-outline-info"
                               data-bs-toggle="tooltip" title="Upload Documents">
                                <i class="bi bi-upload"></i>
                            </a>
                            <a href="add_check.php?worker_id=<?= $check['worker_id'] ?>"
                               class="btn btn-sm btn-outline-success"
                               data-bs-toggle="tooltip" title="New Check">
                                <i class="bi bi-plus"></i>
                            </a>
                            <?php if ($check['outcome'] == 'pass'): ?>
                            <button onclick="downloadPDF(<?= $check['id'] ?>)"
                                    class="btn btn-sm btn-outline-warning"
                                    data-bs-toggle="tooltip" title="Download Certificate">
                                <i class="bi bi-file-earmark-pdf"></i>
                            </button>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
    <nav aria-label="Page navigation" class="mt-4">
        <ul class="pagination justify-content-center">
            <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    <i class="bi bi-chevron-left"></i> Previous
                </a>
            </li>

            <?php
            $start_page = max(1, $page - 2);
            $end_page = min($total_pages, $page + 2);

            if ($start_page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=1&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">1</a>
                </li>
                <?php if ($start_page > 2): ?>
                    <li class="page-item disabled"><span class="page-link">...</span></li>
                <?php endif; ?>
            <?php endif; ?>

            <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                        <?= $i ?>
                    </a>
                </li>
            <?php endfor; ?>

            <?php if ($end_page < $total_pages): ?>
                <?php if ($end_page < $total_pages - 1): ?>
                    <li class="page-item disabled"><span class="page-link">...</span></li>
                <?php endif; ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?= $total_pages ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                        <?= $total_pages ?>
                    </a>
                </li>
            <?php endif; ?>

            <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Next <i class="bi bi-chevron-right"></i>
                </a>
            </li>
        </ul>
    </nav>
    <?php endif; ?>
</div>

<script>
// Initialize tooltips
document.addEventListener('DOMContentLoaded', function() {
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl)
    });
});

// Download PDF function
function downloadPDF(checkId) {
    window.open('view_check.php?id=' + checkId + '&download=1', '_blank');
}

// Export to CSV function
function exportToCSV() {
    window.location.href = 'export.php?format=csv&' + window.location.search.substring(1);
}
</script>

<?php require_once '../../includes/footer.php'; ?>