<?php
/**
 * RTW Checks List - Matching Worker Module Design
 */

$page_title = 'Right to Work Checks';
$current_page = 'rtw';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

// Filters
$worker_id = $_GET['worker_id'] ?? '';
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$department = $_GET['department'] ?? '';

// Get statistics FIRST (before building query)
$stats_sql = "SELECT
    COUNT(*) as total_checks,
    SUM(CASE WHEN outcome = 'pass' THEN 1 ELSE 0 END) as passed,
    SUM(CASE WHEN outcome = 'fail' THEN 1 ELSE 0 END) as failed,
    SUM(CASE WHEN next_check_due < CURDATE() THEN 1 ELSE 0 END) as overdue
    FROM " . DB_PREFIX . "worker_rtw_checks";
$stats = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);

// Build query for list
$where = ["1=1"];
$params = [];

if ($worker_id) {
    $where[] = "r.worker_id = ?";
    $params[] = $worker_id;
}

if ($search) {
    $where[] = "(w.full_name LIKE ? OR r.document_type LIKE ? OR r.document_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

if ($status === 'due') {
    $where[] = "r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
} elseif ($status === 'overdue') {
    $where[] = "r.next_check_due < CURDATE()";
} elseif ($status === 'passed') {
    $where[] = "r.outcome = 'pass'";
} elseif ($status === 'failed') {
    $where[] = "r.outcome = 'fail'";
}

if ($department) {
    $where[] = "w.department = ?";
    $params[] = $department;
}

$where_clause = implode(' AND ', $where);

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$per_page = 25;
$offset = ($page - 1) * $per_page;

// Count total
$count_sql = "SELECT COUNT(*) FROM " . DB_PREFIX . "worker_rtw_checks r
              LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
              WHERE $where_clause";
$stmt = $pdo->prepare($count_sql);
$stmt->execute($params);
$total = $stmt->fetchColumn();

// Fetch checks
$sql = "SELECT r.*,
        w.full_name as worker_name,
        w.employee_number,
        w.department,
        CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name,
        DATEDIFF(r.next_check_due, CURDATE()) as days_to_next_check
        FROM " . DB_PREFIX . "worker_rtw_checks r
        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
        LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
        WHERE $where_clause
        ORDER BY r.check_date DESC
        LIMIT $per_page OFFSET $offset";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get workers for filter
$workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers
                ORDER BY full_name";
$workers = $pdo->query($workers_sql)->fetchAll(PDO::FETCH_ASSOC);

// Get departments for filter
$dept_sql = "SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers
             WHERE department IS NOT NULL ORDER BY department";
$departments = $pdo->query($dept_sql)->fetchAll(PDO::FETCH_COLUMN);

// Get pending share code submissions count and details
try {
    $pending_sql = "SELECT COUNT(*) FROM " . DB_PREFIX . "rtw_share_code_submissions
                    WHERE submission_status = 'pending'";
    $pending_share_codes = $pdo->query($pending_sql)->fetchColumn();

    // Get recent pending submissions for display
    $recent_pending_sql = "SELECT s.*, w.full_name, w.employee_number, w.department
                          FROM " . DB_PREFIX . "rtw_share_code_submissions s
                          LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON s.worker_id = w.id
                          WHERE s.submission_status = 'pending'
                          ORDER BY s.submitted_at DESC
                          LIMIT 5";
    $recent_pending = $pdo->query($recent_pending_sql)->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $pending_share_codes = 0;
    $recent_pending = [];
}

// Get active alerts
try {
    $alerts_sql = "SELECT * FROM " . DB_PREFIX . "rtw_alerts
                  WHERE status = 'active'
                  ORDER BY severity DESC, created_at DESC
                  LIMIT 10";
    $active_alerts = $pdo->query($alerts_sql)->fetchAll(PDO::FETCH_ASSOC);
    $total_alerts = count($active_alerts);
} catch (PDOException $e) {
    $active_alerts = [];
    $total_alerts = 0;
}

// Get workers with expiring checks in next 30 days
$expiring_sql = "SELECT r.*, w.full_name, w.employee_number, w.department,
                 DATEDIFF(r.next_check_due, CURDATE()) as days_until
                 FROM " . DB_PREFIX . "worker_rtw_checks r
                 LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
                 WHERE r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
                 AND r.outcome = 'pass'
                 ORDER BY r.next_check_due ASC
                 LIMIT 5";
$expiring_checks = $pdo->query($expiring_sql)->fetchAll(PDO::FETCH_ASSOC);

$total_pages = ceil($total / $per_page);
?>

<!-- Main Container with system-wide stat-card styling like Workers module -->
<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-file-earmark-check"></i> Right to Work Checks</h4>
            <p class="text-muted mb-0">Total: <?= number_format($total) ?> records
            <?php if ($pending_share_codes > 0): ?>
                | <span class="badge bg-warning text-dark"><?= $pending_share_codes ?> pending share codes</span>
            <?php endif; ?>
            </p>
        </div>
        <div class="d-flex gap-2">
            <a href="quick_check.php" class="btn btn-success">
                <i class="bi bi-lightning"></i> Quick Check
            </a>
            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/rtw/add' : 'add_check.php'; ?>" class="btn btn-primary">
                <i class="bi bi-plus-circle"></i> Add RTW Check
            </a>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="quick-actions-grid mb-4">
        <!-- Total Checks -->
        <div class="quick-action-btn gradient-blue">
            <div class="quick-action-icon">
                <i class="fas fa-clipboard-check"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['total_checks'] ?? 0); ?> Total Checks</span>
        </div>

        <!-- Passed -->
        <div class="quick-action-btn gradient-green">
            <div class="quick-action-icon">
                <i class="ri-checkbox-circle-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['passed'] ?? 0); ?> Passed</span>
        </div>

        <!-- Failed -->
        <div class="quick-action-btn gradient-red">
            <div class="quick-action-icon">
                <span class="material-symbols-outlined" style="font-size: 1.75rem;">cancel</span>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['failed'] ?? 0); ?> Failed</span>
        </div>

        <!-- Overdue -->
        <div class="quick-action-btn gradient-orange">
            <div class="quick-action-icon">
                <i class="ri-error-warning-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats['overdue'] ?? 0); ?> Overdue</span>
        </div>
    </div>

    <!-- Active Alerts Section -->
    <?php if (!empty($active_alerts) || !empty($recent_pending) || !empty($expiring_checks)): ?>
    <div class="row mb-4">
        <?php if (!empty($active_alerts)): ?>
        <div class="col-md-4">
            <div class="modern-card h-100">
                <div class="card-header-modern bg-danger bg-opacity-10">
                    <h6 class="card-title-modern mb-0">
                        <i class="bi bi-bell-fill text-danger"></i> Active Alerts
                        <span class="badge bg-danger ms-2"><?= $total_alerts ?></span>
                    </h6>
                </div>
                <div class="card-body-modern" style="max-height: 300px; overflow-y: auto;">
                    <?php foreach ($active_alerts as $alert): ?>
                    <?php
                    $alert_icons = [
                        'danger' => 'exclamation-triangle-fill',
                        'warning' => 'exclamation-circle-fill',
                        'info' => 'info-circle-fill',
                        'success' => 'check-circle-fill'
                    ];
                    $alert_icon = $alert_icons[$alert['severity']] ?? 'bell-fill';
                    ?>
                    <div class="alert alert-<?= $alert['severity'] ?> py-2 mb-2">
                        <div class="d-flex justify-content-between align-items-start">
                            <div class="flex-grow-1">
                                <i class="bi bi-<?= $alert_icon ?>"></i>
                                <strong><?= htmlspecialchars($alert['title']) ?></strong>
                                <p class="mb-0 small"><?= htmlspecialchars($alert['message']) ?></p>
                            </div>
                            <?php if ($alert['action_url']): ?>
                            <a href="<?= htmlspecialchars($alert['action_url']) ?>" class="btn btn-sm btn-link">
                                <i class="bi bi-arrow-right"></i>
                            </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($recent_pending)): ?>
        <div class="col-md-4">
            <div class="modern-card h-100">
                <div class="card-header-modern bg-warning bg-opacity-10">
                    <h6 class="card-title-modern mb-0">
                        <i class="bi bi-clock-history text-warning"></i> Pending Share Codes
                        <span class="badge bg-warning text-dark ms-2"><?= $pending_share_codes ?></span>
                    </h6>
                </div>
                <div class="card-body-modern" style="max-height: 300px; overflow-y: auto;">
                    <?php foreach ($recent_pending as $pending): ?>
                    <div class="border-bottom pb-2 mb-2">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <strong><?= htmlspecialchars($pending['full_name'] ?? 'Unknown') ?></strong>
                                <br>
                                <small class="text-muted">
                                    <?= htmlspecialchars($pending['employee_number'] ?? 'No ID') ?>
                                    | <?= date('d/m H:i', strtotime($pending['submitted_at'])) ?>
                                </small>
                                <br>
                                <code><?= htmlspecialchars($pending['share_code']) ?></code>
                            </div>
                            <a href="verify_share_codes.php?id=<?= $pending['id'] ?>"
                               class="btn btn-sm btn-outline-success">
                                Verify
                            </a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    <?php if ($pending_share_codes > 5): ?>
                    <div class="text-center">
                        <a href="verify_share_codes.php" class="btn btn-sm btn-outline-warning">
                            View All <?= $pending_share_codes ?> Pending
                        </a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <?php if (!empty($expiring_checks)): ?>
        <div class="col-md-4">
            <div class="modern-card h-100">
                <div class="card-header-modern bg-info bg-opacity-10">
                    <h6 class="card-title-modern mb-0">
                        <i class="bi bi-calendar-event text-info"></i> Expiring Soon
                    </h6>
                </div>
                <div class="card-body-modern" style="max-height: 300px; overflow-y: auto;">
                    <?php foreach ($expiring_checks as $expiring): ?>
                    <div class="border-bottom pb-2 mb-2">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <strong><?= htmlspecialchars($expiring['full_name']) ?></strong>
                                <br>
                                <small class="text-muted">
                                    <?= htmlspecialchars($expiring['department'] ?? 'No Dept') ?>
                                </small>
                                <br>
                                <?php
                                $badge_color = $expiring['days_until'] <= 7 ? 'danger' :
                                              ($expiring['days_until'] <= 14 ? 'warning' : 'info');
                                ?>
                                <span class="badge bg-<?= $badge_color ?>">
                                    <?= $expiring['days_until'] ?> days
                                </span>
                            </div>
                            <a href="quick_check.php?worker_id=<?= $expiring['worker_id'] ?>"
                               class="btn btn-sm btn-outline-primary">
                                Renew
                            </a>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    <div class="text-center">
                        <a href="expiry_alerts.php" class="btn btn-sm btn-outline-info">
                            View All Expiring
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>
    </div>
    <?php endif; ?>

    <!-- Quick Actions Panel -->
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-lightning-fill"></i> Quick Actions
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="row g-3">
                <div class="col-md-3">
                    <a href="quick_check.php" class="btn btn-outline-success w-100">
                        <i class="bi bi-search"></i> Quick Share Code Check
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="verify_share_codes.php" class="btn btn-outline-warning w-100 position-relative">
                        <i class="bi bi-patch-check"></i> Verify Submissions
                        <?php if ($pending_share_codes > 0): ?>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                <?= $pending_share_codes ?>
                            </span>
                        <?php endif; ?>
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="expiry_alerts.php" class="btn btn-outline-danger w-100">
                        <i class="bi bi-exclamation-triangle"></i> Expiry Alerts
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="settings.php" class="btn btn-outline-secondary w-100">
                        <i class="bi bi-gear"></i> Module Settings
                    </a>
                </div>
            </div>
            <div class="row g-3 mt-2">
                <div class="col-md-3">
                    <a href="export.php?format=csv" class="btn btn-outline-info w-100">
                        <i class="bi bi-download"></i> Export CSV
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="submit_share_code.php" class="btn btn-outline-primary w-100">
                        <i class="bi bi-person-plus"></i> Worker Portal
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="compliance_report.php" class="btn btn-outline-dark w-100">
                        <i class="bi bi-file-text"></i> Compliance Report
                    </a>
                </div>
                <div class="col-md-3">
                    <a href="https://www.gov.uk/view-right-to-work" target="_blank" class="btn btn-outline-secondary w-100">
                        <i class="bi bi-box-arrow-up-right"></i> UK Gov Portal
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters (matching Workers module layout) -->
    <form method="GET" class="row g-3 mb-4">
        <div class="col-md-4">
            <input type="text" name="search" class="form-control"
                   placeholder="Search by name, document or employee #..."
                   value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-3">
            <select name="status" class="form-select">
                <option value="">All Statuses</option>
                <option value="passed" <?= $status === 'passed' ? 'selected' : '' ?>>Passed</option>
                <option value="failed" <?= $status === 'failed' ? 'selected' : '' ?>>Failed</option>
                <option value="due" <?= $status === 'due' ? 'selected' : '' ?>>Due Soon (30 days)</option>
                <option value="overdue" <?= $status === 'overdue' ? 'selected' : '' ?>>Overdue</option>
            </select>
        </div>
        <div class="col-md-3">
            <select name="department" class="form-select">
                <option value="">All Departments</option>
                <?php foreach ($departments as $dept): ?>
                    <option value="<?= htmlspecialchars($dept) ?>" <?= $department === $dept ? 'selected' : '' ?>>
                        <?= htmlspecialchars($dept) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="submit" class="btn btn-secondary w-100">
                <i class="bi bi-search"></i> Search
            </button>
        </div>
    </form>

    <!-- RTW Checks Table -->
    <div class="table-responsive">
        <table class="table table-hover">
            <thead>
                <tr>
                    <th>Employee #</th>
                    <th>Name</th>
                    <th>Department</th>
                    <th>Check Date</th>
                    <th>Document Type</th>
                    <th>Status</th>
                    <th>Next Check</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($checks)): ?>
                <tr>
                    <td colspan="8" class="text-center py-4">
                        <i class="bi bi-inbox fs-1 text-muted"></i>
                        <p class="text-muted mt-2">No RTW checks found</p>
                        <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/rtw/add' : 'add_check.php'; ?>" class="btn btn-primary mt-2">
                            <i class="bi bi-plus-circle"></i> Add First Check
                        </a>
                    </td>
                </tr>
                <?php else: ?>
                <?php foreach ($checks as $check): ?>
                <tr>
                    <td><?= htmlspecialchars($check['employee_number'] ?: 'N/A') ?></td>
                    <td>
                        <strong><?= htmlspecialchars($check['worker_name'] ?: 'Unknown') ?></strong>
                    </td>
                    <td><?= htmlspecialchars($check['department'] ?: 'N/A') ?></td>
                    <td><?= date('d/m/Y', strtotime($check['check_date'])) ?></td>
                    <td>
                        <?= htmlspecialchars($check['document_type']) ?>
                        <?php if ($check['document_number']): ?>
                            <br><small class="text-muted"><?= htmlspecialchars($check['document_number']) ?></small>
                        <?php endif; ?>
                    </td>
                    <td>
                        <?php
                        $outcome_badges = [
                            'pass' => 'success',
                            'fail' => 'danger',
                            'refer' => 'warning',
                            'pending' => 'secondary'
                        ];
                        $badge_class = $outcome_badges[$check['outcome']] ?? 'secondary';
                        ?>
                        <span class="badge bg-<?= $badge_class ?>">
                            <?= ucfirst($check['outcome']) ?>
                        </span>
                    </td>
                    <td>
                        <?php if ($check['next_check_due']): ?>
                            <?= date('d/m/Y', strtotime($check['next_check_due'])) ?>
                            <?php if ($check['days_to_next_check'] !== null): ?>
                                <?php if ($check['days_to_next_check'] < 0): ?>
                                    <br><span class="badge bg-danger">Overdue</span>
                                <?php elseif ($check['days_to_next_check'] <= 30): ?>
                                    <br><span class="badge bg-warning"><?= $check['days_to_next_check'] ?> days</span>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="text-muted">N/A</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <div class="btn-group" role="group">
                            <a href="view_check.php?id=<?= $check['id'] ?>"
                               class="btn btn-sm btn-outline-primary"
                               data-bs-toggle="tooltip" title="View Details">
                                <i class="bi bi-eye"></i>
                            </a>
                            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/rtw/add?worker_id=' . $check['worker_id'] : 'add_check.php?worker_id=' . $check['worker_id']; ?>"
                               class="btn btn-sm btn-outline-info"
                               data-bs-toggle="tooltip" title="New Check">
                                <i class="bi bi-pencil"></i>
                            </a>
                            <button type="button"
                                    class="btn btn-sm btn-outline-danger"
                                    data-bs-toggle="tooltip" title="Delete"
                                    onclick="if(confirm('Delete this RTW check?')) location.href='delete_check.php?id=<?= $check['id'] ?>'">
                                <i class="bi bi-trash"></i>
                            </button>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <?php if ($total_pages > 1): ?>
    <nav aria-label="Page navigation" class="mt-4">
        <ul class="pagination justify-content-center">
            <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page - 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Previous
                </a>
            </li>
            <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                        <?= $i ?>
                    </a>
                </li>
            <?php endfor; ?>
            <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                <a class="page-link" href="?page=<?= $page + 1 ?>&search=<?= urlencode($search) ?>&status=<?= urlencode($status) ?>&department=<?= urlencode($department) ?>">
                    Next
                </a>
            </li>
        </ul>
    </nav>
    <?php endif; ?>
</div>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>