<?php
/**
 * CLI Script to install all RTW module tables
 * Run: php install_tables_cli.php
 */

// CLI check
if (php_sapi_name() !== 'cli') {
    die("This script must be run from command line\n");
}

require_once dirname(__FILE__) . '/../../config/database.php';

echo "Installing RTW Module Tables...\n";
echo "================================\n\n";

$sql_file = dirname(__FILE__) . '/install_complete.sql';

if (!file_exists($sql_file)) {
    die("Error: install_complete.sql file not found\n");
}

// Read SQL file
$sql_content = file_get_contents($sql_file);

// Split by semicolons but handle them inside string literals
$statements = preg_split('/;\s*$/m', $sql_content);

$success_count = 0;
$error_count = 0;
$errors = [];

foreach ($statements as $statement) {
    $statement = trim($statement);

    // Skip empty statements and comments
    if (empty($statement) || strpos($statement, '--') === 0) {
        continue;
    }

    // Add semicolon back if needed
    if (substr($statement, -1) !== ';') {
        $statement .= ';';
    }

    try {
        $pdo->exec($statement);

        // Try to extract table name for feedback
        if (preg_match('/CREATE TABLE.*?`(\w+)`/i', $statement, $matches)) {
            echo "✓ Created table: " . $matches[1] . "\n";
            $success_count++;
        } elseif (preg_match('/INSERT.*?INTO.*?`(\w+)`/i', $statement, $matches)) {
            echo "✓ Inserted data into: " . $matches[1] . "\n";
            $success_count++;
        } else {
            echo "✓ Executed statement\n";
            $success_count++;
        }
    } catch (PDOException $e) {
        // Check if it's a "table already exists" error
        if (strpos($e->getMessage(), 'already exists') !== false) {
            if (preg_match('/`(\w+)`/', $statement, $matches)) {
                echo "→ Table already exists: " . $matches[1] . " (skipped)\n";
            }
        } else {
            $error_count++;
            $errors[] = $e->getMessage();
            echo "✗ Error: " . $e->getMessage() . "\n";
        }
    }
}

echo "\n================================\n";
echo "Installation Complete!\n";
echo "Successful operations: $success_count\n";

if ($error_count > 0) {
    echo "Errors encountered: $error_count\n";
    echo "\nError details:\n";
    foreach ($errors as $error) {
        echo "  - $error\n";
    }
} else {
    echo "All tables created successfully!\n";
}

echo "\nRTW Module is ready to use.\n";