<?php
/**
 * RTW Alert Helper Functions
 * Manages alerts and activity logging for RTW module
 */

class RTWAlertHelper {
    private $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    /**
     * Create a new alert
     */
    public function createAlert($type, $severity, $title, $message, $worker_id = null, $submission_id = null, $action_url = null) {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO " . DB_PREFIX . "rtw_alerts
                                         (alert_type, severity, title, message, worker_id, submission_id,
                                          action_url, status, created_at, created_by)
                                         VALUES (?, ?, ?, ?, ?, ?, ?, 'active', NOW(), ?)");

            $user_id = $_SESSION['user_id'] ?? null;
            $stmt->execute([$type, $severity, $title, $message, $worker_id, $submission_id, $action_url, $user_id]);

            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Failed to create alert: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Log an activity
     */
    public function logActivity($type, $description, $worker_id = null, $submission_id = null, $check_id = null) {
        try {
            $stmt = $this->pdo->prepare("INSERT INTO " . DB_PREFIX . "rtw_activity_log
                                         (activity_type, activity_description, worker_id, submission_id,
                                          check_id, user_id, ip_address, created_at)
                                         VALUES (?, ?, ?, ?, ?, ?, ?, NOW())");

            $user_id = $_SESSION['user_id'] ?? null;
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? null;

            $stmt->execute([$type, $description, $worker_id, $submission_id, $check_id, $user_id, $ip_address]);

            return true;
        } catch (PDOException $e) {
            error_log("Failed to log activity: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Create expiry alert for a worker
     */
    public function createExpiryAlert($worker_id, $worker_name, $days_until_expiry, $check_date) {
        $severity = 'info';
        $type = 'expiry_warning';

        if ($days_until_expiry <= 0) {
            $severity = 'danger';
            $type = 'expired';
            $title = "RTW Expired: {$worker_name}";
            $message = "RTW check expired " . abs($days_until_expiry) . " days ago (last check: " . date('d/m/Y', strtotime($check_date)) . ")";
        } elseif ($days_until_expiry <= 7) {
            $severity = 'danger';
            $title = "RTW Expiring Soon: {$worker_name}";
            $message = "RTW check expires in {$days_until_expiry} days";
        } elseif ($days_until_expiry <= 14) {
            $severity = 'warning';
            $title = "RTW Expiring: {$worker_name}";
            $message = "RTW check expires in {$days_until_expiry} days";
        } else {
            $title = "RTW Check Due: {$worker_name}";
            $message = "RTW check expires in {$days_until_expiry} days";
        }

        $action_url = "quick_check.php?worker_id={$worker_id}";

        // Check if alert already exists for this worker
        $stmt = $this->pdo->prepare("SELECT id FROM " . DB_PREFIX . "rtw_alerts
                                     WHERE worker_id = ? AND alert_type IN ('expiry_warning', 'expired')
                                     AND status = 'active'");
        $stmt->execute([$worker_id]);
        $existing = $stmt->fetch();

        if ($existing) {
            // Update existing alert
            $stmt = $this->pdo->prepare("UPDATE " . DB_PREFIX . "rtw_alerts
                                        SET severity = ?, title = ?, message = ?, created_at = NOW()
                                        WHERE id = ?");
            $stmt->execute([$severity, $title, $message, $existing['id']]);
            return $existing['id'];
        } else {
            // Create new alert
            return $this->createAlert($type, $severity, $title, $message, $worker_id, null, $action_url);
        }
    }

    /**
     * Create share code submission alert
     */
    public function createShareCodeAlert($submission_id, $worker_name, $share_code) {
        $title = "New Share Code: {$worker_name}";
        $message = "Share code {$share_code} submitted and awaiting verification";
        $action_url = "verify_share_codes.php?id={$submission_id}";

        return $this->createAlert('share_code_pending', 'warning', $title, $message, null, $submission_id, $action_url);
    }

    /**
     * Mark alert as acknowledged
     */
    public function acknowledgeAlert($alert_id, $user_id = null) {
        try {
            $stmt = $this->pdo->prepare("UPDATE " . DB_PREFIX . "rtw_alerts
                                        SET status = 'acknowledged',
                                            acknowledged_at = NOW(),
                                            acknowledged_by = ?
                                        WHERE id = ?");

            $user_id = $user_id ?: ($_SESSION['user_id'] ?? null);
            $stmt->execute([$user_id, $alert_id]);

            return true;
        } catch (PDOException $e) {
            return false;
        }
    }

    /**
     * Resolve an alert
     */
    public function resolveAlert($alert_id, $user_id = null) {
        try {
            $stmt = $this->pdo->prepare("UPDATE " . DB_PREFIX . "rtw_alerts
                                        SET status = 'resolved',
                                            resolved_at = NOW(),
                                            resolved_by = ?
                                        WHERE id = ?");

            $user_id = $user_id ?: ($_SESSION['user_id'] ?? null);
            $stmt->execute([$user_id, $alert_id]);

            return true;
        } catch (PDOException $e) {
            return false;
        }
    }

    /**
     * Auto-resolve alerts based on conditions
     */
    public function autoResolveAlerts() {
        // Resolve share code alerts when submission is verified
        $sql = "UPDATE " . DB_PREFIX . "rtw_alerts a
                INNER JOIN " . DB_PREFIX . "rtw_share_code_submissions s ON a.submission_id = s.id
                SET a.status = 'resolved', a.resolved_at = NOW()
                WHERE a.alert_type = 'share_code_pending'
                AND a.status = 'active'
                AND s.submission_status IN ('verified', 'rejected')";

        $this->pdo->exec($sql);

        // Resolve expiry alerts when new check is completed
        $sql = "UPDATE " . DB_PREFIX . "rtw_alerts a
                SET a.status = 'resolved', a.resolved_at = NOW()
                WHERE a.alert_type IN ('expiry_warning', 'expired')
                AND a.status = 'active'
                AND a.worker_id IN (
                    SELECT worker_id FROM " . DB_PREFIX . "worker_rtw_checks
                    WHERE outcome = 'pass'
                    AND next_check_due > CURDATE()
                    GROUP BY worker_id
                )";

        $this->pdo->exec($sql);
    }

    /**
     * Get alert summary for dashboard
     */
    public function getAlertSummary() {
        $stmt = $this->pdo->query("SELECT
                                   COUNT(*) as total,
                                   SUM(CASE WHEN severity = 'danger' THEN 1 ELSE 0 END) as danger,
                                   SUM(CASE WHEN severity = 'warning' THEN 1 ELSE 0 END) as warning,
                                   SUM(CASE WHEN severity = 'info' THEN 1 ELSE 0 END) as info
                                   FROM " . DB_PREFIX . "rtw_alerts
                                   WHERE status = 'active'");

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Create reminder sent alert
     */
    public function createReminderAlert($worker_id, $worker_name, $email) {
        $title = "Reminder Sent: {$worker_name}";
        $message = "RTW expiry reminder sent to {$email}";

        $this->createAlert('reminder_sent', 'success', $title, $message, $worker_id);
        $this->logActivity('reminder_sent', "Expiry reminder sent to {$worker_name} ({$email})", $worker_id);
    }
}