<?php
/**
 * Generate PDF Certificate for RTW Check
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();

$check_id = $_GET['id'] ?? 0;

if (!$check_id) {
    die('Invalid check ID');
}

// Get check details
$sql = "SELECT r.*,
        w.full_name as worker_name,
        w.employee_number,
        w.department,
        w.date_of_birth,
        w.ni_number,
        CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name
        FROM " . DB_PREFIX . "worker_rtw_checks r
        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
        LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
        WHERE r.id = ?";

$stmt = $pdo->prepare($sql);
$stmt->execute([$check_id]);
$check = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$check) {
    die('RTW check not found');
}

// For now, generate a simple HTML certificate that can be printed as PDF
// In production, you would use a PDF library like TCPDF or mPDF

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>RTW Certificate - <?= htmlspecialchars($check['worker_name']) ?></title>
    <style>
        @media print {
            body { margin: 0; }
            .no-print { display: none; }
        }

        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }

        .certificate {
            border: 2px solid #000;
            padding: 40px;
            margin: 20px 0;
            background: white;
        }

        .header {
            text-align: center;
            margin-bottom: 30px;
            border-bottom: 2px solid #000;
            padding-bottom: 20px;
        }

        h1 {
            color: #2c3e50;
            margin: 0 0 10px 0;
            font-size: 28px;
        }

        h2 {
            color: #34495e;
            margin: 0;
            font-size: 18px;
            font-weight: normal;
        }

        .section {
            margin: 25px 0;
        }

        .section-title {
            font-weight: bold;
            color: #2c3e50;
            border-bottom: 1px solid #ddd;
            padding-bottom: 5px;
            margin-bottom: 10px;
        }

        .row {
            display: flex;
            margin: 10px 0;
        }

        .label {
            font-weight: bold;
            width: 200px;
            color: #555;
        }

        .value {
            flex: 1;
            color: #000;
        }

        .status-pass {
            color: #27ae60;
            font-weight: bold;
            font-size: 18px;
        }

        .status-fail {
            color: #e74c3c;
            font-weight: bold;
            font-size: 18px;
        }

        .footer {
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #ddd;
            text-align: center;
            font-size: 12px;
            color: #777;
        }

        .signature-section {
            margin-top: 40px;
            display: flex;
            justify-content: space-between;
        }

        .signature-box {
            width: 45%;
        }

        .signature-line {
            border-bottom: 1px solid #000;
            margin-bottom: 5px;
            height: 40px;
        }

        .signature-label {
            font-size: 12px;
            color: #666;
        }

        .stamp {
            position: absolute;
            top: 100px;
            right: 50px;
            transform: rotate(-15deg);
            opacity: 0.3;
            font-size: 48px;
            font-weight: bold;
            color: <?= $check['outcome'] == 'pass' ? '#27ae60' : '#e74c3c' ?>;
            border: 3px solid;
            padding: 10px 20px;
            border-radius: 10px;
        }

        @media print {
            .certificate {
                page-break-inside: avoid;
            }
        }
    </style>
</head>
<body>
    <div class="no-print" style="text-align: center; margin: 20px;">
        <button onclick="window.print()" style="padding: 10px 20px; font-size: 16px; cursor: pointer;">
            🖨️ Print Certificate
        </button>
        <button onclick="window.close()" style="padding: 10px 20px; font-size: 16px; cursor: pointer; margin-left: 10px;">
            Close
        </button>
    </div>

    <div class="certificate">
        <div class="stamp"><?= strtoupper($check['outcome']) ?></div>

        <div class="header">
            <h1>RIGHT TO WORK CERTIFICATE</h1>
            <h2>UK Home Office Compliance Check</h2>
            <p>Certificate Reference: RTW-<?= str_pad($check_id, 6, '0', STR_PAD_LEFT) ?></p>
        </div>

        <div class="section">
            <div class="section-title">EMPLOYEE INFORMATION</div>
            <div class="row">
                <span class="label">Full Name:</span>
                <span class="value"><?= htmlspecialchars($check['worker_name']) ?></span>
            </div>
            <div class="row">
                <span class="label">Employee Number:</span>
                <span class="value"><?= htmlspecialchars($check['employee_number'] ?: 'N/A') ?></span>
            </div>
            <div class="row">
                <span class="label">Department:</span>
                <span class="value"><?= htmlspecialchars($check['department'] ?: 'N/A') ?></span>
            </div>
            <?php if ($check['date_of_birth']): ?>
            <div class="row">
                <span class="label">Date of Birth:</span>
                <span class="value"><?= date('d/m/Y', strtotime($check['date_of_birth'])) ?></span>
            </div>
            <?php endif; ?>
        </div>

        <div class="section">
            <div class="section-title">CHECK DETAILS</div>
            <div class="row">
                <span class="label">Check Date:</span>
                <span class="value"><?= date('d/m/Y', strtotime($check['check_date'])) ?></span>
            </div>
            <div class="row">
                <span class="label">Check Type:</span>
                <span class="value"><?= ucfirst(str_replace('_', ' ', $check['check_type'])) ?></span>
            </div>
            <div class="row">
                <span class="label">Check Outcome:</span>
                <span class="value <?= $check['outcome'] == 'pass' ? 'status-pass' : 'status-fail' ?>">
                    <?= strtoupper($check['outcome']) ?>
                </span>
            </div>
        </div>

        <div class="section">
            <div class="section-title">DOCUMENTS VERIFIED</div>
            <div class="row">
                <span class="label">Primary Document:</span>
                <span class="value"><?= htmlspecialchars($check['document_type'] ?: 'N/A') ?></span>
            </div>
            <?php if ($check['document_number']): ?>
            <div class="row">
                <span class="label">Document Reference:</span>
                <span class="value">****<?= substr($check['document_number'], -4) ?></span>
            </div>
            <?php endif; ?>
            <?php if ($check['document_expiry']): ?>
            <div class="row">
                <span class="label">Document Expiry:</span>
                <span class="value"><?= date('d/m/Y', strtotime($check['document_expiry'])) ?></span>
            </div>
            <?php endif; ?>
            <div class="row">
                <span class="label">Documents Seen:</span>
                <span class="value"><?= nl2br(htmlspecialchars($check['documents_seen'])) ?></span>
            </div>
        </div>

        <?php if ($check['next_check_due']): ?>
        <div class="section">
            <div class="section-title">FOLLOW-UP REQUIREMENTS</div>
            <div class="row">
                <span class="label">Next Check Due:</span>
                <span class="value"><?= date('d/m/Y', strtotime($check['next_check_due'])) ?></span>
            </div>
        </div>
        <?php endif; ?>

        <div class="section">
            <div class="section-title">VERIFICATION</div>
            <div class="row">
                <span class="label">Checked By:</span>
                <span class="value"><?= htmlspecialchars($check['checked_by_name'] ?: 'System Administrator') ?></span>
            </div>
            <div class="row">
                <span class="label">Date & Time:</span>
                <span class="value"><?= date('d/m/Y H:i', strtotime($check['created_at'])) ?></span>
            </div>
        </div>

        <?php if ($check['notes']): ?>
        <div class="section">
            <div class="section-title">ADDITIONAL NOTES</div>
            <p><?= nl2br(htmlspecialchars($check['notes'])) ?></p>
        </div>
        <?php endif; ?>

        <div class="signature-section">
            <div class="signature-box">
                <div class="signature-line"></div>
                <div class="signature-label">HR Officer Signature</div>
            </div>
            <div class="signature-box">
                <div class="signature-line"></div>
                <div class="signature-label">Date</div>
            </div>
        </div>

        <div class="footer">
            <p><strong>IMPORTANT:</strong> This certificate confirms that a Right to Work check has been conducted
            in accordance with UK Home Office requirements. Original documents have been verified and copies retained.</p>
            <p>Generated on <?= date('d/m/Y H:i') ?> | System Reference: SCMS-RTW-<?= $check_id ?></p>
            <p>© <?= date('Y') ?> Staff Compliance Management System | Confidential Document</p>
        </div>
    </div>

    <script>
        // Auto-print if requested
        <?php if (isset($_GET['auto_print'])): ?>
        window.onload = function() {
            window.print();
        }
        <?php endif; ?>
    </script>
</body>
</html>