<?php
/**
 * Export RTW Checks to CSV
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();

$format = $_GET['format'] ?? 'csv';

// Get filters from query string
$status = $_GET['status'] ?? '';
$search = $_GET['search'] ?? '';
$department = $_GET['department'] ?? '';

// Build query
$where = ["1=1"];
$params = [];

if ($search) {
    $where[] = "(w.full_name LIKE ? OR r.document_type LIKE ? OR r.document_number LIKE ?)";
    $search_term = "%{$search}%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

if ($status === 'due') {
    $where[] = "r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
} elseif ($status === 'overdue') {
    $where[] = "r.next_check_due < CURDATE()";
} elseif ($status === 'passed') {
    $where[] = "r.outcome = 'pass'";
} elseif ($status === 'failed') {
    $where[] = "r.outcome = 'fail'";
}

if ($department) {
    $where[] = "w.department = ?";
    $params[] = $department;
}

$where_clause = implode(' AND ', $where);

// Fetch all RTW checks
$sql = "SELECT
        r.*,
        w.full_name as worker_name,
        w.employee_number,
        w.department,
        w.email,
        w.visa_expiry,
        CONCAT(u.first_name, ' ', COALESCE(u.last_name, '')) as checked_by_name,
        DATEDIFF(r.next_check_due, CURDATE()) as days_to_next_check
        FROM " . DB_PREFIX . "worker_rtw_checks r
        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
        LEFT JOIN " . DB_PREFIX . "users u ON r.checked_by = u.id
        WHERE $where_clause
        ORDER BY r.check_date DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Set headers for CSV download
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="rtw_checks_' . date('Y-m-d') . '.csv"');

// Create output stream
$output = fopen('php://output', 'w');

// Add UTF-8 BOM for Excel compatibility
fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

// Write CSV headers
fputcsv($output, [
    'Employee Number',
    'Worker Name',
    'Department',
    'Email',
    'Check Date',
    'Check Type',
    'Document Type',
    'Document Number',
    'Document Expiry',
    'Outcome',
    'Next Check Due',
    'Days Until Next',
    'Visa Expiry',
    'Checked By',
    'Documents Seen',
    'Notes',
    'Created At'
]);

// Write data rows
foreach ($checks as $check) {
    // Format dates
    $check_date = $check['check_date'] ? date('d/m/Y', strtotime($check['check_date'])) : '';
    $doc_expiry = $check['document_expiry'] ? date('d/m/Y', strtotime($check['document_expiry'])) : '';
    $next_check = $check['next_check_due'] ? date('d/m/Y', strtotime($check['next_check_due'])) : 'N/A';
    $visa_expiry = $check['visa_expiry'] ? date('d/m/Y', strtotime($check['visa_expiry'])) : '';
    $created = $check['created_at'] ? date('d/m/Y H:i', strtotime($check['created_at'])) : '';

    // Status text
    $days_text = '';
    if ($check['days_to_next_check'] !== null) {
        if ($check['days_to_next_check'] < 0) {
            $days_text = 'OVERDUE';
        } else {
            $days_text = $check['days_to_next_check'] . ' days';
        }
    }

    fputcsv($output, [
        $check['employee_number'] ?? '',
        $check['worker_name'] ?? '',
        $check['department'] ?? '',
        $check['email'] ?? '',
        $check_date,
        ucfirst(str_replace('_', ' ', $check['check_type'])),
        $check['document_type'] ?? '',
        $check['document_number'] ?? '',
        $doc_expiry,
        ucfirst($check['outcome']),
        $next_check,
        $days_text,
        $visa_expiry,
        $check['checked_by_name'] ?? '',
        $check['documents_seen'] ?? '',
        $check['notes'] ?? '',
        $created
    ]);
}

fclose($output);
exit;