<?php
/**
 * RTW Expiry Alerts Dashboard
 * Shows all workers with expiring or expired RTW checks
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';
require_authentication();

$page_title = 'RTW Expiry Alerts';
$current_page = 'rtw';

// Get filter
$filter = $_GET['filter'] ?? 'all';
$days = (int)($_GET['days'] ?? 30);

// Build query based on filter
$where_clause = "1=1";
$params = [];

switch ($filter) {
    case 'overdue':
        $where_clause = "r.next_check_due < CURDATE()";
        break;
    case 'week':
        $where_clause = "r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
        break;
    case 'month':
        $where_clause = "r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
        break;
    case 'custom':
        $where_clause = "r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)";
        $params[] = $days;
        break;
    default:
        $where_clause = "r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL 90 DAY)";
}

// Add outcome filter
$where_clause .= " AND r.outcome = 'pass'";

// Get expiring/expired checks
$sql = "SELECT r.*, w.full_name, w.employee_number, w.department, w.email,
               DATEDIFF(r.next_check_due, CURDATE()) as days_until
        FROM " . DB_PREFIX . "worker_rtw_checks r
        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
        WHERE $where_clause
        AND r.id IN (SELECT MAX(id) FROM " . DB_PREFIX . "worker_rtw_checks GROUP BY worker_id)
        ORDER BY r.next_check_due ASC";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$checks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get statistics
$stats_sql = "SELECT
              COUNT(CASE WHEN r.next_check_due < CURDATE() THEN 1 END) as overdue,
              COUNT(CASE WHEN r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 1 END) as week,
              COUNT(CASE WHEN r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1 END) as month,
              COUNT(CASE WHEN r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 90 DAY) THEN 1 END) as quarter
              FROM " . DB_PREFIX . "worker_rtw_checks r
              WHERE r.outcome = 'pass'
              AND r.id IN (SELECT MAX(id) FROM " . DB_PREFIX . "worker_rtw_checks GROUP BY worker_id)";

$stats = $pdo->query($stats_sql)->fetch(PDO::FETCH_ASSOC);

include '../../includes/header.php';
?>

<div class="stat-card mb-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h4><i class="bi bi-exclamation-triangle"></i> RTW Expiry Alerts</h4>
            <p class="text-muted mb-0">Monitor and manage expiring Right to Work checks</p>
        </div>
        <div>
            <button class="btn btn-outline-primary me-2" onclick="sendBulkReminders()">
                <i class="bi bi-envelope"></i> Send Bulk Reminders
            </button>
            <a href="list.php" class="btn btn-secondary">
                <i class="bi bi-arrow-left"></i> Back to List
            </a>
        </div>
    </div>

    <!-- Statistics Cards -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="modern-card <?= $filter === 'overdue' ? 'border-danger' : '' ?>">
                <div class="card-body-modern">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-danger bg-opacity-10 text-danger">
                            <i class="bi bi-x-circle fs-4"></i>
                        </div>
                        <div class="ms-3">
                            <h3 class="mb-0"><?= number_format($stats['overdue']) ?></h3>
                            <p class="text-muted mb-0">Overdue</p>
                        </div>
                    </div>
                    <?php if ($filter !== 'overdue'): ?>
                    <div class="mt-2">
                        <a href="?filter=overdue" class="btn btn-sm btn-outline-danger w-100">View</a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="modern-card <?= $filter === 'week' ? 'border-warning' : '' ?>">
                <div class="card-body-modern">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-warning bg-opacity-10 text-warning">
                            <i class="bi bi-calendar-week fs-4"></i>
                        </div>
                        <div class="ms-3">
                            <h3 class="mb-0"><?= number_format($stats['week']) ?></h3>
                            <p class="text-muted mb-0">Next 7 Days</p>
                        </div>
                    </div>
                    <?php if ($filter !== 'week'): ?>
                    <div class="mt-2">
                        <a href="?filter=week" class="btn btn-sm btn-outline-warning w-100">View</a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="modern-card <?= $filter === 'month' ? 'border-info' : '' ?>">
                <div class="card-body-modern">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-info bg-opacity-10 text-info">
                            <i class="bi bi-calendar-month fs-4"></i>
                        </div>
                        <div class="ms-3">
                            <h3 class="mb-0"><?= number_format($stats['month']) ?></h3>
                            <p class="text-muted mb-0">Next 30 Days</p>
                        </div>
                    </div>
                    <?php if ($filter !== 'month'): ?>
                    <div class="mt-2">
                        <a href="?filter=month" class="btn btn-sm btn-outline-info w-100">View</a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="modern-card">
                <div class="card-body-modern">
                    <div class="d-flex align-items-center">
                        <div class="stat-icon bg-primary bg-opacity-10 text-primary">
                            <i class="bi bi-calendar3 fs-4"></i>
                        </div>
                        <div class="ms-3">
                            <h3 class="mb-0"><?= number_format($stats['quarter']) ?></h3>
                            <p class="text-muted mb-0">Next 90 Days</p>
                        </div>
                    </div>
                    <?php if ($filter !== 'all'): ?>
                    <div class="mt-2">
                        <a href="?filter=all" class="btn btn-sm btn-outline-primary w-100">View All</a>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Filter Pills -->
    <ul class="nav nav-pills mb-4">
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'all' ? 'active' : '' ?>" href="?filter=all">
                All Expiring
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'overdue' ? 'active' : '' ?>" href="?filter=overdue">
                <i class="bi bi-x-circle"></i> Overdue
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'week' ? 'active' : '' ?>" href="?filter=week">
                <i class="bi bi-calendar-week"></i> Next 7 Days
            </a>
        </li>
        <li class="nav-item">
            <a class="nav-link <?= $filter === 'month' ? 'active' : '' ?>" href="?filter=month">
                <i class="bi bi-calendar-month"></i> Next 30 Days
            </a>
        </li>
    </ul>

    <!-- Expiring Checks List -->
    <div class="modern-card">
        <div class="card-body-modern">
            <?php if (empty($checks)): ?>
                <div class="text-center py-5">
                    <i class="bi bi-check-circle" style="font-size: 3rem; color: var(--text-success);"></i>
                    <p class="text-muted mt-3">No RTW checks expiring in this period</p>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>
                                    <input type="checkbox" id="selectAll" onchange="toggleSelectAll(this)">
                                </th>
                                <th>Worker</th>
                                <th>Department</th>
                                <th>Last Check</th>
                                <th>Expiry Date</th>
                                <th>Status</th>
                                <th class="text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($checks as $check): ?>
                            <tr class="<?= $check['days_until'] < 0 ? 'table-danger' : ($check['days_until'] <= 7 ? 'table-warning' : '') ?>">
                                <td>
                                    <input type="checkbox" class="worker-checkbox" value="<?= $check['worker_id'] ?>"
                                           data-email="<?= htmlspecialchars($check['email'] ?? '') ?>">
                                </td>
                                <td>
                                    <strong><?= htmlspecialchars($check['full_name']) ?></strong><br>
                                    <small class="text-muted">
                                        <?= htmlspecialchars($check['employee_number'] ?? 'No ID') ?>
                                        <?php if ($check['email']): ?>
                                            | <?= htmlspecialchars($check['email']) ?>
                                        <?php endif; ?>
                                    </small>
                                </td>
                                <td><?= htmlspecialchars($check['department'] ?? 'N/A') ?></td>
                                <td>
                                    <?= date('d/m/Y', strtotime($check['check_date'])) ?><br>
                                    <small class="text-muted"><?= htmlspecialchars($check['document_type']) ?></small>
                                </td>
                                <td>
                                    <strong><?= date('d/m/Y', strtotime($check['next_check_due'])) ?></strong>
                                </td>
                                <td>
                                    <?php if ($check['days_until'] < 0): ?>
                                        <span class="badge bg-danger">
                                            Overdue <?= abs($check['days_until']) ?> days
                                        </span>
                                    <?php elseif ($check['days_until'] == 0): ?>
                                        <span class="badge bg-danger">Expires Today</span>
                                    <?php elseif ($check['days_until'] <= 7): ?>
                                        <span class="badge bg-warning">
                                            <?= $check['days_until'] ?> days left
                                        </span>
                                    <?php elseif ($check['days_until'] <= 30): ?>
                                        <span class="badge bg-info">
                                            <?= $check['days_until'] ?> days left
                                        </span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">
                                            <?= $check['days_until'] ?> days left
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td class="text-end">
                                    <div class="btn-group">
                                        <a href="quick_check.php?worker_id=<?= $check['worker_id'] ?>"
                                           class="btn btn-sm btn-outline-success" title="Perform RTW Check">
                                            <i class="bi bi-check"></i> Check
                                        </a>
                                        <?php if ($check['email']): ?>
                                        <button class="btn btn-sm btn-outline-primary"
                                                onclick="sendReminder(<?= $check['worker_id'] ?>, '<?= htmlspecialchars($check['full_name']) ?>')"
                                                title="Send Reminder">
                                            <i class="bi bi-envelope"></i>
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
function toggleSelectAll(checkbox) {
    const checkboxes = document.querySelectorAll('.worker-checkbox');
    checkboxes.forEach(cb => cb.checked = checkbox.checked);
}

function sendReminder(workerId, workerName) {
    if (confirm(`Send RTW expiry reminder to ${workerName}?`)) {
        // TODO: Implement AJAX call to send reminder
        alert('Reminder functionality to be implemented');
    }
}

function sendBulkReminders() {
    const selected = document.querySelectorAll('.worker-checkbox:checked');
    if (selected.length === 0) {
        alert('Please select workers to send reminders to');
        return;
    }

    if (confirm(`Send reminders to ${selected.length} selected workers?`)) {
        // TODO: Implement bulk reminder sending
        alert('Bulk reminder functionality to be implemented');
    }
}
</script>

<?php include '../../includes/footer.php'; ?>