<?php
/**
 * RTW Expiry Tracking & Alerts
 * Monitor and alert for expiring RTW checks
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

$page_title = 'RTW Expiry Alerts';
$current_page = 'rtw';

// Get module settings
$stmt = $pdo->prepare("SELECT settings FROM " . DB_PREFIX . "modules WHERE module_id = 'rtw'");
$stmt->execute();
$module = $stmt->fetch();
$settings = $module ? json_decode($module['settings'], true) : [];

$alert_days = $settings['annual_check_reminder'] ?? 30;
$follow_up_days = $settings['list_b_follow_up_days'] ?? 90;

// Handle reminder actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['create_reminder'])) {
        $worker_id = $_POST['worker_id'];
        $reminder_type = $_POST['reminder_type'];
        $reminder_date = $_POST['reminder_date'];

        $stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "rtw_reminders
            (worker_id, reminder_type, reminder_date, days_before, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$worker_id, $reminder_type, $reminder_date, $alert_days]);

        $_SESSION['success'] = 'Reminder created successfully';
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }

    if (isset($_POST['acknowledge_reminder'])) {
        $reminder_id = $_POST['reminder_id'];

        $stmt = $pdo->prepare("
            UPDATE " . DB_PREFIX . "rtw_reminders
            SET acknowledged = 1, acknowledged_by = ?, acknowledged_date = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$_SESSION['user_id'], $reminder_id]);

        $_SESSION['success'] = 'Reminder acknowledged';
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Get expiring checks
$stmt = $pdo->prepare("
    SELECT
        c.id as check_id,
        c.worker_id,
        w.full_name,
        w.employee_number,
        w.department,
        w.location,
        w.email,
        w.manager_email,
        c.check_date,
        c.check_type,
        c.next_check_due,
        c.document_type,
        DATEDIFF(c.next_check_due, CURDATE()) as days_remaining,
        CASE
            WHEN c.next_check_due < CURDATE() THEN 'overdue'
            WHEN DATEDIFF(c.next_check_due, CURDATE()) <= 7 THEN 'critical'
            WHEN DATEDIFF(c.next_check_due, CURDATE()) <= 30 THEN 'warning'
            WHEN DATEDIFF(c.next_check_due, CURDATE()) <= 90 THEN 'notice'
            ELSE 'ok'
        END as alert_level
    FROM " . DB_PREFIX . "worker_rtw_checks c
    JOIN " . DB_PREFIX . "sponsored_workers w ON c.worker_id = w.id
    WHERE c.id IN (
        SELECT MAX(id) FROM " . DB_PREFIX . "worker_rtw_checks GROUP BY worker_id
    )
    AND (c.next_check_due IS NOT NULL)
    AND w.status = 'active'
    ORDER BY c.next_check_due ASC
");
$stmt->execute();
$expiring_checks = $stmt->fetchAll();

// Categorize alerts
$alerts_by_level = [
    'overdue' => [],
    'critical' => [],
    'warning' => [],
    'notice' => []
];

foreach ($expiring_checks as $check) {
    if ($check['alert_level'] != 'ok') {
        $alerts_by_level[$check['alert_level']][] = $check;
    }
}

// Get pending reminders
$stmt = $pdo->prepare("
    SELECT
        r.*,
        w.full_name,
        w.employee_number,
        u.first_name as ack_fname,
        u.last_name as ack_lname
    FROM " . DB_PREFIX . "rtw_reminders r
    JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
    LEFT JOIN " . DB_PREFIX . "users u ON r.acknowledged_by = u.id
    WHERE r.sent = 0 OR r.acknowledged = 0
    ORDER BY r.reminder_date ASC
");
$stmt->execute();
$pending_reminders = $stmt->fetchAll();

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-bell"></i> RTW Expiry Alerts & Reminders
            </h1>
            <p class="welcome-subtitle">Monitor expiring checks and manage reminders</p>
        </div>
        <div class="welcome-actions">
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#sendAlertsModal">
                <i class="bi bi-envelope"></i> Send Alert Emails
            </button>
            <a href="compliance_report.php" class="btn btn-secondary">
                <i class="bi bi-graph-up"></i> Compliance Report
            </a>
        </div>
    </div>

    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show">
            <?php echo htmlspecialchars($_SESSION['success']); ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>

    <!-- Alert Summary -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="stat-card bg-danger text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo count($alerts_by_level['overdue']); ?></div>
                        <div class="stat-label">Overdue</div>
                    </div>
                    <i class="bi bi-exclamation-octagon-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-warning text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo count($alerts_by_level['critical']); ?></div>
                        <div class="stat-label">Critical (≤7 days)</div>
                    </div>
                    <i class="bi bi-exclamation-triangle-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-info text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo count($alerts_by_level['warning']); ?></div>
                        <div class="stat-label">Warning (8-30 days)</div>
                    </div>
                    <i class="bi bi-info-circle-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-secondary text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo count($alerts_by_level['notice']); ?></div>
                        <div class="stat-label">Notice (31-90 days)</div>
                    </div>
                    <i class="bi bi-bell-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
    </div>

    <!-- Overdue Checks -->
    <?php if (!empty($alerts_by_level['overdue'])): ?>
    <div class="modern-card mb-4 border-danger">
        <div class="card-header-modern bg-danger text-white">
            <h5 class="card-title-modern mb-0">
                <i class="bi bi-exclamation-octagon"></i> OVERDUE CHECKS - IMMEDIATE ACTION REQUIRED
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Department</th>
                            <th>Last Check</th>
                            <th>Due Date</th>
                            <th>Days Overdue</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($alerts_by_level['overdue'] as $check): ?>
                        <tr class="table-danger">
                            <td>
                                <strong><?php echo htmlspecialchars($check['full_name']); ?></strong><br>
                                <small><?php echo htmlspecialchars($check['employee_number']); ?></small>
                            </td>
                            <td><?php echo htmlspecialchars($check['department'] ?: '-'); ?></td>
                            <td><?php echo date('d/m/Y', strtotime($check['check_date'])); ?></td>
                            <td>
                                <strong class="text-danger">
                                    <?php echo date('d/m/Y', strtotime($check['next_check_due'])); ?>
                                </strong>
                            </td>
                            <td>
                                <span class="badge bg-danger">
                                    <?php echo abs($check['days_remaining']); ?> days overdue
                                </span>
                            </td>
                            <td>
                                <a href="add_check.php?worker_id=<?php echo $check['worker_id']; ?>" class="btn btn-sm btn-danger">
                                    <i class="bi bi-plus-circle"></i> Add Check Now
                                </a>
                                <button class="btn btn-sm btn-outline-danger" onclick="createReminder(<?php echo $check['worker_id']; ?>, 'overdue')">
                                    <i class="bi bi-bell"></i> Set Reminder
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Critical Alerts -->
    <?php if (!empty($alerts_by_level['critical'])): ?>
    <div class="modern-card mb-4 border-warning">
        <div class="card-header-modern bg-warning text-dark">
            <h5 class="card-title-modern mb-0">
                <i class="bi bi-exclamation-triangle"></i> Critical - Due Within 7 Days
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Department</th>
                            <th>Document Type</th>
                            <th>Due Date</th>
                            <th>Days Remaining</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($alerts_by_level['critical'] as $check): ?>
                        <tr class="table-warning">
                            <td>
                                <strong><?php echo htmlspecialchars($check['full_name']); ?></strong><br>
                                <small><?php echo htmlspecialchars($check['employee_number']); ?></small>
                            </td>
                            <td><?php echo htmlspecialchars($check['department'] ?: '-'); ?></td>
                            <td><?php echo htmlspecialchars($check['document_type'] ?: 'Not specified'); ?></td>
                            <td>
                                <strong><?php echo date('d/m/Y', strtotime($check['next_check_due'])); ?></strong>
                            </td>
                            <td>
                                <span class="badge bg-warning text-dark">
                                    <?php echo $check['days_remaining']; ?> days
                                </span>
                            </td>
                            <td>
                                <a href="add_check.php?worker_id=<?php echo $check['worker_id']; ?>" class="btn btn-sm btn-warning">
                                    <i class="bi bi-plus-circle"></i> Schedule Check
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Warning & Notice Alerts (Collapsible) -->
    <?php if (!empty($alerts_by_level['warning']) || !empty($alerts_by_level['notice'])): ?>
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-calendar-check"></i> Upcoming Expiries
                <button class="btn btn-sm btn-outline-secondary float-end" data-bs-toggle="collapse" data-bs-target="#upcomingExpiries">
                    <i class="bi bi-chevron-down"></i> Show/Hide
                </button>
            </h5>
        </div>
        <div class="collapse show" id="upcomingExpiries">
            <div class="card-body-modern">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Worker</th>
                                <th>Department</th>
                                <th>Due Date</th>
                                <th>Days Remaining</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_merge($alerts_by_level['warning'], $alerts_by_level['notice']) as $check): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($check['full_name']); ?></strong><br>
                                    <small><?php echo htmlspecialchars($check['employee_number']); ?></small>
                                </td>
                                <td><?php echo htmlspecialchars($check['department'] ?: '-'); ?></td>
                                <td><?php echo date('d/m/Y', strtotime($check['next_check_due'])); ?></td>
                                <td><?php echo $check['days_remaining']; ?> days</td>
                                <td>
                                    <?php if ($check['alert_level'] == 'warning'): ?>
                                        <span class="badge bg-info">Warning</span>
                                    <?php else: ?>
                                        <span class="badge bg-secondary">Notice</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="add_check.php?worker_id=<?php echo $check['worker_id']; ?>" class="btn btn-sm btn-outline-primary">
                                        <i class="bi bi-plus-circle"></i> Schedule
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>

    <!-- Pending Reminders -->
    <?php if (!empty($pending_reminders)): ?>
    <div class="modern-card">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-clock"></i> Pending Reminders
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Worker</th>
                            <th>Type</th>
                            <th>Reminder Date</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pending_reminders as $reminder): ?>
                        <tr>
                            <td>
                                <?php echo htmlspecialchars($reminder['full_name']); ?><br>
                                <small><?php echo htmlspecialchars($reminder['employee_number']); ?></small>
                            </td>
                            <td><?php echo ucfirst(str_replace('_', ' ', $reminder['reminder_type'])); ?></td>
                            <td><?php echo date('d/m/Y', strtotime($reminder['reminder_date'])); ?></td>
                            <td>
                                <?php if ($reminder['acknowledged']): ?>
                                    <span class="badge bg-success">Acknowledged</span>
                                    <small>by <?php echo htmlspecialchars($reminder['ack_fname'] . ' ' . $reminder['ack_lname']); ?></small>
                                <?php elseif ($reminder['sent']): ?>
                                    <span class="badge bg-info">Sent</span>
                                <?php else: ?>
                                    <span class="badge bg-warning">Pending</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!$reminder['acknowledged']): ?>
                                <form method="POST" class="d-inline">
                                    <input type="hidden" name="reminder_id" value="<?php echo $reminder['id']; ?>">
                                    <button type="submit" name="acknowledge_reminder" class="btn btn-sm btn-success">
                                        <i class="bi bi-check"></i> Acknowledge
                                    </button>
                                </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Send Alerts Modal -->
<div class="modal fade" id="sendAlertsModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Send Alert Emails</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p>This will send email alerts for:</p>
                <ul>
                    <li><strong><?php echo count($alerts_by_level['overdue']); ?></strong> overdue checks</li>
                    <li><strong><?php echo count($alerts_by_level['critical']); ?></strong> critical alerts (due within 7 days)</li>
                    <li><strong><?php echo count($alerts_by_level['warning']); ?></strong> warning alerts (due within 30 days)</li>
                </ul>
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> Emails will be sent to workers and their managers based on the alert level.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="sendAlertEmails()">
                    <i class="bi bi-send"></i> Send Alerts
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Create Reminder Modal -->
<div class="modal fade" id="createReminderModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Create Reminder</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <input type="hidden" name="worker_id" id="reminder_worker_id">
                    <div class="mb-3">
                        <label class="form-label">Reminder Type</label>
                        <select name="reminder_type" class="form-select" required>
                            <option value="expiry">Expiry Reminder</option>
                            <option value="follow_up">Follow-up Required</option>
                            <option value="overdue">Overdue Alert</option>
                            <option value="missing">Missing Documents</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Reminder Date</label>
                        <input type="date" name="reminder_date" class="form-control" required
                               min="<?php echo date('Y-m-d'); ?>">
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="create_reminder" class="btn btn-primary">
                        <i class="bi bi-bell"></i> Create Reminder
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function createReminder(workerId, type) {
    document.getElementById('reminder_worker_id').value = workerId;
    document.querySelector('select[name="reminder_type"]').value = type || 'expiry';

    // Set default date based on type
    const reminderDate = document.querySelector('input[name="reminder_date"]');
    const today = new Date();
    if (type === 'overdue') {
        reminderDate.value = today.toISOString().split('T')[0];
    } else {
        today.setDate(today.getDate() + 7);
        reminderDate.value = today.toISOString().split('T')[0];
    }

    new bootstrap.Modal(document.getElementById('createReminderModal')).show();
}

function sendAlertEmails() {
    if (confirm('Are you sure you want to send alert emails?')) {
        // Implementation would send emails via AJAX
        alert('Alert emails would be sent (implementation required)');
    }
}
</script>

<style>
.stat-card {
    padding: 1.5rem;
    border-radius: 0.5rem;
    height: 100%;
}

.stat-value {
    font-size: 2.5rem;
    font-weight: bold;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 0.875rem;
    opacity: 0.9;
    text-transform: uppercase;
}
</style>

<?php include '../../includes/footer.php'; ?>