<?php
/**
 * RTW Document Upload & Management
 * Handles document evidence for Right to Work checks
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(60);

$page_title = 'Upload RTW Documents';
$current_page = 'rtw';

$check_id = $_GET['check_id'] ?? '';
$worker_id = $_GET['worker_id'] ?? '';
$errors = [];
$success = [];

// Get check details if check_id provided
$check_details = null;
if ($check_id) {
    $stmt = $pdo->prepare("
        SELECT c.*, w.full_name, w.employee_number
        FROM " . DB_PREFIX . "worker_rtw_checks c
        JOIN " . DB_PREFIX . "sponsored_workers w ON c.worker_id = w.id
        WHERE c.id = ?
    ");
    $stmt->execute([$check_id]);
    $check_details = $stmt->fetch();
    $worker_id = $check_details['worker_id'];
}

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['rtw_documents'])) {
    $upload_dir = '../../uploads/rtw/' . date('Y') . '/' . date('m') . '/';

    // Create directory if it doesn't exist
    if (!file_exists($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }

    $uploaded_files = [];
    $files = $_FILES['rtw_documents'];

    // Reorganize files array for multiple uploads
    $file_count = count($files['name']);
    for ($i = 0; $i < $file_count; $i++) {
        if ($files['error'][$i] === UPLOAD_ERR_OK) {
            $tmp_name = $files['tmp_name'][$i];
            $original_name = $files['name'][$i];
            $file_size = $files['size'][$i];
            $mime_type = $files['type'][$i];

            // Validate file type
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'application/pdf'];
            if (!in_array($mime_type, $allowed_types)) {
                $errors[] = "File $original_name: Invalid file type. Only JPEG, PNG, GIF and PDF allowed.";
                continue;
            }

            // Validate file size (max 10MB)
            if ($file_size > 10485760) {
                $errors[] = "File $original_name: File size exceeds 10MB limit.";
                continue;
            }

            // Generate unique filename
            $ext = pathinfo($original_name, PATHINFO_EXTENSION);
            $safe_name = 'rtw_' . $worker_id . '_' . uniqid() . '.' . $ext;
            $file_path = $upload_dir . $safe_name;

            // Move uploaded file
            if (move_uploaded_file($tmp_name, $file_path)) {
                // Save to database
                $stmt = $pdo->prepare("
                    INSERT INTO " . DB_PREFIX . "rtw_documents
                    (check_id, worker_id, document_name, file_name, file_path, file_size, mime_type, page_number, uploaded_by)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");

                $stmt->execute([
                    $check_id ?: null,
                    $worker_id,
                    $_POST['document_names'][$i] ?? $original_name,
                    $safe_name,
                    str_replace('../../', '', $file_path),
                    $file_size,
                    $mime_type,
                    $_POST['page_numbers'][$i] ?? 1,
                    $_SESSION['user_id']
                ]);

                $uploaded_files[] = $original_name;

                // Log audit
                $audit_stmt = $pdo->prepare("
                    INSERT INTO " . DB_PREFIX . "rtw_audit
                    (worker_id, check_id, action, details, performed_by, ip_address)
                    VALUES (?, ?, ?, ?, ?, ?)
                ");
                $audit_stmt->execute([
                    $worker_id,
                    $check_id,
                    'DOCUMENT_UPLOADED',
                    "Uploaded: $original_name",
                    $_SESSION['user_id'],
                    $_SERVER['REMOTE_ADDR']
                ]);
            } else {
                $errors[] = "Failed to upload $original_name";
            }
        }
    }

    if (!empty($uploaded_files)) {
        $success[] = "Successfully uploaded " . count($uploaded_files) . " document(s)";
    }
}

// Get existing documents
$documents = [];
if ($worker_id) {
    $stmt = $pdo->prepare("
        SELECT d.*, u.first_name, u.last_name,
               v.first_name as verifier_fname, v.last_name as verifier_lname
        FROM " . DB_PREFIX . "rtw_documents d
        LEFT JOIN " . DB_PREFIX . "users u ON d.uploaded_by = u.id
        LEFT JOIN " . DB_PREFIX . "users v ON d.verified_by = v.id
        WHERE d.worker_id = ?
        " . ($check_id ? "AND d.check_id = ?" : "") . "
        ORDER BY d.uploaded_at DESC
    ");

    if ($check_id) {
        $stmt->execute([$worker_id, $check_id]);
    } else {
        $stmt->execute([$worker_id]);
    }
    $documents = $stmt->fetchAll();
}

// Handle document verification
if (isset($_POST['verify_document'])) {
    $doc_id = $_POST['document_id'];
    $stmt = $pdo->prepare("
        UPDATE " . DB_PREFIX . "rtw_documents
        SET verified = 1, verified_by = ?, verified_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$_SESSION['user_id'], $doc_id]);

    $_SESSION['success'] = 'Document verified successfully';
    header('Location: ' . $_SERVER['PHP_SELF'] . '?worker_id=' . $worker_id . ($check_id ? '&check_id=' . $check_id : ''));
    exit;
}

// Handle document deletion
if (isset($_POST['delete_document'])) {
    $doc_id = $_POST['document_id'];

    // Get file path first
    $stmt = $pdo->prepare("SELECT file_path FROM " . DB_PREFIX . "rtw_documents WHERE id = ?");
    $stmt->execute([$doc_id]);
    $doc = $stmt->fetch();

    if ($doc) {
        // Delete physical file
        $full_path = '../../' . $doc['file_path'];
        if (file_exists($full_path)) {
            unlink($full_path);
        }

        // Delete database record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "rtw_documents WHERE id = ?");
        $stmt->execute([$doc_id]);

        $_SESSION['success'] = 'Document deleted successfully';
    }

    header('Location: ' . $_SERVER['PHP_SELF'] . '?worker_id=' . $worker_id . ($check_id ? '&check_id=' . $check_id : ''));
    exit;
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="row">
        <div class="col-lg-12">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-cloud-upload"></i> RTW Document Management
                        <?php if ($check_details): ?>
                            <small class="text-muted">
                                - <?php echo htmlspecialchars($check_details['full_name']); ?>
                                (Check: <?php echo date('d/m/Y', strtotime($check_details['check_date'])); ?>)
                            </small>
                        <?php endif; ?>
                    </h5>
                </div>

                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($success)): ?>
                        <div class="alert alert-success">
                            <?php foreach ($success as $msg): ?>
                                <div><?php echo htmlspecialchars($msg); ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <?php if (isset($_SESSION['success'])): ?>
                        <div class="alert alert-success">
                            <?php echo htmlspecialchars($_SESSION['success']); ?>
                        </div>
                        <?php unset($_SESSION['success']); ?>
                    <?php endif; ?>

                    <!-- Upload Form -->
                    <div class="card mb-4">
                        <div class="card-header bg-primary text-white">
                            <h6 class="mb-0"><i class="bi bi-upload"></i> Upload New Documents</h6>
                        </div>
                        <div class="card-body">
                            <form method="POST" enctype="multipart/form-data" id="uploadForm">
                                <input type="hidden" name="worker_id" value="<?php echo $worker_id; ?>">
                                <?php if ($check_id): ?>
                                    <input type="hidden" name="check_id" value="<?php echo $check_id; ?>">
                                <?php endif; ?>

                                <div id="fileInputsContainer">
                                    <div class="file-input-group mb-3">
                                        <div class="row">
                                            <div class="col-md-4">
                                                <label class="form-label">Document Type</label>
                                                <input type="text" name="document_names[]" class="form-control"
                                                       placeholder="e.g., Passport Front" required>
                                            </div>
                                            <div class="col-md-2">
                                                <label class="form-label">Page #</label>
                                                <input type="number" name="page_numbers[]" class="form-control"
                                                       value="1" min="1">
                                            </div>
                                            <div class="col-md-6">
                                                <label class="form-label">Select File</label>
                                                <input type="file" name="rtw_documents[]" class="form-control"
                                                       accept=".jpg,.jpeg,.png,.gif,.pdf" required>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="d-flex justify-content-between">
                                    <button type="button" class="btn btn-secondary" id="addMoreFiles">
                                        <i class="bi bi-plus-circle"></i> Add More Files
                                    </button>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="bi bi-cloud-upload"></i> Upload Documents
                                    </button>
                                </div>

                                <div class="alert alert-info mt-3">
                                    <small>
                                        <strong>Requirements:</strong>
                                        <ul class="mb-0">
                                            <li>Accepted formats: JPEG, PNG, GIF, PDF</li>
                                            <li>Maximum file size: 10MB per file</li>
                                            <li>Ensure documents are clear and legible</li>
                                            <li>Include all pages of multi-page documents</li>
                                        </ul>
                                    </small>
                                </div>
                            </form>
                        </div>
                    </div>

                    <!-- Existing Documents -->
                    <?php if (!empty($documents)): ?>
                    <div class="card">
                        <div class="card-header bg-secondary text-white">
                            <h6 class="mb-0"><i class="bi bi-files"></i> Uploaded Documents</h6>
                        </div>
                        <div class="card-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Document</th>
                                            <th>Type</th>
                                            <th>Size</th>
                                            <th>Uploaded</th>
                                            <th>Verified</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($documents as $doc): ?>
                                        <tr>
                                            <td>
                                                <?php echo htmlspecialchars($doc['document_name']); ?>
                                                <?php if ($doc['page_number'] > 1): ?>
                                                    <span class="badge bg-secondary">Page <?php echo $doc['page_number']; ?></span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php
                                                $icon = 'bi-file-earmark';
                                                if (strpos($doc['mime_type'], 'pdf') !== false) {
                                                    $icon = 'bi-file-earmark-pdf text-danger';
                                                } elseif (strpos($doc['mime_type'], 'image') !== false) {
                                                    $icon = 'bi-file-earmark-image text-primary';
                                                }
                                                ?>
                                                <i class="bi <?php echo $icon; ?>"></i>
                                                <?php echo strtoupper(pathinfo($doc['file_name'], PATHINFO_EXTENSION)); ?>
                                            </td>
                                            <td><?php echo number_format($doc['file_size'] / 1024, 1); ?> KB</td>
                                            <td>
                                                <?php echo date('d/m/Y H:i', strtotime($doc['uploaded_at'])); ?><br>
                                                <small class="text-muted">
                                                    by <?php echo htmlspecialchars($doc['first_name'] . ' ' . $doc['last_name']); ?>
                                                </small>
                                            </td>
                                            <td>
                                                <?php if ($doc['verified']): ?>
                                                    <span class="badge bg-success">
                                                        <i class="bi bi-check-circle"></i> Verified
                                                    </span><br>
                                                    <small class="text-muted">
                                                        <?php echo date('d/m/Y', strtotime($doc['verified_at'])); ?>
                                                        by <?php echo htmlspecialchars($doc['verifier_fname'] . ' ' . $doc['verifier_lname']); ?>
                                                    </small>
                                                <?php else: ?>
                                                    <span class="badge bg-warning">Pending</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <a href="../../<?php echo $doc['file_path']; ?>"
                                                       target="_blank" class="btn btn-sm btn-outline-primary"
                                                       title="View">
                                                        <i class="bi bi-eye"></i>
                                                    </a>

                                                    <?php if (!$doc['verified'] && $_SESSION['permission_level'] >= 80): ?>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="document_id" value="<?php echo $doc['id']; ?>">
                                                        <button type="submit" name="verify_document"
                                                                class="btn btn-sm btn-outline-success"
                                                                title="Verify"
                                                                onclick="return confirm('Verify this document?')">
                                                            <i class="bi bi-check-circle"></i>
                                                        </button>
                                                    </form>
                                                    <?php endif; ?>

                                                    <?php if ($_SESSION['permission_level'] >= 90): ?>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="document_id" value="<?php echo $doc['id']; ?>">
                                                        <button type="submit" name="delete_document"
                                                                class="btn btn-sm btn-outline-danger"
                                                                title="Delete"
                                                                onclick="return confirm('Delete this document? This action cannot be undone.')">
                                                            <i class="bi bi-trash"></i>
                                                        </button>
                                                    </form>
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> No documents uploaded yet for this worker.
                    </div>
                    <?php endif; ?>

                    <!-- Action Buttons -->
                    <div class="mt-4">
                        <?php if ($check_id): ?>
                            <a href="view_check.php?id=<?php echo $check_id; ?>" class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Back to Check
                            </a>
                        <?php elseif ($worker_id): ?>
                            <a href="../workers/view.php?id=<?php echo $worker_id; ?>" class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Back to Worker
                            </a>
                        <?php else: ?>
                            <a href="list.php" class="btn btn-secondary">
                                <i class="bi bi-arrow-left"></i> Back to RTW Checks
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('addMoreFiles').addEventListener('click', function() {
    const container = document.getElementById('fileInputsContainer');
    const newGroup = document.createElement('div');
    newGroup.className = 'file-input-group mb-3';
    newGroup.innerHTML = `
        <div class="row">
            <div class="col-md-4">
                <input type="text" name="document_names[]" class="form-control"
                       placeholder="e.g., BRP Back" required>
            </div>
            <div class="col-md-2">
                <input type="number" name="page_numbers[]" class="form-control"
                       value="1" min="1">
            </div>
            <div class="col-md-5">
                <input type="file" name="rtw_documents[]" class="form-control"
                       accept=".jpg,.jpeg,.png,.gif,.pdf" required>
            </div>
            <div class="col-md-1">
                <button type="button" class="btn btn-danger btn-sm" onclick="this.closest('.file-input-group').remove()">
                    <i class="bi bi-trash"></i>
                </button>
            </div>
        </div>
    `;
    container.appendChild(newGroup);
});
</script>

<?php include '../../includes/footer.php'; ?>