<?php
/**
 * RTW (Right to Work) Module - Delete RTW Check Record
 * Role-based deletion:
 * - Permission 70-89: Can delete RTW check records
 * - Permission 90-99: Can delete RTW check records
 * - Permission 100+: Can PERMANENT delete with PIN
 *
 * Note: RTW checks are critical compliance records - deletion is logged extensively
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        header('Location: ../../login.php');
        exit;
    }
}

// Check minimum permission - delete requires level 70
if ($current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete RTW check records.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
    exit;
}

$rtw_id = (int)($_POST['id'] ?? 0);
$pin = $_POST['pin'] ?? '';

if ($rtw_id <= 0) {
    $_SESSION['error'] = 'Invalid RTW check ID.';
    header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if RTW check exists
    $stmt = $pdo->prepare("SELECT r.*, sw.full_name, sw.employee_number
                          FROM " . DB_PREFIX . "worker_rtw_checks r
                          LEFT JOIN " . DB_PREFIX . "sponsored_workers sw ON r.worker_id = sw.id
                          WHERE r.id = ?");
    $stmt->execute([$rtw_id]);
    $rtw = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$rtw) {
        $_SESSION['error'] = 'RTW check record not found.';
        header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
        exit;
    }

    $permission_level = $current_user['permission_level'];

    // For permission 70-99: Allow deletion with audit log
    if ($permission_level >= 70 && $permission_level < 100) {
        // Log the deletion for audit trail (IMPORTANT for compliance)
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'DELETE', 'worker_rtw_checks', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'worker_name' => $rtw['full_name'],
            'employee_number' => $rtw['employee_number'],
            'check_date' => $rtw['check_date'],
            'check_type' => $rtw['check_type'],
            'checker_name' => $rtw['checker_name'],
            'outcome' => $rtw['outcome'],
            'reason' => 'User deleted RTW check record'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $rtw_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "worker_rtw_checks WHERE id = ?");
        $stmt->execute([$rtw_id]);

        $_SESSION['success'] = 'RTW check for ' . htmlspecialchars($rtw['full_name']) . ' dated ' . date('d/m/Y', strtotime($rtw['check_date'])) . ' has been deleted.';
    }

    // For permission 100+: Require PIN for deletion
    elseif ($permission_level >= 100) {
        // REQUIRE PIN for admin deletion
        if (empty($pin)) {
            $_SESSION['error'] = 'PIN is required for deletion.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
            exit;
        }

        // Verify PIN
        if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
            $_SESSION['error'] = 'Invalid PIN. Deletion cancelled.';
            header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
            exit;
        }

        // Log the deletion for audit trail (IMPORTANT for compliance)
        $audit_stmt = $pdo->prepare("
            INSERT INTO " . DB_PREFIX . "audit_log
            (user_id, action, table_name, record_id, details, ip_address, created_at)
            VALUES (?, 'ADMIN_DELETE', 'worker_rtw_checks', ?, ?, ?, NOW())
        ");
        $details = json_encode([
            'worker_name' => $rtw['full_name'],
            'employee_number' => $rtw['employee_number'],
            'check_date' => $rtw['check_date'],
            'check_type' => $rtw['check_type'],
            'checker_name' => $rtw['checker_name'],
            'outcome' => $rtw['outcome'],
            'reason' => 'Admin deletion with PIN verification'
        ]);
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $audit_stmt->execute([$current_user['id'], $rtw_id, $details, $ip]);

        // Delete the record
        $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "worker_rtw_checks WHERE id = ?");
        $stmt->execute([$rtw_id]);

        $_SESSION['success'] = 'RTW check for ' . htmlspecialchars($rtw['full_name']) . ' dated ' . date('d/m/Y', strtotime($rtw['check_date'])) . ' has been PERMANENTLY DELETED.';
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting RTW check: ' . $e->getMessage();
}

// Redirect back to list
header('Location: ' . (defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php'));
exit;
?>
