<?php
/**
 * Cron Job: Share Code Reminder Emails
 * Run daily to send reminders for expiring RTW checks
 *
 * Setup cron to run daily:
 * 0 9 * * * /usr/bin/php /path/to/scms/modules/rtw/cron_share_code_reminders.php
 */

// Allow this script to run from CLI only
if (php_sapi_name() !== 'cli') {
    die('This script can only be run from the command line.');
}

require_once dirname(__FILE__) . '/../../config/database.php';
require_once dirname(__FILE__) . '/../../includes/database.php';
require_once dirname(__FILE__) . '/../../includes/email_helper.php';
require_once dirname(__FILE__) . '/email_templates.php';
require_once dirname(__FILE__) . '/includes/alert_helper.php';

echo "[" . date('Y-m-d H:i:s') . "] Starting RTW Share Code Reminder Cron Job\n";

// Get module settings
$settings = [];
$stmt = $pdo->query("SELECT setting_key, setting_value FROM " . DB_PREFIX . "system_settings WHERE setting_key LIKE 'rtw_%' OR setting_key LIKE 'smtp_%'");
while ($row = $stmt->fetch()) {
    $settings[$row['setting_key']] = $row['setting_value'];
}

// Check if reminders are enabled
if (empty($settings['rtw_share_code_reminder_enabled']) || $settings['rtw_share_code_reminder_enabled'] != '1') {
    echo "Share code reminders are disabled in settings. Exiting.\n";
    exit;
}

// Check if SMTP is enabled
if (empty($settings['smtp_enabled']) || $settings['smtp_enabled'] != '1') {
    echo "SMTP is not enabled. Cannot send emails. Exiting.\n";
    exit;
}

$reminder_frequency = $settings['rtw_share_code_reminder_frequency'] ?? 30;
$advance_notice = $settings['rtw_share_code_advance_notice'] ?? 14;
$expiry_alert_days = explode(',', $settings['rtw_expiry_alert_days'] ?? '30,14,7,1');

// Initialize helpers
$emailHelper = new EmailHelper($pdo);
$alertHelper = new RTWAlertHelper($pdo);

$emails_sent = 0;
$errors = 0;

try {
    // 1. Find workers with expiring RTW checks
    $sql = "SELECT w.*, r.check_date, r.document_type, r.next_check_due,
                   DATEDIFF(r.next_check_due, CURDATE()) as days_until_expiry
            FROM " . DB_PREFIX . "sponsored_workers w
            INNER JOIN " . DB_PREFIX . "worker_rtw_checks r ON w.id = r.worker_id
            WHERE r.next_check_due IS NOT NULL
            AND r.next_check_due <= DATE_ADD(CURDATE(), INTERVAL ? DAY)
            AND r.outcome = 'pass'
            AND w.email IS NOT NULL AND w.email != ''
            GROUP BY w.id
            ORDER BY r.next_check_due ASC";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([$advance_notice]);
    $workers = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo "Found " . count($workers) . " workers with expiring RTW checks\n";

    foreach ($workers as $worker) {
        $days_until = $worker['days_until_expiry'];

        // Check if we should send a reminder today based on alert days
        $should_send = false;
        foreach ($expiry_alert_days as $alert_day) {
            if ($days_until == intval(trim($alert_day))) {
                $should_send = true;
                break;
            }
        }

        // Also send if overdue
        if ($days_until < 0) {
            // Check last reminder date to avoid spamming
            $stmt = $pdo->prepare("SELECT MAX(sent_at) as last_sent FROM " . DB_PREFIX . "rtw_reminder_log
                                  WHERE worker_id = ? AND reminder_type = 'expiry_alert'");
            $stmt->execute([$worker['id']]);
            $last_sent = $stmt->fetchColumn();

            if (!$last_sent || strtotime($last_sent) < strtotime('-7 days')) {
                $should_send = true;
            }
        }

        if (!$should_send) {
            continue;
        }

        echo "Sending reminder to {$worker['full_name']} ({$worker['email']}) - {$days_until} days until expiry\n";

        // Get email template
        $email_data = RTWEmailTemplates::getExpiryAlertEmail(
            $worker['full_name'],
            $days_until,
            $worker['document_type'],
            $worker['check_date']
        );

        // Add submission link to email body
        $submission_link = "https://{$_SERVER['HTTP_HOST']}/modules/rtw/submit_share_code.php";
        $email_data['body'] .= "
        <div style='background-color: #007bff; padding: 20px; margin: 20px 0; text-align: center; border-radius: 5px;'>
            <a href='{$submission_link}' style='color: white; text-decoration: none; font-size: 18px; font-weight: bold;'>
                Click here to submit your new Share Code
            </a>
        </div>";

        // Send email
        $result = $emailHelper->sendEmail(
            $worker['email'],
            $worker['full_name'],
            $email_data['subject'],
            $email_data['body'],
            true
        );

        if ($result['success']) {
            $emails_sent++;

            // Log the reminder
            $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "rtw_reminder_log
                                  (worker_id, reminder_type, sent_at, sent_to, email_status, next_reminder_date)
                                  VALUES (?, 'expiry_alert', NOW(), ?, 'sent', DATE_ADD(CURDATE(), INTERVAL ? DAY))");

            $next_reminder_days = min(7, max(1, intval($days_until / 2)));
            $stmt->execute([$worker['id'], $worker['email'], $next_reminder_days]);

            // Create/Update alert for expiry
            $alertHelper->createExpiryAlert($worker['id'], $worker['full_name'], $days_until, $worker['check_date']);

            // Log activity
            $alertHelper->logActivity('reminder_sent',
                "RTW expiry reminder sent to {$worker['full_name']} ({$worker['email']}) - {$days_until} days until expiry",
                $worker['id']);

        } else {
            $errors++;
            echo "Failed to send email to {$worker['email']}: {$result['message']}\n";
        }
    }

    // 2. Send reminders for pending share code submissions
    $sql = "SELECT s.*, w.full_name, w.email
            FROM " . DB_PREFIX . "rtw_share_code_submissions s
            LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON s.worker_id = w.id
            WHERE s.submission_status = 'pending'
            AND s.submitted_at < DATE_SUB(NOW(), INTERVAL 48 HOUR)
            AND w.email IS NOT NULL";

    $stmt = $pdo->query($sql);
    $pending_submissions = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo "\nFound " . count($pending_submissions) . " pending share code submissions needing follow-up\n";

    // 3. Send HR digest if there are overdue checks or pending submissions
    if (!empty($settings['rtw_hr_notification_email'])) {
        $overdue_sql = "SELECT w.full_name, w.employee_number, r.next_check_due,
                               DATEDIFF(CURDATE(), r.next_check_due) as days_overdue,
                               r.check_date as last_check
                        FROM " . DB_PREFIX . "worker_rtw_checks r
                        LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
                        WHERE r.next_check_due < CURDATE()
                        AND r.outcome = 'pass'
                        ORDER BY r.next_check_due ASC";

        $overdue_checks = $pdo->query($overdue_sql)->fetchAll(PDO::FETCH_ASSOC);

        $expiring_sql = "SELECT w.full_name, w.employee_number, r.next_check_due as expiry_date,
                                DATEDIFF(r.next_check_due, CURDATE()) as days_until
                         FROM " . DB_PREFIX . "worker_rtw_checks r
                         LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON r.worker_id = w.id
                         WHERE r.next_check_due BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
                         AND r.outcome = 'pass'
                         ORDER BY r.next_check_due ASC";

        $expiring_checks = $pdo->query($expiring_sql)->fetchAll(PDO::FETCH_ASSOC);

        if (!empty($overdue_checks) || !empty($expiring_checks)) {
            echo "\nSending HR notification email to {$settings['rtw_hr_notification_email']}\n";
            echo "- " . count($overdue_checks) . " overdue checks\n";
            echo "- " . count($expiring_checks) . " expiring checks\n";

            $hr_email = RTWEmailTemplates::getHRNotificationEmail($expiring_checks, $overdue_checks);

            $result = $emailHelper->sendEmail(
                $settings['rtw_hr_notification_email'],
                'HR Department',
                $hr_email['subject'],
                $hr_email['body'],
                true
            );

            if ($result['success']) {
                $emails_sent++;
                echo "HR notification sent successfully\n";
            } else {
                $errors++;
                echo "Failed to send HR notification: {$result['message']}\n";
            }
        }
    }

} catch (Exception $e) {
    echo "Error: " . $e->getMessage() . "\n";
    $errors++;
}

echo "\n[" . date('Y-m-d H:i:s') . "] RTW Reminder Cron Job Completed\n";
echo "Emails sent: {$emails_sent}\n";
echo "Errors: {$errors}\n";

// Log to database
try {
    $stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "system_logs
                          (log_type, log_message, created_at)
                          VALUES ('rtw_cron', ?, NOW())");

    $log_message = "RTW reminder cron completed. Emails sent: {$emails_sent}, Errors: {$errors}";
    $stmt->execute([$log_message]);
} catch (Exception $e) {
    // Ignore logging errors
}

exit($errors > 0 ? 1 : 0);