<?php
/**
 * RTW Compliance Report
 * Comprehensive compliance reporting for Right to Work checks
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();
require_permission(70);

$page_title = 'RTW Compliance Report';
$current_page = 'rtw';

// Get filter parameters
$from_date = $_GET['from'] ?? date('Y-m-01'); // Default to start of month
$to_date = $_GET['to'] ?? date('Y-m-d');
$department = $_GET['department'] ?? '';
$location = $_GET['location'] ?? '';

// Generate report data
$report_data = [];

// 1. Overall Compliance Statistics
$stmt = $pdo->prepare("
    SELECT
        COUNT(DISTINCT w.id) as total_workers,
        COUNT(DISTINCT CASE WHEN c.id IS NOT NULL THEN w.id END) as workers_with_checks,
        COUNT(DISTINCT CASE WHEN c.outcome = 'pass' AND c.next_check_due >= CURDATE() THEN w.id END) as compliant_workers,
        COUNT(DISTINCT CASE WHEN c.next_check_due < CURDATE() THEN w.id END) as overdue_workers,
        COUNT(DISTINCT CASE WHEN c.id IS NULL THEN w.id END) as no_checks_workers
    FROM " . DB_PREFIX . "sponsored_workers w
    LEFT JOIN " . DB_PREFIX . "worker_rtw_checks c ON w.id = c.worker_id
        AND c.id = (SELECT MAX(id) FROM " . DB_PREFIX . "worker_rtw_checks WHERE worker_id = w.id)
    WHERE w.status = 'active'
    " . ($department ? "AND w.department = ?" : "") . "
    " . ($location ? "AND w.location = ?" : "")
);

$params = [];
if ($department) $params[] = $department;
if ($location) $params[] = $location;
$stmt->execute($params);
$overall_stats = $stmt->fetch();

// Calculate compliance rate
$compliance_rate = 0;
if ($overall_stats['total_workers'] > 0) {
    $compliance_rate = ($overall_stats['compliant_workers'] / $overall_stats['total_workers']) * 100;
}

// 2. Checks performed in date range
$stmt = $pdo->prepare("
    SELECT
        COUNT(*) as total_checks,
        SUM(CASE WHEN outcome = 'pass' THEN 1 ELSE 0 END) as passed,
        SUM(CASE WHEN outcome = 'fail' THEN 1 ELSE 0 END) as failed,
        SUM(CASE WHEN outcome = 'refer' THEN 1 ELSE 0 END) as referred,
        SUM(CASE WHEN check_method = 'online' THEN 1 ELSE 0 END) as online_checks,
        SUM(CASE WHEN check_method = 'manual' THEN 1 ELSE 0 END) as manual_checks
    FROM " . DB_PREFIX . "worker_rtw_checks
    WHERE check_date BETWEEN ? AND ?
");
$stmt->execute([$from_date, $to_date]);
$period_stats = $stmt->fetch();

// 3. Upcoming checks
$stmt = $pdo->prepare("
    SELECT
        COUNT(CASE WHEN DATEDIFF(next_check_due, CURDATE()) <= 7 THEN 1 END) as next_7_days,
        COUNT(CASE WHEN DATEDIFF(next_check_due, CURDATE()) BETWEEN 8 AND 30 THEN 1 END) as next_30_days,
        COUNT(CASE WHEN DATEDIFF(next_check_due, CURDATE()) BETWEEN 31 AND 90 THEN 1 END) as next_90_days
    FROM " . DB_PREFIX . "worker_rtw_checks
    WHERE next_check_due >= CURDATE()
");
$stmt->execute();
$upcoming = $stmt->fetch();

// 4. Workers with missing or expired checks (HIGH RISK)
$stmt = $pdo->prepare("
    SELECT
        w.id,
        w.full_name,
        w.employee_number,
        w.department,
        w.location,
        w.start_date,
        c.check_date as last_check_date,
        c.next_check_due,
        c.outcome as last_outcome,
        DATEDIFF(CURDATE(), c.next_check_due) as days_overdue
    FROM " . DB_PREFIX . "sponsored_workers w
    LEFT JOIN " . DB_PREFIX . "worker_rtw_checks c ON w.id = c.worker_id
        AND c.id = (SELECT MAX(id) FROM " . DB_PREFIX . "worker_rtw_checks WHERE worker_id = w.id)
    WHERE w.status = 'active'
    AND (c.id IS NULL OR c.next_check_due < CURDATE())
    " . ($department ? "AND w.department = ?" : "") . "
    " . ($location ? "AND w.location = ?" : "") . "
    ORDER BY days_overdue DESC NULLS FIRST
    LIMIT 50
");

$params = [];
if ($department) $params[] = $department;
if ($location) $params[] = $location;
$stmt->execute($params);
$high_risk_workers = $stmt->fetchAll();

// 5. Document statistics
$stmt = $pdo->prepare("
    SELECT
        COUNT(*) as total_documents,
        SUM(CASE WHEN verified = 1 THEN 1 ELSE 0 END) as verified_documents,
        COUNT(DISTINCT worker_id) as workers_with_documents
    FROM " . DB_PREFIX . "rtw_documents
");
$stmt->execute();
$doc_stats = $stmt->fetch();

// 6. Statutory excuse coverage
$stmt = $pdo->prepare("
    SELECT
        COUNT(DISTINCT worker_id) as workers_with_excuse,
        SUM(CASE WHEN valid_until >= CURDATE() THEN 1 ELSE 0 END) as valid_excuses,
        SUM(CASE WHEN valid_until < CURDATE() THEN 1 ELSE 0 END) as expired_excuses
    FROM " . DB_PREFIX . "rtw_statutory_excuse
");
$stmt->execute();
$excuse_stats = $stmt->fetch();

// Get departments and locations for filters
$departments = $pdo->query("SELECT DISTINCT department FROM " . DB_PREFIX . "sponsored_workers WHERE department IS NOT NULL ORDER BY department")->fetchAll(PDO::FETCH_COLUMN);
$locations = $pdo->query("SELECT DISTINCT location FROM " . DB_PREFIX . "sponsored_workers WHERE location IS NOT NULL ORDER BY location")->fetchAll(PDO::FETCH_COLUMN);

// Handle export
if (isset($_GET['export'])) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="rtw_compliance_report_' . date('Y-m-d') . '.csv"');

    $output = fopen('php://output', 'w');

    // Write headers
    fputcsv($output, ['RTW Compliance Report - Generated: ' . date('d/m/Y H:i')]);
    fputcsv($output, ['Period: ' . date('d/m/Y', strtotime($from_date)) . ' to ' . date('d/m/Y', strtotime($to_date))]);
    fputcsv($output, []);

    // Summary statistics
    fputcsv($output, ['SUMMARY STATISTICS']);
    fputcsv($output, ['Metric', 'Value']);
    fputcsv($output, ['Total Active Workers', $overall_stats['total_workers']]);
    fputcsv($output, ['Compliant Workers', $overall_stats['compliant_workers']]);
    fputcsv($output, ['Compliance Rate', number_format($compliance_rate, 1) . '%']);
    fputcsv($output, ['Overdue Checks', $overall_stats['overdue_workers']]);
    fputcsv($output, ['Workers Without Checks', $overall_stats['no_checks_workers']]);
    fputcsv($output, []);

    // High risk workers
    fputcsv($output, ['HIGH RISK WORKERS - MISSING OR EXPIRED CHECKS']);
    fputcsv($output, ['Employee Number', 'Name', 'Department', 'Location', 'Last Check', 'Days Overdue']);
    foreach ($high_risk_workers as $worker) {
        fputcsv($output, [
            $worker['employee_number'],
            $worker['full_name'],
            $worker['department'],
            $worker['location'],
            $worker['last_check_date'] ? date('d/m/Y', strtotime($worker['last_check_date'])) : 'Never',
            $worker['days_overdue'] ?: 'No check'
        ]);
    }

    fclose($output);
    exit;
}

include '../../includes/header.php';
?>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title">
                <i class="bi bi-graph-up"></i> RTW Compliance Report
            </h1>
            <p class="welcome-subtitle">Comprehensive compliance overview and risk assessment</p>
        </div>
        <div class="welcome-actions">
            <a href="?export=1&from=<?php echo $from_date; ?>&to=<?php echo $to_date; ?>&department=<?php echo $department; ?>&location=<?php echo $location; ?>" class="btn btn-success">
                <i class="bi bi-download"></i> Export CSV
            </a>
            <button onclick="window.print()" class="btn btn-secondary">
                <i class="bi bi-printer"></i> Print Report
            </button>
        </div>
    </div>

    <!-- Filters -->
    <div class="modern-card mb-4">
        <div class="card-body-modern">
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <label class="form-label">From Date</label>
                    <input type="date" name="from" class="form-control" value="<?php echo $from_date; ?>">
                </div>
                <div class="col-md-3">
                    <label class="form-label">To Date</label>
                    <input type="date" name="to" class="form-control" value="<?php echo $to_date; ?>">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Department</label>
                    <select name="department" class="form-select">
                        <option value="">All</option>
                        <?php foreach ($departments as $dept): ?>
                            <option value="<?php echo htmlspecialchars($dept); ?>" <?php echo $department == $dept ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($dept); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Location</label>
                    <select name="location" class="form-select">
                        <option value="">All</option>
                        <?php foreach ($locations as $loc): ?>
                            <option value="<?php echo htmlspecialchars($loc); ?>" <?php echo $location == $loc ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($loc); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">&nbsp;</label>
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="bi bi-filter"></i> Apply Filters
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Compliance Overview -->
    <div class="row g-3 mb-4">
        <div class="col-md-3">
            <div class="stat-card bg-primary text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo $overall_stats['total_workers']; ?></div>
                        <div class="stat-label">Total Workers</div>
                    </div>
                    <i class="bi bi-people-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card <?php echo $compliance_rate >= 90 ? 'bg-success' : ($compliance_rate >= 70 ? 'bg-warning' : 'bg-danger'); ?> text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo number_format($compliance_rate, 1); ?>%</div>
                        <div class="stat-label">Compliance Rate</div>
                    </div>
                    <i class="bi bi-check-circle-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-danger text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo $overall_stats['overdue_workers']; ?></div>
                        <div class="stat-label">Overdue Checks</div>
                    </div>
                    <i class="bi bi-exclamation-triangle-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stat-card bg-warning text-white">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <div class="stat-value"><?php echo $overall_stats['no_checks_workers']; ?></div>
                        <div class="stat-label">No RTW Check</div>
                    </div>
                    <i class="bi bi-question-circle-fill" style="font-size: 2rem; opacity: 0.7;"></i>
                </div>
            </div>
        </div>
    </div>

    <!-- Period Statistics -->
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-calendar-check"></i> Checks Performed (<?php echo date('d/m/Y', strtotime($from_date)); ?> - <?php echo date('d/m/Y', strtotime($to_date)); ?>)
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="row">
                <div class="col-md-6">
                    <canvas id="checksChart"></canvas>
                </div>
                <div class="col-md-6">
                    <table class="table">
                        <tr>
                            <th>Total Checks</th>
                            <td><?php echo $period_stats['total_checks']; ?></td>
                        </tr>
                        <tr>
                            <th>Passed</th>
                            <td><span class="badge bg-success"><?php echo $period_stats['passed']; ?></span></td>
                        </tr>
                        <tr>
                            <th>Failed</th>
                            <td><span class="badge bg-danger"><?php echo $period_stats['failed']; ?></span></td>
                        </tr>
                        <tr>
                            <th>Referred</th>
                            <td><span class="badge bg-warning"><?php echo $period_stats['referred']; ?></span></td>
                        </tr>
                        <tr>
                            <th>Online Checks</th>
                            <td><?php echo $period_stats['online_checks']; ?></td>
                        </tr>
                        <tr>
                            <th>Manual Checks</th>
                            <td><?php echo $period_stats['manual_checks']; ?></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Upcoming Checks -->
    <div class="modern-card mb-4">
        <div class="card-header-modern">
            <h5 class="card-title-modern">
                <i class="bi bi-calendar-event"></i> Upcoming Checks
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="row text-center">
                <div class="col-md-4">
                    <h3 class="text-danger"><?php echo $upcoming['next_7_days']; ?></h3>
                    <p>Next 7 Days</p>
                </div>
                <div class="col-md-4">
                    <h3 class="text-warning"><?php echo $upcoming['next_30_days']; ?></h3>
                    <p>8-30 Days</p>
                </div>
                <div class="col-md-4">
                    <h3 class="text-info"><?php echo $upcoming['next_90_days']; ?></h3>
                    <p>31-90 Days</p>
                </div>
            </div>
        </div>
    </div>

    <!-- High Risk Workers -->
    <?php if (!empty($high_risk_workers)): ?>
    <div class="modern-card mb-4">
        <div class="card-header-modern bg-danger text-white">
            <h5 class="card-title-modern mb-0">
                <i class="bi bi-exclamation-triangle"></i> High Risk - Workers with Missing or Expired Checks
            </h5>
        </div>
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Employee #</th>
                            <th>Name</th>
                            <th>Department</th>
                            <th>Location</th>
                            <th>Last Check</th>
                            <th>Status</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($high_risk_workers as $worker): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($worker['employee_number']); ?></td>
                            <td><?php echo htmlspecialchars($worker['full_name']); ?></td>
                            <td><?php echo htmlspecialchars($worker['department'] ?: '-'); ?></td>
                            <td><?php echo htmlspecialchars($worker['location'] ?: '-'); ?></td>
                            <td>
                                <?php if ($worker['last_check_date']): ?>
                                    <?php echo date('d/m/Y', strtotime($worker['last_check_date'])); ?>
                                <?php else: ?>
                                    <span class="text-danger">Never Checked</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if (!$worker['last_check_date']): ?>
                                    <span class="badge bg-danger">No Check</span>
                                <?php elseif ($worker['days_overdue'] > 0): ?>
                                    <span class="badge bg-danger"><?php echo $worker['days_overdue']; ?> days overdue</span>
                                <?php else: ?>
                                    <span class="badge bg-warning">Due</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <a href="add_check.php?worker_id=<?php echo $worker['id']; ?>" class="btn btn-sm btn-danger">
                                    <i class="bi bi-plus-circle"></i> Add Check
                                </a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php if (count($high_risk_workers) == 50): ?>
            <div class="alert alert-warning mt-3">
                <i class="bi bi-info-circle"></i> Showing first 50 high-risk workers. Export the report for complete list.
            </div>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- Document & Statutory Excuse Statistics -->
    <div class="row g-3">
        <div class="col-md-6">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-file-earmark-check"></i> Document Statistics
                    </h5>
                </div>
                <div class="card-body-modern">
                    <table class="table">
                        <tr>
                            <th>Total Documents</th>
                            <td><?php echo $doc_stats['total_documents']; ?></td>
                        </tr>
                        <tr>
                            <th>Verified Documents</th>
                            <td>
                                <?php echo $doc_stats['verified_documents']; ?>
                                <?php if ($doc_stats['total_documents'] > 0): ?>
                                    (<?php echo number_format(($doc_stats['verified_documents'] / $doc_stats['total_documents']) * 100, 1); ?>%)
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Workers with Documents</th>
                            <td><?php echo $doc_stats['workers_with_documents']; ?></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
        <div class="col-md-6">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-shield-check"></i> Statutory Excuse Coverage
                    </h5>
                </div>
                <div class="card-body-modern">
                    <table class="table">
                        <tr>
                            <th>Workers with Excuse</th>
                            <td><?php echo $excuse_stats['workers_with_excuse']; ?></td>
                        </tr>
                        <tr>
                            <th>Valid Excuses</th>
                            <td><span class="badge bg-success"><?php echo $excuse_stats['valid_excuses']; ?></span></td>
                        </tr>
                        <tr>
                            <th>Expired Excuses</th>
                            <td><span class="badge bg-danger"><?php echo $excuse_stats['expired_excuses']; ?></span></td>
                        </tr>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Chart.js for visualizations -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Checks outcome chart
const ctx = document.getElementById('checksChart').getContext('2d');
new Chart(ctx, {
    type: 'doughnut',
    data: {
        labels: ['Passed', 'Failed', 'Referred'],
        datasets: [{
            data: [
                <?php echo $period_stats['passed']; ?>,
                <?php echo $period_stats['failed']; ?>,
                <?php echo $period_stats['referred']; ?>
            ],
            backgroundColor: ['#28a745', '#dc3545', '#ffc107']
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom'
            },
            title: {
                display: true,
                text: 'Check Outcomes'
            }
        }
    }
});
</script>

<style>
.stat-card {
    padding: 1.5rem;
    border-radius: 0.5rem;
    height: 100%;
}

.stat-value {
    font-size: 2.5rem;
    font-weight: bold;
    margin-bottom: 0.5rem;
}

.stat-label {
    font-size: 0.875rem;
    opacity: 0.9;
    text-transform: uppercase;
}

@media print {
    .welcome-actions, .btn, form {
        display: none !important;
    }
}
</style>

<?php include '../../includes/footer.php'; ?>