<?php
/**
 * RTW Module - Add Right to Work Check
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
    require_permission(60);
}

// Get current user
$current_user = get_logged_in_user();
if (!$current_user) {
    header('Location: ../../login.php');
    exit;
}

$page_title = 'Add RTW Check';
$current_page = 'rtw';

$worker_id = $_GET['worker_id'] ?? '';
$errors = [];

// Get database connection
$pdo = get_database_connection();

// Get workers
$workers_sql = "SELECT id, full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers
                WHERE status = 'active' ORDER BY full_name";
$workers = $pdo->query($workers_sql)->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $worker_id = $_POST['worker_id'] ?? '';
    $check_date = $_POST['check_date'] ?? '';
    $check_type = $_POST['check_type'] ?? '';
    $documents_seen = trim($_POST['documents_seen'] ?? '');
    $share_code_used = trim($_POST['share_code_used'] ?? '');
    $online_check = isset($_POST['online_check']) ? 1 : 0;
    $outcome = $_POST['outcome'] ?? '';
    $next_check_due = $_POST['next_check_due'] ?? null;
    $notes = trim($_POST['notes'] ?? '');

    // Validation
    if (empty($worker_id)) $errors[] = "Worker is required";
    if (empty($check_date)) $errors[] = "Check date is required";
    if (empty($check_type)) $errors[] = "Check type is required";
    if (empty($documents_seen)) $errors[] = "Documents seen is required";
    if (empty($outcome)) $errors[] = "Outcome is required";

    if (empty($errors)) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "worker_rtw_checks
                (worker_id, check_date, check_type, documents_seen, share_code_used,
                 online_check, outcome, next_check_due, checked_by, notes, created_at)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
            ");

            $stmt->execute([
                $worker_id,
                $check_date,
                $check_type,
                $documents_seen,
                $share_code_used,
                $online_check,
                $outcome,
                $next_check_due ?: null,
                $_SESSION['user_id'],
                $notes
            ]);

            $rtw_check_id = $pdo->lastInsertId();

            // Auto-activate worker if RTW check passed and worker is pending_rtw
            if ($outcome === 'pass') {
                // Get worker status
                $worker_stmt = $pdo->prepare("SELECT status, full_name FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
                $worker_stmt->execute([$worker_id]);
                $worker = $worker_stmt->fetch();

                if ($worker && $worker['status'] === 'pending_rtw') {
                    // Activate worker with check_date as start_date
                    $update_worker = $pdo->prepare("UPDATE " . DB_PREFIX . "sponsored_workers
                                                     SET status = 'active',
                                                         start_date = ?,
                                                         updated_at = NOW()
                                                     WHERE id = ?");
                    $update_worker->execute([$check_date, $worker_id]);

                    // Record activation in history
                    $activation_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "worker_activation
                                                       (worker_id, activated_by, start_date, previous_status, new_status,
                                                        rtw_check_id, activation_notes, confirmed_details)
                                                       VALUES (?, ?, ?, 'pending_rtw', 'active', ?, ?, ?)");

                    $activation_notes = "Auto-activated after successful RTW check";
                    $confirmed_details = json_encode(['worker_name' => $worker['full_name'], 'rtw_check_id' => $rtw_check_id]);

                    $activation_stmt->execute([
                        $worker_id,
                        $_SESSION['user_id'],
                        $check_date,
                        $rtw_check_id,
                        $activation_notes,
                        $confirmed_details
                    ]);

                    $_SESSION['success'] = 'RTW check added successfully and worker has been activated!';
                } else {
                    $_SESSION['success'] = 'RTW check added successfully';
                }
            } else {
                $_SESSION['success'] = 'RTW check added successfully';
            }

            // Redirect to worker view or list
            if (!empty($_POST['return_to_worker'])) {
                if (defined('MODULE_EMBEDDED')) {
                    header('Location: ../../m/workers/view/' . $worker_id);
                } else {
                    header('Location: ../workers/view.php?id=' . $worker_id);
                }
            } else {
                if (defined('MODULE_EMBEDDED')) {
                    header('Location: ../../m/rtw/list?success=1');
                } else {
                    header('Location: list.php?success=1');
                }
            }
            exit;

        } catch (PDOException $e) {
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
}

// Only include header if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern">
                        <i class="bi bi-plus-circle"></i> Add Right to Work Check
                    </h5>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <?php if (!empty($_GET['worker_id'])): ?>
                            <input type="hidden" name="return_to_worker" value="1">
                        <?php endif; ?>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Worker <span class="text-danger">*</span></label>
                                <select name="worker_id" class="form-select" required <?php echo !empty($worker_id) ? '' : ''; ?>>
                                    <option value="">Select worker...</option>
                                    <?php foreach ($workers as $worker): ?>
                                        <option value="<?php echo $worker['id']; ?>" <?php echo $worker_id == $worker['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($worker['full_name']); ?> (<?php echo $worker['employee_number']; ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check Date <span class="text-danger">*</span></label>
                                <input type="date" name="check_date" class="form-control" required value="<?php echo date('Y-m-d'); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Check Type <span class="text-danger">*</span></label>
                                <select name="check_type" class="form-select" required>
                                    <option value="">Select type...</option>
                                    <option value="initial">Initial (Pre-employment)</option>
                                    <option value="follow_up">Follow-up</option>
                                    <option value="annual">Annual</option>
                                </select>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label">Outcome <span class="text-danger">*</span></label>
                                <select name="outcome" class="form-select" required>
                                    <option value="">Select outcome...</option>
                                    <option value="pass">Pass</option>
                                    <option value="fail">Fail</option>
                                    <option value="refer">Refer</option>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Documents Seen <span class="text-danger">*</span></label>
                            <textarea name="documents_seen" class="form-control" rows="3" required
                                      placeholder="List all documents checked (e.g., Passport, BRP, Share Code...)"></textarea>
                            <small class="text-muted">Provide details of all documents examined</small>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Share Code (if applicable)</label>
                                <input type="text" name="share_code_used" class="form-control" maxlength="20"
                                       placeholder="e.g., AB12C3D4E">
                                <small class="text-muted">For online RTW checks</small>
                            </div>

                            <div class="col-md-6 mb-3">
                                <label class="form-label">Next Check Due</label>
                                <input type="date" name="next_check_due" class="form-control">
                                <small class="text-muted">Leave blank if not required</small>
                            </div>
                        </div>

                        <div class="mb-3">
                            <div class="form-check">
                                <input type="checkbox" class="form-check-input" name="online_check" value="1" id="onlineCheck">
                                <label class="form-check-label" for="onlineCheck">
                                    Online Check (using Home Office online service)
                                </label>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="4"
                                      placeholder="Additional notes about this check..."></textarea>
                        </div>

                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i>
                            <strong>Remember:</strong>
                            <ul class="mb-0 mt-2">
                                <li>Keep copies of all RTW documents for audit</li>
                                <li>Record check date when documents were verified</li>
                                <li>Set follow-up dates for List B documents</li>
                                <li>Ensure statutory excuse is established</li>
                            </ul>
                        </div>

                        <hr>

                        <div class="d-flex justify-content-between">
                            <?php
                            if (!empty($_GET['worker_id'])) {
                                $cancel_url = defined('MODULE_EMBEDDED') ? '../../m/workers/view/' . $_GET['worker_id'] : '../workers/view.php?id=' . $_GET['worker_id'];
                            } else {
                                $cancel_url = defined('MODULE_EMBEDDED') ? '../../m/rtw/list' : 'list.php';
                            }
                            ?>
                            <a href="<?php echo $cancel_url; ?>" class="btn btn-secondary">
                                <i class="bi bi-x-circle"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="bi bi-check-circle"></i> Add RTW Check
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Help Card -->
            <div class="modern-card mt-3">
                <div class="card-header-modern">
                    <h6 class="card-title-modern">
                        <i class="bi bi-question-circle"></i> RTW Check Types
                    </h6>
                </div>
                <div class="card-body-modern">
                    <p><strong>Initial Check:</strong> Before employment starts</p>
                    <p><strong>Follow-up Check:</strong> Before visa expiry for List B documents</p>
                    <p><strong>Annual Check:</strong> Yearly verification for indefinite leave to remain</p>

                    <hr>

                    <p class="mb-0"><strong>List A Documents:</strong> No follow-up required (permanent right to work)</p>
                    <p class="mb-0"><strong>List B Documents:</strong> Follow-up required before expiry</p>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Only include footer if not embedded through router
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
