<?php
// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
}

// Get database connection
$pdo = get_database_connection();

// Check permission level
if (($_SESSION['permission_level'] ?? 0) < 30) {
    die('Access denied. You do not have permission to view locations.');
}

$location_id = $_GET['id'] ?? 0;

// Fetch location
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "work_locations WHERE id = ?");
$stmt->execute([$location_id]);
$location = $stmt->fetch(PDO::FETCH_ASSOC);

// Get manager details if manager_id exists in the table
if (!empty($location['manager_id'])) {
    $manager_stmt = $pdo->prepare("SELECT full_name, employee_number FROM " . DB_PREFIX . "sponsored_workers WHERE id = ?");
    $manager_stmt->execute([$location['manager_id']]);
    $manager = $manager_stmt->fetch(PDO::FETCH_ASSOC);
    if ($manager) {
        $location['manager_name'] = $manager['full_name'];
        $location['manager_emp_no'] = $manager['employee_number'];
    }
}

if (!$location) {
    $_SESSION['error'] = 'Location not found';
    $redirect_url = defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php';
    header('Location: ' . $redirect_url);
    exit;
}

// Get shift patterns for this location
$shift_patterns_stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ? AND is_active = 1 ORDER BY is_default DESC, id ASC");
$shift_patterns_stmt->execute([$location_id]);
$shift_patterns = $shift_patterns_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get workers assigned to this location
$workers_sql = "SELECT w.id, w.full_name, w.employee_number, w.job_title, w.department, w.status
    FROM " . DB_PREFIX . "sponsored_workers w
    WHERE w.location_id = ? AND w.status = 'active'
    ORDER BY w.full_name";
$workers_stmt = $pdo->prepare($workers_sql);
$workers_stmt->execute([$location_id]);
$workers = $workers_stmt->fetchAll(PDO::FETCH_ASSOC);

$page_title = 'Location - ' . $location['name'];

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<div class="dashboard-container">
    <div class="modern-card">
        <div class="card-header-modern d-flex justify-content-between align-items-center">
            <h5 class="card-title-modern">
                <i class="bi bi-geo-alt"></i> <?php echo htmlspecialchars($location['name']); ?>
            </h5>
            <div>
                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/edit?id=' . $location['id'] : 'edit.php?id=' . $location['id']; ?>" class="btn btn-sm btn-primary">
                    <i class="bi bi-pencil"></i> Edit
                </a>
                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php'; ?>" class="btn btn-sm btn-secondary">
                    <i class="bi bi-arrow-left"></i> Back to List
                </a>
            </div>
        </div>
        <div class="card-body-modern">
            <div class="row">
                <!-- Location Details -->
                <div class="col-md-6">
                    <h6 class="border-bottom pb-2 mb-3"><i class="bi bi-info-circle"></i> Location Details</h6>
                    <table class="table table-borderless table-sm">
                        <tr>
                            <th width="40%">Name:</th>
                            <td><strong><?php echo htmlspecialchars($location['name']); ?></strong></td>
                        </tr>
                        <tr>
                            <th>Type:</th>
                            <td>
                                <?php if ($location['type']): ?>
                                    <span class="badge bg-secondary"><?php echo ucfirst($location['type']); ?></span>
                                <?php else: ?>
                                    <span class="text-muted">Not specified</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Status:</th>
                            <td>
                                <span class="badge bg-<?php echo $location['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                    <?php echo ucfirst($location['status']); ?>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>Capacity:</th>
                            <td><strong><?php echo $location['capacity'] ?? 0; ?></strong> workers</td>
                        </tr>
                        <tr>
                            <th>Current Occupancy:</th>
                            <td>
                                <strong><?php echo count($workers); ?></strong> workers
                                <?php if ($location['capacity'] > 0): ?>
                                    <span class="text-muted">(<?php echo round((count($workers) / $location['capacity']) * 100); ?>% full)</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Address & Manager -->
                <div class="col-md-6">
                    <h6 class="border-bottom pb-2 mb-3"><i class="bi bi-house"></i> Address & Management</h6>
                    <table class="table table-borderless table-sm">
                        <tr>
                            <th width="40%">Address:</th>
                            <td><?php echo $location['address'] ? htmlspecialchars($location['address']) : '<span class="text-muted">Not provided</span>'; ?></td>
                        </tr>
                        <tr>
                            <th>City:</th>
                            <td><?php echo $location['city'] ? htmlspecialchars($location['city']) : '<span class="text-muted">-</span>'; ?></td>
                        </tr>
                        <tr>
                            <th>Postcode:</th>
                            <td><?php echo $location['postcode'] ? htmlspecialchars($location['postcode']) : '<span class="text-muted">-</span>'; ?></td>
                        </tr>
                        <tr>
                            <th>Site Manager:</th>
                            <td>
                                <?php if ($location['manager_name']): ?>
                                    <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/workers/view?id=' . $location['manager_id'] : '../workers/view.php?id=' . $location['manager_id']; ?>">
                                        <?php echo htmlspecialchars($location['manager_name']); ?>
                                    </a>
                                    <br><small class="text-muted"><?php echo htmlspecialchars($location['manager_emp_no']); ?></small>
                                <?php else: ?>
                                    <span class="text-muted">Not assigned</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                </div>
            </div>

            <!-- Shift Patterns Section -->
            <hr>
            <div class="row">
                <div class="col-12">
                    <h6 class="border-bottom pb-2 mb-3">
                        <i class="bi bi-clock"></i> Shift Patterns
                        <span class="badge bg-secondary ms-2"><?php echo count($shift_patterns); ?> pattern<?php echo count($shift_patterns) !== 1 ? 's' : ''; ?></span>
                    </h6>

                    <?php if (empty($shift_patterns)): ?>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle"></i> No shift patterns defined for this location.
                        </div>
                    <?php else: ?>
                        <div class="row">
                            <?php foreach ($shift_patterns as $pattern): ?>
                            <div class="col-md-6 mb-3">
                                <div class="card h-100 <?php echo $pattern['is_default'] ? 'border-success' : ''; ?>">
                                    <div class="card-header <?php echo $pattern['is_default'] ? 'bg-success text-white' : 'bg-light'; ?>">
                                        <strong>
                                            <i class="bi bi-clock-fill"></i>
                                            <?php echo htmlspecialchars($pattern['shift_name']); ?>
                                        </strong>
                                        <?php if ($pattern['is_default']): ?>
                                            <span class="badge bg-white text-success float-end">Default</span>
                                        <?php endif; ?>
                                        <br>
                                        <small>
                                            <?php
                                            $type_badges = [
                                                'day' => 'bg-warning',
                                                'night' => 'bg-dark',
                                                'evening' => 'bg-info',
                                                'split' => 'bg-primary',
                                                'rotating' => 'bg-secondary',
                                                'flexible' => 'bg-success',
                                                'temporary' => 'bg-danger',
                                                'special' => 'bg-purple'
                                            ];
                                            $badge_class = $type_badges[$pattern['shift_type']] ?? 'bg-secondary';
                                            ?>
                                            <span class="badge <?php echo $badge_class; ?>"><?php echo ucfirst($pattern['shift_type']); ?> Shift</span>
                                        </small>
                                    </div>
                                    <div class="card-body">
                                        <table class="table table-borderless table-sm mb-0">
                                            <tr>
                                                <td width="40%"><strong>Time:</strong></td>
                                                <td>
                                                    <span class="badge bg-success">
                                                        <i class="bi bi-clock"></i> <?php echo date('H:i', strtotime($pattern['start_time'])); ?>
                                                    </span>
                                                    <i class="bi bi-arrow-right mx-1"></i>
                                                    <span class="badge bg-danger">
                                                        <i class="bi bi-clock-fill"></i> <?php echo date('H:i', strtotime($pattern['end_time'])); ?>
                                                    </span>
                                                </td>
                                            </tr>
                                            <?php if ($pattern['break_duration'] > 0): ?>
                                            <tr>
                                                <td><strong>Break:</strong></td>
                                                <td><?php echo $pattern['break_duration']; ?> minutes</td>
                                            </tr>
                                            <?php endif; ?>
                                            <?php if ($pattern['days_of_week']): ?>
                                            <tr>
                                                <td><strong>Days:</strong></td>
                                                <td>
                                                    <?php
                                                    $days = explode(',', $pattern['days_of_week']);
                                                    foreach ($days as $day) {
                                                        echo '<span class="badge bg-secondary me-1">' . htmlspecialchars(trim($day)) . '</span>';
                                                    }
                                                    ?>
                                                </td>
                                            </tr>
                                            <?php endif; ?>
                                            <?php if ($pattern['shift_type'] === 'temporary' && ($pattern['valid_from'] || $pattern['valid_until'])): ?>
                                            <tr>
                                                <td><strong>Valid Period:</strong></td>
                                                <td>
                                                    <?php if ($pattern['valid_from']): ?>
                                                        <?php echo date('d/m/Y', strtotime($pattern['valid_from'])); ?>
                                                    <?php endif; ?>
                                                    <?php if ($pattern['valid_from'] && $pattern['valid_until']): ?>
                                                        -
                                                    <?php endif; ?>
                                                    <?php if ($pattern['valid_until']): ?>
                                                        <?php echo date('d/m/Y', strtotime($pattern['valid_until'])); ?>
                                                    <?php endif; ?>
                                                </td>
                                            </tr>
                                            <?php endif; ?>
                                            <?php if ($pattern['notes']): ?>
                                            <tr>
                                                <td colspan="2">
                                                    <small class="text-muted">
                                                        <i class="bi bi-sticky"></i>
                                                        <?php echo nl2br(htmlspecialchars($pattern['notes'])); ?>
                                                    </small>
                                                </td>
                                            </tr>
                                            <?php endif; ?>
                                        </table>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <div class="alert alert-info small mt-2">
                            <i class="bi bi-info-circle"></i> These shift times are automatically populated when marking attendance for workers assigned to this location.
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <?php if ($location['notes']): ?>
            <hr>
            <div class="row">
                <div class="col-12">
                    <h6 class="border-bottom pb-2 mb-3"><i class="bi bi-sticky"></i> Notes</h6>
                    <div class="alert alert-secondary">
                        <?php echo nl2br(htmlspecialchars($location['notes'])); ?>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <hr>
            <div class="row">
                <div class="col-12">
                    <h6 class="border-bottom pb-2 mb-3">
                        <i class="bi bi-people"></i> Workers at this Location (<?php echo count($workers); ?>)
                    </h6>
                    <?php if (empty($workers)): ?>
                        <div class="alert alert-info">
                            <i class="bi bi-info-circle"></i> No workers currently assigned to this location.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-sm table-hover">
                                <thead>
                                    <tr>
                                        <th>Employee #</th>
                                        <th>Name</th>
                                        <th>Job Title</th>
                                        <th>Department</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($workers as $worker): ?>
                                    <tr>
                                        <td><strong><?php echo htmlspecialchars($worker['employee_number']); ?></strong></td>
                                        <td><?php echo htmlspecialchars($worker['full_name']); ?></td>
                                        <td><?php echo htmlspecialchars($worker['job_title'] ?? '-'); ?></td>
                                        <td><?php echo htmlspecialchars($worker['department'] ?? '-'); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $worker['status'] === 'active' ? 'success' : 'secondary'; ?>">
                                                <?php echo ucfirst($worker['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/workers/view?id=' . $worker['id'] : '../workers/view.php?id=' . $worker['id']; ?>" class="btn btn-sm btn-outline-primary">
                                                <i class="bi bi-eye"></i> View
                                            </a>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>

            <hr>
            <div class="row">
                <div class="col-12">
                    <h6 class="border-bottom pb-2 mb-3"><i class="bi bi-clock-history"></i> Timeline</h6>
                    <ul class="list-unstyled">
                        <li class="mb-2">
                            <i class="bi bi-circle-fill text-primary" style="font-size: 8px;"></i>
                            <strong>Created:</strong> <?php echo date('d/m/Y H:i', strtotime($location['created_at'])); ?>
                        </li>
                        <?php if ($location['updated_at'] && $location['updated_at'] !== $location['created_at']): ?>
                        <li class="mb-2">
                            <i class="bi bi-circle-fill text-info" style="font-size: 8px;"></i>
                            <strong>Last Updated:</strong> <?php echo date('d/m/Y H:i', strtotime($location['updated_at'])); ?>
                        </li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
