<?php
/**
 * Database Migration - Add Shift Timings to Locations
 * Run this once to add shift timing fields to work_locations table
 * and location_id to sponsored_workers table
 */

session_start();
require_once '../../config/database.php';
require_once '../../includes/database.php';
require_once '../../includes/session_manager.php';

require_authentication();

// Check admin permission
$current_user = get_logged_in_user();
if ($current_user['permission_level'] < 100) {
    die('Access denied. Admin permission required.');
}

$pdo = get_database_connection();
$results = [];
$errors = [];

// Check if migration already ran
try {
    $check = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "work_locations LIKE 'shift_start_time'");
    if ($check->rowCount() > 0) {
        $results[] = "✓ Migration already completed - shift timings fields already exist";
        $already_exists = true;
    } else {
        $already_exists = false;
    }
} catch (PDOException $e) {
    $errors[] = "Error checking migration status: " . $e->getMessage();
}

if (!$already_exists && empty($errors)) {
    try {
        // Add shift timing fields to work_locations
        $pdo->exec("ALTER TABLE " . DB_PREFIX . "work_locations
            ADD COLUMN shift_start_time TIME DEFAULT '09:00:00' AFTER capacity,
            ADD COLUMN shift_end_time TIME DEFAULT '17:00:00' AFTER shift_start_time,
            ADD COLUMN shift_name VARCHAR(100) DEFAULT 'Day Shift' AFTER shift_end_time");

        $results[] = "✓ Added shift timing fields to work_locations table";

    } catch (PDOException $e) {
        $errors[] = "Error adding shift fields: " . $e->getMessage();
    }
}

// Check if location_id field exists in sponsored_workers
try {
    $check2 = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "sponsored_workers LIKE 'location_id'");
    if ($check2->rowCount() > 0) {
        $results[] = "✓ location_id field already exists in sponsored_workers table";
    } else {
        // Add location_id to sponsored_workers
        $pdo->exec("ALTER TABLE " . DB_PREFIX . "sponsored_workers
            ADD COLUMN location_id INT NULL AFTER department,
            ADD INDEX idx_location_id (location_id)");

        $results[] = "✓ Added location_id field to sponsored_workers table";
    }
} catch (PDOException $e) {
    $errors[] = "Error adding location_id field: " . $e->getMessage();
}

header('Content-Type: text/plain');
echo "=== SHIFT TIMINGS MIGRATION ===\n\n";

if (!empty($results)) {
    echo "RESULTS:\n";
    foreach ($results as $result) {
        echo "$result\n";
    }
    echo "\n";
}

if (!empty($errors)) {
    echo "ERRORS:\n";
    foreach ($errors as $error) {
        echo "✗ $error\n";
    }
    echo "\n";
}

if (empty($errors)) {
    echo "✓ Migration completed successfully!\n\n";
    echo "Next steps:\n";
    echo "1. Go to Locations module and edit each location to set shift times\n";
    echo "2. Go to Workers module and assign locations to workers\n";
    echo "3. Attendance will auto-populate check-in/out times based on location shifts\n";
} else {
    echo "✗ Migration failed. Please fix errors and try again.\n";
}
?>
