<?php
/**
 * Work Locations List
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
}

$page_title = 'Work Locations';
$current_page = 'locations';

// Get database connection
$pdo = get_database_connection();

// Get current user for permission checks
$current_user = get_logged_in_user();
$is_super_admin = ($current_user['permission_level'] ?? 0) >= 100;

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/header.php';
}

// Fetch locations with site manager information
$locations_query = "SELECT l.*, w.full_name as manager_name, w.employee_number as manager_emp_no
    FROM " . DB_PREFIX . "work_locations l
    LEFT JOIN " . DB_PREFIX . "sponsored_workers w ON l.manager_id = w.id
    ORDER BY l.name";
$locations = $pdo->query($locations_query)->fetchAll();
$stats_query = $pdo->query("SELECT
    COUNT(*) as total,
    SUM(capacity) as total_capacity,
    SUM(CASE WHEN status = 'active' THEN 1 ELSE 0 END) as active,
    SUM(CASE WHEN type = 'office' THEN 1 ELSE 0 END) as offices
    FROM " . DB_PREFIX . "work_locations")->fetch();
?>

<div class="dashboard-container">
    <div class="welcome-section">
        <div class="welcome-content">
            <h1 class="welcome-title"><i class="bi bi-geo-alt"></i> Work Locations</h1>
            <p class="welcome-subtitle">Manage work locations and capacities</p>
        </div>
        <div class="welcome-actions">
            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/add' : 'add.php'; ?>" class="btn btn-primary"><i class="bi bi-plus-circle"></i> Add Location</a>
        </div>
    </div>

    <!-- Quick Actions -->
    <div class="quick-actions-grid mb-4">
        <!-- Total Locations -->
        <div class="quick-action-btn gradient-pink">
            <div class="quick-action-icon">
                <span class="material-symbols-outlined" style="font-size: 1.75rem;">location_on</span>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats_query['total']); ?> Total Locations</span>
        </div>

        <!-- Active Locations -->
        <div class="quick-action-btn gradient-green">
            <div class="quick-action-icon">
                <i class="ri-map-pin-user-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats_query['active']); ?> Active</span>
        </div>

        <!-- Total Capacity -->
        <div class="quick-action-btn gradient-blue">
            <div class="quick-action-icon">
                <i class="fas fa-users"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats_query['total_capacity']); ?> Total Capacity</span>
        </div>

        <!-- Offices -->
        <div class="quick-action-btn gradient-orange">
            <div class="quick-action-icon">
                <i class="ri-building-2-fill"></i>
            </div>
            <span class="quick-action-label"><?php echo number_format($stats_query['offices']); ?> Offices</span>
        </div>
    </div>

    <div class="modern-card">
        <div class="card-body-modern">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Type</th>
                            <th>Address</th>
                            <th>City</th>
                            <th>Site Manager</th>
                            <th>Capacity</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($locations)): ?>
                        <tr>
                            <td colspan="8" class="text-center py-5">
                                <i class="bi bi-inbox" style="font-size: 3rem; color: #ccc;"></i>
                                <p class="text-muted mt-2">No locations found</p>
                                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/add' : 'add.php'; ?>" class="btn btn-primary">Add First Location</a>
                            </td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($locations as $loc): ?>
                        <tr>
                            <td>
                                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/view?id=' . $loc['id'] : 'view.php?id=' . $loc['id']; ?>" class="text-decoration-none">
                                    <strong><?php echo htmlspecialchars($loc['name']); ?></strong>
                                </a>
                            </td>
                            <td><?php echo htmlspecialchars($loc['type'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($loc['address'] ?? '-'); ?></td>
                            <td><?php echo htmlspecialchars($loc['city'] ?? '-'); ?></td>
                            <td>
                                <?php if ($loc['manager_name']): ?>
                                    <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/workers/view?id=' . $loc['manager_id'] : '../workers/view.php?id=' . $loc['manager_id']; ?>" class="text-decoration-none">
                                        <?php echo htmlspecialchars($loc['manager_name']); ?>
                                    </a>
                                    <?php if ($loc['manager_emp_no']): ?>
                                        <br><small class="text-muted"><?php echo htmlspecialchars($loc['manager_emp_no']); ?></small>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <span class="text-muted">-</span>
                                <?php endif; ?>
                            </td>
                            <td><?php echo $loc['capacity'] ?? 0; ?></td>
                            <td><span class="badge bg-<?php echo $loc['status'] === 'active' ? 'success' : 'secondary'; ?>"><?php echo ucfirst($loc['status']); ?></span></td>
                            <td>
                                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/view?id=' . $loc['id'] : 'view.php?id=' . $loc['id']; ?>" class="btn btn-sm btn-outline-info" title="View Location">
                                    <i class="bi bi-eye"></i>
                                </a>
                                <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/edit?id=' . $loc['id'] : 'edit.php?id=' . $loc['id']; ?>" class="btn btn-sm btn-outline-primary" title="Edit Location">
                                    <i class="bi bi-pencil"></i>
                                </a>
                                <?php if ($loc['status'] === 'closed'): ?>
                                    <!-- Restore button for ALL users with delete permission -->
                                    <form method="POST" action="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/delete' : 'delete.php'; ?>" style="display:inline;" onsubmit="return confirm('Restore this location to active status?');">
                                        <input type="hidden" name="id" value="<?php echo $loc['id']; ?>">
                                        <input type="hidden" name="delete_type" value="restore">
                                        <button type="submit" class="btn btn-sm btn-outline-success" title="Restore Location">
                                            <i class="bi bi-arrow-counterclockwise"></i> Restore
                                        </button>
                                    </form>
                                    <?php if ($is_super_admin): ?>
                                        <!-- Permanent Delete button (Super Admin only) -->
                                        <button type="button" class="btn btn-sm btn-danger" onclick="showPermanentDeleteModal(<?php echo $loc['id']; ?>, '<?php echo addslashes($loc['name']); ?>')" title="Permanent Delete - Remove from Database">
                                            <i class="bi bi-trash-fill"></i> Delete Forever
                                        </button>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <!-- Soft Delete (Close) for active/inactive locations -->
                                    <form method="POST" action="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/delete' : 'delete.php'; ?>" style="display:inline;" onsubmit="return confirm('Close this location?\n\nIt will be marked as closed but can be restored later.');">
                                        <input type="hidden" name="id" value="<?php echo $loc['id']; ?>">
                                        <input type="hidden" name="delete_type" value="soft">
                                        <button type="submit" class="btn btn-sm btn-outline-warning" title="Close Location (Soft Delete)">
                                            <i class="bi bi-x-circle"></i> Close
                                        </button>
                                    </form>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Permanent Delete Modal (Super Admin Only) -->
<div class="modal fade" id="permanentDeleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title"><i class="bi bi-exclamation-triangle"></i> Permanent Delete</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/delete' : 'delete.php'; ?>">
                <div class="modal-body">
                    <div class="alert alert-danger">
                        <strong>⚠️ WARNING:</strong> This action is IRREVERSIBLE!
                    </div>
                    <p>You are about to <strong>PERMANENTLY DELETE</strong> the location:</p>
                    <p class="text-center"><strong id="deleteLocationName" class="fs-5 text-danger"></strong></p>
                    <p class="mb-3">This will <strong>completely remove ALL records from the database</strong>, including:</p>
                    <ul>
                        <li>Location information and settings</li>
                        <li>Shift timings</li>
                        <li>Worker assignments to this location</li>
                        <li>All historical data</li>
                    </ul>
                    <p class="text-danger fw-bold">⚠️ This CANNOT be undone. The data will be gone forever!</p>

                    <input type="hidden" name="id" id="deleteLocationId">
                    <input type="hidden" name="delete_type" value="permanent">

                    <div class="mb-3">
                        <label class="form-label">Enter your PIN to confirm:</label>
                        <input type="password" name="pin" class="form-control" required placeholder="Your PIN" autocomplete="off">
                        <small class="text-muted">Super Admin PIN required for permanent deletion</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">
                        <i class="bi bi-trash-fill"></i> Permanent Delete
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showPermanentDeleteModal(id, name) {
    document.getElementById('deleteLocationId').value = id;
    document.getElementById('deleteLocationName').textContent = name;
    var modal = new bootstrap.Modal(document.getElementById('permanentDeleteModal'));
    modal.show();
}
</script>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    require_once '../../includes/footer.php';
}
?>
