<?php
// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication();
}

// Get database connection
$pdo = get_database_connection();

// Check permission level
if (($_SESSION['permission_level'] ?? 0) < 70) {
    die('Access denied. You do not have permission to edit locations.');
}

$location_id = $_GET['id'] ?? 0;
$errors = [];

// Fetch location
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "work_locations WHERE id = ?");
$stmt->execute([$location_id]);
$location = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$location) {
    $_SESSION['error'] = 'Location not found';
    $redirect_url = defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php';
    header('Location: ' . $redirect_url);
    exit;
}

// Fetch existing shift patterns
$stmt = $pdo->prepare("SELECT * FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ? ORDER BY is_default DESC, id ASC");
$stmt->execute([$location_id]);
$existing_shift_patterns = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Check if manager_id column exists
$columns_check = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "work_locations LIKE 'manager_id'");
$has_manager_id = $columns_check->rowCount() > 0;

// Get managers (workers) for assignment if manager_id field exists
$managers = [];
if ($has_manager_id) {
    $managers = $pdo->query("SELECT id, full_name, employee_number, job_title FROM " . DB_PREFIX . "sponsored_workers WHERE status = 'active' ORDER BY full_name")->fetchAll();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $type = $_POST['type'] ?? '';
    $address = trim($_POST['address'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $postcode = trim($_POST['postcode'] ?? '');
    $capacity = $_POST['capacity'] ?? 0;
    $manager_id = $_POST['manager_id'] ?: null;
    $status = $_POST['status'] ?? 'active';
    $notes = trim($_POST['notes'] ?? '');

    // Shift patterns from POST (multiple shifts)
    $shift_patterns = isset($_POST['shift_patterns']) ? json_decode($_POST['shift_patterns'], true) : [];

    if (empty($name)) $errors[] = "Name is required";
    if (empty($shift_patterns)) $errors[] = "At least one shift pattern is required";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Check if manager_id column exists
            $columns_check = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "work_locations LIKE 'manager_id'");
            $has_manager_id = $columns_check->rowCount() > 0;

            // Use first shift pattern for backward compatibility in main table
            $first_shift = $shift_patterns[0] ?? null;

            if ($has_manager_id) {
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "work_locations
                    SET name = ?, type = ?, address = ?, city = ?, postcode = ?,
                        capacity = ?, shift_start_time = ?, shift_end_time = ?, shift_name = ?,
                        manager_id = ?, status = ?, notes = ?, updated_at = NOW()
                    WHERE id = ?");

                $stmt->execute([
                    $name, $type, $address, $city, $postcode, $capacity,
                    $first_shift['start_time'] ?? '09:00',
                    $first_shift['end_time'] ?? '17:00',
                    $first_shift['shift_name'] ?? 'Day Shift',
                    $manager_id, $status, $notes, $location_id
                ]);
            } else {
                $stmt = $pdo->prepare("UPDATE " . DB_PREFIX . "work_locations
                    SET name = ?, type = ?, address = ?, city = ?, postcode = ?,
                        capacity = ?, shift_start_time = ?, shift_end_time = ?, shift_name = ?,
                        status = ?, notes = ?, updated_at = NOW()
                    WHERE id = ?");

                $stmt->execute([
                    $name, $type, $address, $city, $postcode, $capacity,
                    $first_shift['start_time'] ?? '09:00',
                    $first_shift['end_time'] ?? '17:00',
                    $first_shift['shift_name'] ?? 'Day Shift',
                    $status, $notes, $location_id
                ]);
            }

            // Delete all existing shift patterns for this location
            $pdo->prepare("DELETE FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ?")
                ->execute([$location_id]);

            // Insert updated shift patterns
            $shift_stmt = $pdo->prepare("INSERT INTO " . DB_PREFIX . "location_shift_patterns
                (location_id, shift_name, shift_type, start_time, end_time, break_duration, days_of_week, is_default, is_active, valid_from, valid_until, notes)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");

            foreach ($shift_patterns as $index => $shift) {
                $shift_stmt->execute([
                    $location_id,
                    $shift['shift_name'] ?? 'Shift ' . ($index + 1),
                    $shift['shift_type'] ?? 'day',
                    $shift['start_time'] ?? '09:00',
                    $shift['end_time'] ?? '17:00',
                    $shift['break_duration'] ?? 0,
                    $shift['days_of_week'] ?? 'Mon,Tue,Wed,Thu,Fri',
                    $index === 0 ? 1 : 0, // First shift is default
                    1, // is_active
                    $shift['valid_from'] ?? null,
                    $shift['valid_until'] ?? null,
                    $shift['notes'] ?? null
                ]);
            }

            $pdo->commit();
            $_SESSION['success'] = 'Location updated successfully with ' . count($shift_patterns) . ' shift pattern(s)';

            // Handle redirect based on environment
            if (defined('MODULE_EMBEDDED')) {
                // Router mode - use JavaScript redirect
                echo '<script>window.location.href = "../../m/locations/list";</script>';
                echo '<noscript><meta http-equiv="refresh" content="0;url=../../m/locations/list"></noscript>';
            } else {
                // Direct access - use header redirect
                header('Location: list.php');
            }
            exit;
        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors[] = "Database error: " . $e->getMessage();
        }
    }
} else {
    // Pre-fill form
    $_POST = $location;
}

$page_title = 'Edit Location';

// Only include header if not embedded
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/header.php';
}
?>

<div class="dashboard-container">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="modern-card">
                <div class="card-header-modern">
                    <h5 class="card-title-modern"><i class="bi bi-pencil"></i> Edit Work Location</h5>
                </div>
                <div class="card-body-modern">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0"><?php foreach ($errors as $error): ?><li><?php echo htmlspecialchars($error); ?></li><?php endforeach; ?></ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <div class="col-md-8 mb-3">
                                <label class="form-label">Location Name <span class="text-danger">*</span></label>
                                <input type="text" name="name" class="form-control" required value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>">
                            </div>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Type</label>
                                <select name="type" class="form-select">
                                    <option value="">Select...</option>
                                    <option value="office" <?php echo ($_POST['type'] ?? '') === 'office' ? 'selected' : ''; ?>>Office</option>
                                    <option value="warehouse" <?php echo ($_POST['type'] ?? '') === 'warehouse' ? 'selected' : ''; ?>>Warehouse</option>
                                    <option value="site" <?php echo ($_POST['type'] ?? '') === 'site' ? 'selected' : ''; ?>>Site</option>
                                    <option value="remote" <?php echo ($_POST['type'] ?? '') === 'remote' ? 'selected' : ''; ?>>Remote</option>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Address</label>
                            <input type="text" name="address" class="form-control" value="<?php echo htmlspecialchars($_POST['address'] ?? ''); ?>">
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">City</label>
                                <input type="text" name="city" class="form-control" value="<?php echo htmlspecialchars($_POST['city'] ?? ''); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Postcode</label>
                                <input type="text" name="postcode" class="form-control" value="<?php echo htmlspecialchars($_POST['postcode'] ?? ''); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-<?php echo $has_manager_id ? '4' : '6'; ?> mb-3">
                                <label class="form-label">Capacity</label>
                                <input type="number" name="capacity" class="form-control" value="<?php echo htmlspecialchars($_POST['capacity'] ?? '0'); ?>">
                            </div>
                            <?php if ($has_manager_id): ?>
                            <div class="col-md-4 mb-3">
                                <label class="form-label">Site Manager</label>
                                <select name="manager_id" class="form-select">
                                    <option value="">None</option>
                                    <?php foreach ($managers as $manager): ?>
                                        <option value="<?php echo $manager['id']; ?>" <?php echo ($_POST['manager_id'] ?? '') == $manager['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($manager['full_name']); ?>
                                            <?php if ($manager['job_title']): ?>
                                                - <?php echo htmlspecialchars($manager['job_title']); ?>
                                            <?php endif; ?>
                                            <?php if ($manager['employee_number']): ?>
                                                (<?php echo htmlspecialchars($manager['employee_number']); ?>)
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Optional: Assign a site manager</small>
                            </div>
                            <?php endif; ?>
                            <div class="col-md-<?php echo $has_manager_id ? '4' : '6'; ?> mb-3">
                                <label class="form-label">Status</label>
                                <select name="status" class="form-select">
                                    <option value="active" <?php echo ($_POST['status'] ?? '') === 'active' ? 'selected' : ''; ?>>Active</option>
                                    <option value="inactive" <?php echo ($_POST['status'] ?? '') === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                                    <option value="closed" <?php echo ($_POST['status'] ?? '') === 'closed' ? 'selected' : ''; ?>>Closed</option>
                                </select>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Notes</label>
                            <textarea name="notes" class="form-control" rows="3"><?php echo htmlspecialchars($_POST['notes'] ?? ''); ?></textarea>
                        </div>

                        <hr>

                        <!-- Shift Patterns Section -->
                        <h6 class="mb-3">
                            <i class="bi bi-clock"></i> Shift Patterns
                            <button type="button" class="btn btn-sm btn-primary float-end" onclick="addShiftPattern()">
                                <i class="bi bi-plus-circle"></i> Add Shift Pattern
                            </button>
                        </h6>

                        <div id="shift-patterns-container" class="mb-3">
                            <!-- Shift patterns will be added here dynamically -->
                        </div>

                        <div class="alert alert-info small">
                            <i class="bi bi-info-circle"></i> Define shift patterns for this location. Workers will automatically use these times when marking attendance.
                        </div>

                        <input type="hidden" name="shift_patterns" id="shift_patterns_input" value="">

                        <hr>
                        <div class="d-flex justify-content-between">
                            <a href="<?php echo defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php'; ?>" class="btn btn-secondary"><i class="bi bi-x-circle"></i> Cancel</a>
                            <button type="submit" class="btn btn-primary"><i class="bi bi-check-circle"></i> Update Location</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
let shiftPatterns = [];
let shiftCounter = 0;

// Load existing shift patterns on page load
document.addEventListener('DOMContentLoaded', function() {
    const existingPatterns = <?php echo json_encode($existing_shift_patterns); ?>;

    if (existingPatterns && existingPatterns.length > 0) {
        // Load existing patterns
        existingPatterns.forEach(pattern => {
            addShiftPattern({
                shift_name: pattern.shift_name,
                shift_type: pattern.shift_type,
                start_time: pattern.start_time,
                end_time: pattern.end_time,
                break_duration: pattern.break_duration || 0,
                days_of_week: pattern.days_of_week || 'Mon,Tue,Wed,Thu,Fri',
                valid_from: pattern.valid_from || '',
                valid_until: pattern.valid_until || '',
                notes: pattern.notes || ''
            });
        });
    } else {
        // No existing patterns - load from main location table (backward compatibility)
        addShiftPattern({
            shift_name: '<?php echo addslashes($location['shift_name'] ?? 'Day Shift'); ?>',
            shift_type: 'day',
            start_time: '<?php echo $location['shift_start_time'] ?? '09:00'; ?>',
            end_time: '<?php echo $location['shift_end_time'] ?? '17:00'; ?>',
            break_duration: 60,
            days_of_week: 'Mon,Tue,Wed,Thu,Fri'
        });
    }
});

function addShiftPattern(data = {}) {
    const id = shiftCounter++;
    const pattern = {
        id: id,
        shift_name: data.shift_name || '',
        shift_type: data.shift_type || 'day',
        start_time: data.start_time || '09:00',
        end_time: data.end_time || '17:00',
        break_duration: data.break_duration || 0,
        days_of_week: data.days_of_week || 'Mon,Tue,Wed,Thu,Fri',
        valid_from: data.valid_from || '',
        valid_until: data.valid_until || '',
        notes: data.notes || ''
    };

    shiftPatterns.push(pattern);
    renderShiftPatterns();
}

function removeShiftPattern(id) {
    if (shiftPatterns.length === 1) {
        alert('At least one shift pattern is required');
        return;
    }
    shiftPatterns = shiftPatterns.filter(p => p.id !== id);
    renderShiftPatterns();
}

function updateShiftPattern(id, field, value) {
    const pattern = shiftPatterns.find(p => p.id === id);
    if (pattern) {
        pattern[field] = value;
        updateHiddenInput();
    }
}

function renderShiftPatterns() {
    const container = document.getElementById('shift-patterns-container');
    container.innerHTML = '';

    shiftPatterns.forEach((pattern, index) => {
        const card = document.createElement('div');
        card.className = 'card mb-3';
        card.innerHTML = `
            <div class="card-body">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h6 class="mb-0">
                        <i class="bi bi-clock-fill text-primary"></i>
                        Shift Pattern ${index + 1}
                        ${index === 0 ? '<span class="badge bg-success ms-2">Default</span>' : ''}
                    </h6>
                    ${shiftPatterns.length > 1 ? `
                        <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeShiftPattern(${pattern.id})">
                            <i class="bi bi-trash"></i> Remove
                        </button>
                    ` : ''}
                </div>

                <div class="row">
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Shift Name <span class="text-danger">*</span></label>
                        <input type="text" class="form-control" value="${pattern.shift_name}"
                               onchange="updateShiftPattern(${pattern.id}, 'shift_name', this.value)"
                               placeholder="e.g., Day Shift, Night Shift" required>
                    </div>
                    <div class="col-md-6 mb-3">
                        <label class="form-label">Shift Type</label>
                        <select class="form-select" onchange="updateShiftPattern(${pattern.id}, 'shift_type', this.value); renderShiftPatterns();">
                            <option value="day" ${pattern.shift_type === 'day' ? 'selected' : ''}>Day Shift</option>
                            <option value="night" ${pattern.shift_type === 'night' ? 'selected' : ''}>Night Shift</option>
                            <option value="evening" ${pattern.shift_type === 'evening' ? 'selected' : ''}>Evening Shift</option>
                            <option value="split" ${pattern.shift_type === 'split' ? 'selected' : ''}>Split Shift</option>
                            <option value="rotating" ${pattern.shift_type === 'rotating' ? 'selected' : ''}>Rotating Shift</option>
                            <option value="flexible" ${pattern.shift_type === 'flexible' ? 'selected' : ''}>Flexible Hours</option>
                            <option value="temporary" ${pattern.shift_type === 'temporary' ? 'selected' : ''}>Temporary Shift</option>
                            <option value="special" ${pattern.shift_type === 'special' ? 'selected' : ''}>Special Shift</option>
                        </select>
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Start Time <span class="text-danger">*</span></label>
                        <input type="time" class="form-control" value="${pattern.start_time}"
                               onchange="updateShiftPattern(${pattern.id}, 'start_time', this.value)" required>
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="form-label">End Time <span class="text-danger">*</span></label>
                        <input type="time" class="form-control" value="${pattern.end_time}"
                               onchange="updateShiftPattern(${pattern.id}, 'end_time', this.value)" required>
                    </div>
                    <div class="col-md-4 mb-3">
                        <label class="form-label">Break Duration (minutes)</label>
                        <input type="number" class="form-control" value="${pattern.break_duration}" min="0"
                               onchange="updateShiftPattern(${pattern.id}, 'break_duration', this.value)">
                    </div>
                </div>

                <div class="row">
                    <div class="col-md-12 mb-3">
                        <label class="form-label">Days of Week</label>
                        <div class="btn-group w-100" role="group">
                            ${['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'].map(day => `
                                <input type="checkbox" class="btn-check" id="day_${pattern.id}_${day}"
                                       ${pattern.days_of_week.includes(day) ? 'checked' : ''}
                                       onchange="updateDaysOfWeek(${pattern.id})">
                                <label class="btn btn-outline-primary btn-sm" for="day_${pattern.id}_${day}">${day}</label>
                            `).join('')}
                        </div>
                    </div>
                </div>

                ${pattern.shift_type === 'temporary' ? `
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valid From</label>
                            <input type="date" class="form-control" value="${pattern.valid_from}"
                                   onchange="updateShiftPattern(${pattern.id}, 'valid_from', this.value)">
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Valid Until</label>
                            <input type="date" class="form-control" value="${pattern.valid_until}"
                                   onchange="updateShiftPattern(${pattern.id}, 'valid_until', this.value)">
                        </div>
                    </div>
                ` : ''}

                <div class="mb-3">
                    <label class="form-label">Notes</label>
                    <textarea class="form-control" rows="2"
                              onchange="updateShiftPattern(${pattern.id}, 'notes', this.value)"
                              placeholder="Optional notes about this shift pattern">${pattern.notes}</textarea>
                </div>
            </div>
        `;
        container.appendChild(card);
    });

    updateHiddenInput();
}

function updateDaysOfWeek(patternId) {
    const days = ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'];
    const selected = days.filter(day =>
        document.getElementById(`day_${patternId}_${day}`).checked
    );
    updateShiftPattern(patternId, 'days_of_week', selected.join(','));
}

function updateHiddenInput() {
    const data = shiftPatterns.map(({id, ...rest}) => rest); // Remove id before sending
    document.getElementById('shift_patterns_input').value = JSON.stringify(data);
}

// Validate form before submission
document.querySelector('form').addEventListener('submit', function(e) {
    if (shiftPatterns.length === 0) {
        e.preventDefault();
        alert('Please add at least one shift pattern');
        return false;
    }

    // Ensure all required fields are filled
    for (let pattern of shiftPatterns) {
        if (!pattern.shift_name || !pattern.start_time || !pattern.end_time) {
            e.preventDefault();
            alert('Please fill in all required fields for all shift patterns');
            return false;
        }
    }

    updateHiddenInput();
});
</script>

<?php
// Only include footer if not embedded
if (!defined('MODULE_EMBEDDED')) {
    include '../../includes/footer.php';
}
?>
