<?php
/**
 * Simplified Location Delete for Debugging
 */

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';
    require_authentication('../../login.php');
}

// Get current user
$current_user = get_logged_in_user();

// Debug output
$debug_mode = true;

if ($debug_mode) {
    echo "<h3>Delete Debug Info</h3>";
    echo "Request Method: " . $_SERVER['REQUEST_METHOD'] . "<br>";
    echo "Permission Level: " . ($current_user['permission_level'] ?? 'not set') . "<br>";
    echo "Location ID: " . ($_POST['id'] ?? 'not set') . "<br>";
    echo "<br>";
}

// Check permission
if (!isset($current_user['permission_level']) || $current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You need permission level 70+ to delete locations. Your level: ' . ($current_user['permission_level'] ?? 'unknown');
    echo "ERROR: " . $_SESSION['error'] . "<br>";
    echo '<a href="list.php">Back to List</a>';
    exit;
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = 'Invalid request method';
    echo "ERROR: Must use POST<br>";
    echo '<a href="list.php">Back to List</a>';
    exit;
}

$location_id = (int)($_POST['id'] ?? 0);

if ($location_id <= 0) {
    $_SESSION['error'] = 'Invalid location ID: ' . $location_id;
    echo "ERROR: " . $_SESSION['error'] . "<br>";
    echo '<a href="list.php">Back to List</a>';
    exit;
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if location exists
    $stmt = $pdo->prepare("SELECT id, name, status FROM " . DB_PREFIX . "work_locations WHERE id = ?");
    $stmt->execute([$location_id]);
    $location = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$location) {
        $_SESSION['error'] = 'Location not found.';
        echo "ERROR: " . $_SESSION['error'] . "<br>";
        echo '<a href="list.php">Back to List</a>';
        exit;
    }

    if ($debug_mode) {
        echo "Location found:<br>";
        echo "  - ID: " . $location['id'] . "<br>";
        echo "  - Name: " . htmlspecialchars($location['name']) . "<br>";
        echo "  - Status: " . $location['status'] . "<br>";
        echo "<br>";
    }

    // Perform soft delete (set status to 'closed')
    $stmt = $pdo->prepare("
        UPDATE " . DB_PREFIX . "work_locations
        SET status = 'closed',
            updated_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$location_id]);

    $affected = $stmt->rowCount();

    if ($debug_mode) {
        echo "Update executed. Rows affected: " . $affected . "<br>";
    }

    $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been closed.';
    echo "SUCCESS: " . $_SESSION['success'] . "<br>";
    echo '<br><a href="list.php">Back to List</a>';

} catch (Exception $e) {
    $_SESSION['error'] = 'Error: ' . $e->getMessage();
    echo "ERROR: " . $_SESSION['error'] . "<br>";
    echo '<a href="list.php">Back to List</a>';
}
?>
