<?php
/**
 * Locations Module - Delete Work Location
 * Role-based deletion:
 * - Permission 70-89: Can soft delete only (sets status to 'closed')
 * - Permission 90-99: Can soft delete + restore
 * - Permission 100+: Can soft delete, restore, or PERMANENT delete (requires PIN)
 */

// Helper function to handle redirects in both embedded and non-embedded modes
function safe_redirect($url) {
    if (defined('MODULE_EMBEDDED')) {
        // Headers already sent by router, use JavaScript redirect
        echo '<script>window.location.href = "' . htmlspecialchars($url) . '";</script>';
        echo '<noscript><meta http-equiv="refresh" content="0;url=' . htmlspecialchars($url) . '"></noscript>';
    } else {
        // Direct access, use normal header redirect
        header('Location: ' . $url);
    }
    exit;
}

// Check if embedded through router
if (!defined('MODULE_EMBEDDED')) {
    session_start();
    require_once '../../config/database.php';
    require_once '../../includes/database.php';
    require_once '../../includes/session_manager.php';

    require_authentication('../../login.php');

    $current_user = get_logged_in_user();
    if (!$current_user) {
        safe_redirect('../../login.php');
    }
} else {
    // When loaded through router, we need to get $current_user from the router's scope
    // The router already authenticated the user and set $current_user
    if (!isset($current_user)) {
        $current_user = get_logged_in_user();
    }
}

// Check minimum permission - delete requires level 70
if (!isset($current_user['permission_level']) || $current_user['permission_level'] < 70) {
    $_SESSION['error'] = 'Access denied. You do not have permission to delete locations. (Permission level: ' . ($current_user['permission_level'] ?? 'not set') . ')';
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
}

$location_id = (int)($_POST['id'] ?? 0);
$delete_type = $_POST['delete_type'] ?? 'soft'; // 'soft' or 'permanent'
$pin = $_POST['pin'] ?? '';

// Debug: Log what we received
error_log("Delete request - ID: $location_id, Type: $delete_type, Permission: " . $current_user['permission_level']);

if ($location_id <= 0) {
    $_SESSION['error'] = 'Invalid location ID: ' . $location_id;
    safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
}

// Get database connection
$pdo = get_database_connection();

try {
    // Check if location exists and get current status
    $stmt = $pdo->prepare("SELECT id, name, status FROM " . DB_PREFIX . "work_locations WHERE id = ?");
    $stmt->execute([$location_id]);
    $location = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$location) {
        $_SESSION['error'] = 'Location not found.';
        safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
    }

    // Determine what type of delete to perform based on role and request
    $permission_level = $current_user['permission_level'];
    $current_status = $location['status'];

    // CASE 1: Non-admin users (70-89) - Can only soft delete active locations
    if ($permission_level >= 70 && $permission_level < 90) {
        if ($current_status === 'closed') {
            $_SESSION['error'] = 'This location is already closed. Only admins can permanently remove closed locations.';
            safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
        }

        // Soft delete (set status to 'closed')
        $stmt = $pdo->prepare("
            UPDATE " . DB_PREFIX . "work_locations
            SET status = 'closed',
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$location_id]);
        error_log("Successfully soft-deleted location ID: $location_id");
        $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been closed.';
    }

    // CASE 2: Senior users (90-99) - Can soft delete and restore
    elseif ($permission_level >= 90 && $permission_level < 100) {
        if ($current_status === 'closed') {
            // Restore closed location (set to 'active' status)
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "work_locations
                SET status = 'active',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$location_id]);
            $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been restored.';
        } else {
            // Soft delete (set status to 'closed')
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "work_locations
                SET status = 'closed',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$location_id]);
            $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been closed.';
        }
    }

    // CASE 3: Admins (100+) - Can do everything including permanent delete
    elseif ($permission_level >= 100) {
        // If explicitly requesting restore
        if ($delete_type === 'restore') {
            $stmt = $pdo->prepare("
                UPDATE " . DB_PREFIX . "work_locations
                SET status = 'active',
                    updated_at = NOW()
                WHERE id = ?
            ");
            $stmt->execute([$location_id]);
            $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been restored.';
        }
        // If requesting permanent delete
        elseif ($delete_type === 'permanent') {
            // REQUIRE PIN for permanent delete
            if (empty($pin)) {
                $_SESSION['error'] = 'PIN is required for permanent deletion.';
                safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
            }

            // Verify PIN (check against user's PIN)
            if (!isset($current_user['pin']) || $current_user['pin'] !== $pin) {
                $_SESSION['error'] = 'Invalid PIN. Permanent deletion cancelled.';
                safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
            }

            // Log the permanent deletion for audit trail
            $audit_stmt = $pdo->prepare("
                INSERT INTO " . DB_PREFIX . "audit_log
                (user_id, action, table_name, record_id, details, ip_address, created_at)
                VALUES (?, 'PERMANENT_DELETE', 'work_locations', ?, ?, ?, NOW())
            ");
            $details = json_encode([
                'location_name' => $location['name'],
                'reason' => 'Admin permanent deletion with PIN verification'
            ]);
            $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
            $audit_stmt->execute([$current_user['id'], $location_id, $details, $ip]);

            // PERMANENT DELETE - Remove from database
            $stmt = $pdo->prepare("DELETE FROM " . DB_PREFIX . "work_locations WHERE id = ?");
            $stmt->execute([$location_id]);

            $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been PERMANENTLY DELETED from the database. This action cannot be undone.';
        } else {
            // Soft delete or restore
            if ($current_status === 'closed') {
                // Restore
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "work_locations
                    SET status = 'active',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$location_id]);
                $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been restored.';
            } else {
                // Soft delete (set status to 'closed')
                $stmt = $pdo->prepare("
                    UPDATE " . DB_PREFIX . "work_locations
                    SET status = 'closed',
                        updated_at = NOW()
                    WHERE id = ?
                ");
                $stmt->execute([$location_id]);
                $_SESSION['success'] = 'Location "' . htmlspecialchars($location['name']) . '" has been closed (soft delete).';
            }
        }
    }

} catch (Exception $e) {
    $_SESSION['error'] = 'Error deleting location: ' . $e->getMessage();
}

// Redirect back to list
safe_redirect(defined('MODULE_EMBEDDED') ? '../../m/locations/list' : 'list.php');
?>
