-- Work Locations Module
-- Database Schema for SCMS v3.0

-- Table: scms_work_locations
-- Tracks all work locations (offices, sites, warehouses, etc.)
CREATE TABLE IF NOT EXISTS `scms_work_locations` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `name` varchar(255) NOT NULL COMMENT 'Location name',
  `type` enum('office','warehouse','site','remote','other') DEFAULT NULL COMMENT 'Type of location',
  `address` text DEFAULT NULL COMMENT 'Full address',
  `address_line1` varchar(255) DEFAULT NULL COMMENT 'Address line 1',
  `address_line2` varchar(255) DEFAULT NULL COMMENT 'Address line 2',
  `city` varchar(100) DEFAULT NULL COMMENT 'City',
  `county` varchar(100) DEFAULT NULL COMMENT 'County',
  `postcode` varchar(20) DEFAULT NULL COMMENT 'Postcode',
  `country` varchar(100) DEFAULT 'United Kingdom' COMMENT 'Country',
  `latitude` decimal(10,8) DEFAULT NULL COMMENT 'GPS latitude for mapping',
  `longitude` decimal(11,8) DEFAULT NULL COMMENT 'GPS longitude for mapping',
  `capacity` int(11) DEFAULT 0 COMMENT 'Maximum worker capacity',
  `current_occupancy` int(11) DEFAULT 0 COMMENT 'Current number of workers',
  `manager_id` int(11) DEFAULT NULL COMMENT 'Site manager (worker ID)',
  `contact_phone` varchar(20) DEFAULT NULL COMMENT 'Location contact phone',
  `contact_email` varchar(255) DEFAULT NULL COMMENT 'Location contact email',
  `facilities` text DEFAULT NULL COMMENT 'JSON array of facilities available',
  `health_safety_info` text DEFAULT NULL COMMENT 'Health and safety information',
  `access_instructions` text DEFAULT NULL COMMENT 'How to access the location',
  `parking_info` text DEFAULT NULL COMMENT 'Parking information',
  `status` enum('active','inactive','closed') DEFAULT 'active' COMMENT 'Location status',
  `notes` text DEFAULT NULL COMMENT 'Additional notes',
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  `updated_at` timestamp NULL DEFAULT NULL ON UPDATE current_timestamp(),
  `created_by` int(11) DEFAULT NULL COMMENT 'User who created the location',
  PRIMARY KEY (`id`),
  KEY `manager_id` (`manager_id`),
  KEY `status` (`status`),
  KEY `type` (`type`),
  KEY `postcode` (`postcode`),
  KEY `created_by` (`created_by`),
  CONSTRAINT `fk_location_manager` FOREIGN KEY (`manager_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE SET NULL,
  CONSTRAINT `fk_location_user` FOREIGN KEY (`created_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Work locations and sites';

-- Table: scms_location_assignments
-- Track worker assignments to locations over time (historical record)
CREATE TABLE IF NOT EXISTS `scms_location_assignments` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `worker_id` int(11) NOT NULL COMMENT 'Worker ID',
  `location_id` int(11) NOT NULL COMMENT 'Location ID',
  `assignment_date` date NOT NULL COMMENT 'Date assigned to this location',
  `end_date` date DEFAULT NULL COMMENT 'Date assignment ended (NULL = current)',
  `assignment_type` enum('permanent','temporary','remote') DEFAULT 'permanent' COMMENT 'Type of assignment',
  `reason` varchar(255) DEFAULT NULL COMMENT 'Reason for assignment',
  `notes` text DEFAULT NULL COMMENT 'Additional notes',
  `assigned_by` int(11) DEFAULT NULL COMMENT 'User who made the assignment',
  `created_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `worker_id` (`worker_id`),
  KEY `location_id` (`location_id`),
  KEY `assignment_date` (`assignment_date`),
  KEY `assigned_by` (`assigned_by`),
  CONSTRAINT `fk_assignment_worker` FOREIGN KEY (`worker_id`) REFERENCES `scms_sponsored_workers` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_assignment_location` FOREIGN KEY (`location_id`) REFERENCES `scms_work_locations` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_assignment_user` FOREIGN KEY (`assigned_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Worker location assignment history';

-- Table: scms_location_capacity_log
-- Track capacity changes over time for reporting
CREATE TABLE IF NOT EXISTS `scms_location_capacity_log` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `location_id` int(11) NOT NULL COMMENT 'Location ID',
  `date` date NOT NULL COMMENT 'Date of the count',
  `occupancy` int(11) NOT NULL COMMENT 'Number of workers on this date',
  `capacity` int(11) NOT NULL COMMENT 'Capacity on this date',
  `utilization_percent` decimal(5,2) DEFAULT NULL COMMENT 'Occupancy as percentage of capacity',
  `logged_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  UNIQUE KEY `location_date` (`location_id`,`date`),
  CONSTRAINT `fk_capacity_location` FOREIGN KEY (`location_id`) REFERENCES `scms_work_locations` (`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Historical capacity tracking';

-- Table: scms_location_documents
-- Documents related to locations (lease agreements, health & safety certificates, etc.)
CREATE TABLE IF NOT EXISTS `scms_location_documents` (
  `id` int(11) NOT NULL AUTO_INCREMENT,
  `location_id` int(11) NOT NULL COMMENT 'Location ID',
  `document_type` varchar(100) NOT NULL COMMENT 'Type of document',
  `document_name` varchar(255) NOT NULL COMMENT 'Document name/title',
  `filename` varchar(255) NOT NULL COMMENT 'Original filename',
  `file_path` varchar(500) NOT NULL COMMENT 'Path to stored file',
  `file_size` int(11) DEFAULT NULL COMMENT 'File size in bytes',
  `mime_type` varchar(100) DEFAULT NULL COMMENT 'MIME type',
  `expiry_date` date DEFAULT NULL COMMENT 'Document expiry date if applicable',
  `uploaded_by` int(11) DEFAULT NULL COMMENT 'User who uploaded',
  `uploaded_at` timestamp NOT NULL DEFAULT current_timestamp(),
  PRIMARY KEY (`id`),
  KEY `location_id` (`location_id`),
  KEY `expiry_date` (`expiry_date`),
  KEY `uploaded_by` (`uploaded_by`),
  CONSTRAINT `fk_loc_doc_location` FOREIGN KEY (`location_id`) REFERENCES `scms_work_locations` (`id`) ON DELETE CASCADE,
  CONSTRAINT `fk_loc_doc_user` FOREIGN KEY (`uploaded_by`) REFERENCES `scms_users` (`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Documents for locations';

-- Add work_location_id to sponsored_workers table if not exists
ALTER TABLE `scms_sponsored_workers`
ADD COLUMN IF NOT EXISTS `work_location_id` int(11) DEFAULT NULL COMMENT 'Current work location' AFTER `department`,
ADD KEY IF NOT EXISTS `work_location_id` (`work_location_id`),
ADD CONSTRAINT `fk_worker_location` FOREIGN KEY IF NOT EXISTS (`work_location_id`) REFERENCES `scms_work_locations` (`id`) ON DELETE SET NULL;

-- Index for performance
CREATE INDEX idx_location_status_type ON scms_work_locations(status, type);
CREATE INDEX idx_location_occupancy ON scms_work_locations(current_occupancy, capacity);
CREATE INDEX idx_assignment_current ON scms_location_assignments(worker_id, end_date);

-- Initial data: Location types and facilities for reference
INSERT IGNORE INTO `scms_system_lists` (`list_type`, `list_value`, `list_label`, `sort_order`) VALUES
('location_type', 'office', 'Office', 1),
('location_type', 'warehouse', 'Warehouse', 2),
('location_type', 'site', 'Construction Site', 3),
('location_type', 'remote', 'Remote Location', 4),
('location_type', 'other', 'Other', 99),
('location_facility', 'parking', 'Parking Available', 1),
('location_facility', 'canteen', 'Canteen/Kitchen', 2),
('location_facility', 'toilets', 'Toilet Facilities', 3),
('location_facility', 'showers', 'Shower Facilities', 4),
('location_facility', 'wifi', 'WiFi Internet', 5),
('location_facility', 'first_aid', 'First Aid Station', 6),
('location_facility', 'disabled_access', 'Disabled Access', 7),
('location_document_type', 'lease', 'Lease Agreement', 1),
('location_document_type', 'health_safety', 'Health & Safety Certificate', 2),
('location_document_type', 'fire_cert', 'Fire Safety Certificate', 3),
('location_document_type', 'insurance', 'Insurance Document', 4),
('location_document_type', 'site_plan', 'Site Plan/Layout', 5),
('location_document_type', 'other', 'Other Document', 99);

-- Trigger to update current_occupancy when worker is assigned
DELIMITER //
CREATE TRIGGER IF NOT EXISTS update_location_occupancy_on_assign
AFTER UPDATE ON scms_sponsored_workers
FOR EACH ROW
BEGIN
  IF OLD.work_location_id IS NOT NULL AND OLD.work_location_id != NEW.work_location_id THEN
    UPDATE scms_work_locations
    SET current_occupancy = current_occupancy - 1
    WHERE id = OLD.work_location_id;
  END IF;

  IF NEW.work_location_id IS NOT NULL AND OLD.work_location_id != NEW.work_location_id THEN
    UPDATE scms_work_locations
    SET current_occupancy = current_occupancy + 1
    WHERE id = NEW.work_location_id;
  END IF;
END//
DELIMITER ;
