<?php
/**
 * Create Shift Patterns Table for Locations
 * Run this file once to create the database table
 */

require_once '../../config/database.php';
require_once '../../includes/database.php';

$pdo = get_database_connection();

try {
    // Create shift patterns table
    $sql = "CREATE TABLE IF NOT EXISTS " . DB_PREFIX . "location_shift_patterns (
        id INT(11) AUTO_INCREMENT PRIMARY KEY,
        location_id INT(11) NOT NULL,
        shift_name VARCHAR(100) NOT NULL,
        shift_type ENUM('day', 'night', 'evening', 'split', 'rotating', 'flexible', 'temporary', 'special') DEFAULT 'day',
        start_time TIME NOT NULL,
        end_time TIME NOT NULL,
        break_duration INT(11) DEFAULT 0 COMMENT 'Break duration in minutes',
        days_of_week VARCHAR(50) DEFAULT NULL COMMENT 'Comma-separated: Mon,Tue,Wed,Thu,Fri,Sat,Sun',
        is_default TINYINT(1) DEFAULT 0 COMMENT 'Default shift for this location',
        is_active TINYINT(1) DEFAULT 1,
        valid_from DATE DEFAULT NULL COMMENT 'For temporary shifts',
        valid_until DATE DEFAULT NULL COMMENT 'For temporary shifts',
        notes TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        INDEX idx_location_id (location_id),
        INDEX idx_shift_type (shift_type),
        INDEX idx_is_active (is_active),
        INDEX idx_is_default (is_default)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

    $pdo->exec($sql);
    echo "✓ Table " . DB_PREFIX . "location_shift_patterns created successfully<br>";

    // Migrate existing shift data from work_locations table to shift_patterns table
    echo "<br>Migrating existing shift data...<br>";

    $migrate_sql = "SELECT id, shift_name, shift_start_time, shift_end_time
                    FROM " . DB_PREFIX . "work_locations
                    WHERE shift_start_time IS NOT NULL OR shift_end_time IS NOT NULL";

    $stmt = $pdo->query($migrate_sql);
    $locations_with_shifts = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $migrated = 0;
    foreach ($locations_with_shifts as $location) {
        // Check if shift pattern already exists for this location
        $check = $pdo->prepare("SELECT COUNT(*) FROM " . DB_PREFIX . "location_shift_patterns WHERE location_id = ?");
        $check->execute([$location['id']]);

        if ($check->fetchColumn() == 0) {
            // Insert the existing shift as the default shift pattern
            $insert = $pdo->prepare("INSERT INTO " . DB_PREFIX . "location_shift_patterns
                (location_id, shift_name, shift_type, start_time, end_time, is_default, is_active, days_of_week)
                VALUES (?, ?, 'day', ?, ?, 1, 1, 'Mon,Tue,Wed,Thu,Fri')");

            $insert->execute([
                $location['id'],
                $location['shift_name'] ?: 'Day Shift',
                $location['shift_start_time'] ?: '09:00:00',
                $location['shift_end_time'] ?: '17:00:00'
            ]);

            $migrated++;
        }
    }

    echo "✓ Migrated $migrated existing shift records<br>";

    // Verify table structure
    echo "<br>Table structure:<br>";
    $columns = $pdo->query("SHOW COLUMNS FROM " . DB_PREFIX . "location_shift_patterns");
    echo "<table border='1' cellpadding='5'>";
    echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
    while ($col = $columns->fetch(PDO::FETCH_ASSOC)) {
        echo "<tr>";
        echo "<td>" . htmlspecialchars($col['Field']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Type']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Null']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Key']) . "</td>";
        echo "<td>" . htmlspecialchars($col['Default']) . "</td>";
        echo "</tr>";
    }
    echo "</table>";

    echo "<br><br>✓ All done! You can now use multiple shift patterns for each location.";
    echo "<br><br><a href='list.php'>Go to Locations List</a>";

} catch (PDOException $e) {
    echo "Error: " . $e->getMessage();
}
